/*
 * Userspace program that communicates with the gc_memory device driver
 * through ioctls
 *
 * Martha Barker (mmb2295)
 * Columbia University
 */

#include <stdio.h>
#include <stdlib.h>
#include "GC_Memory.h"
#include "reference.h"
#include "hardware.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stdbool.h>

int gc_memory_fd;

int base = 0;
int bound = 2048;
int threshold = 16;
int num_roots = 100;
int debug = 0;
int num_ops = 20000;
float w_pr = 0.2;
float r_pr = 0.8;

static unsigned long next_rand = 1;
/* Suggested by the rand(3) manpage */
int myrand()
{
  next_rand = next_rand * 1103515245l + 12345l;
  return next_rand >> 16 & 0x7fffffff;
}

void printHelp(){
  printf(
    "-b <n>         bram base\n"
    "-n <n>         bram bound\n"
    "-t <n>         garbage collection threshold\n"
    "-o \"n n\"     percentage of ops \"write, read, kill pointer\" \n"
    "-r <n>         number of roots\n"
    "-c <n>         number of inputs\n"
    "-d             debug mode\n"
    "-h             print help \n");
}

void processArgs(int argc, char** argv){
  const char* const short_opts = "b:n:o:dhr:c:t:";

  while(true){
    const int opt = getopt(argc, argv, short_opts);

    if(opt == -1){
      break;
    }

    switch(opt){
    case 'b':
      {
	base = atoi(optarg);
	break;
      }
    case 'n':
      {
	bound = atoi(optarg);
	break;
      }
    case 't':
      {
	threshold = atoi(optarg);
	break;
      }
    case 'd':
      {
	debug = 1;
	break;
      }
    case 'h':
      {
	printHelp();
	break;
      }
    case 'r':
      {
	num_roots = atoi(optarg);
	break;
      }
    case 'c':
      {
	num_ops = atoi(optarg);
	break;
      }
    case 'o':
      {
	char *token = strtok(optarg, " ");

	w_pr = atoi(token)/100.0;
	token = strtok(NULL, " ");
	r_pr = atoi(token)/100.0;
	break;
      }
    default:
      {
	printHelp();
	break;
      }
    }
  }
}

void print_status(gc_memory_status_t *s) {
  printf("----STATUS----\nread rdy: %d \nwrite rdy: %d \nread_res valid: %d \nwrite_res valid : %d \nstart gc: %d \ngc done: %d \nroot rdy: %d\n--------------\n", s->read_rdy, s->write_rdy, s->read_res_rdy, s->write_res_rdy, s->start_gc, s->gc_done, s->root_rdy);
}

int main(int argc, char *argv[]) {

  static const char filename[] = "/dev/gc_memory";
  gc_memory_status_t status = {0, 0, 0, 0, 0, 0, 0};
  reference_arg_t ref;
  unsigned int *roots;
  unsigned int *hw_bram;
  int writes_in = 0;
  int reads_in = 0;
  int roots_in = 0;
  int num_writes = 0;
  int num_reads = 0;
  int current_roots = 0;
  unsigned int read_op_array[(int)(num_ops*r_pr)+1], write_op_array[(int)(num_ops*w_pr)+1], read_res_array[(int)(num_ops*r_pr)+1], write_res_array[(int)(num_ops*w_pr)+1];
  int read_op_i = 0, read_res_i = 0, write_op_i = 0, write_res_i = 0;
  int gc_mode = 0;
  
  printf("GC Memory userspace program started\n");
  
  if ( (gc_memory_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }
  
  processArgs(argc, argv);
  roots = malloc(num_roots * sizeof(unsigned int));
  hw_bram = malloc((bound-base) * sizeof(unsigned int));

  
  make_reference(&ref, base, bound);
  
  init_memory(gc_memory_fd, base, bound, threshold, num_roots);
  status.read_rdy = 1;
  status.write_rdy = 1;

  while ((num_writes + num_reads) < num_ops) {
    update_status(gc_memory_fd, &status);
    
    if (status.start_gc) {
      gc_mode = 1;
      status.start_gc = 0;
    }

    if ((gc_mode == 1) && (writes_in == 0) && (reads_in == 0)) {
      get_bram(gc_memory_fd, base, bound, hw_bram);
      for (int i = base; i < bound; i++) {
	unsigned int hw = hw_bram[i];
	unsigned int t = (hw & 0x1);
	unsigned int d = ((hw >> 1) & 0xfff);
	unsigned int p = ((hw >> 13) & 0xfff);
	if (t != ref.bram[i].type)
	  printf("types don't match: %x %d %d\n", i, t, ref.bram[i].type);
	if (d != ref.bram[i].data)
	  printf("data doesn't match: %x %x %x\n", i, d, ref.bram[i].data);
	if (p != ref.bram[i].pointer)
	  printf("pointers don't match: %x %x %x\n", i, p, ref.bram[i].pointer);
      }
      status.root_rdy = 1;
      printf("do gc\n");
      gc_reference(&ref, base, bound);
      list_roots(roots, &ref, num_roots);

      while(true) {
      	update_status(gc_memory_fd, &status);
	
      	if (status.gc_done){
	  printf("done gc\n");
      	  status.gc_done = 0;
	  gc_mode = 0;
	  roots_in = 0;
	  get_bram(gc_memory_fd, base, bound, hw_bram);
	  for (int i = base; i < bound; i++) {
	    unsigned int hw = hw_bram[i];
	    unsigned int t = (hw & 0x1);
	    unsigned int d = ((hw >> 1) & 0xfff);
	    unsigned int p = ((hw >> 13) & 0xfff);
	    if (t != ref.bram[i].type)
	      printf("types don't match: %x %d %d\n", i, t, ref.bram[i].type);
	    if (d != ref.bram[i].data)
	      printf("data doesn't match: %x %x %x\n", i, d, ref.bram[i].data);
	    if (p != ref.bram[i].pointer)
	      printf("pointers don't match: %x %x %x\n", i, p, ref.bram[i].pointer);
	  }
      	  break;
      	}
	
      	if (status.root_rdy && (roots_in < num_roots)) {
      	  unsigned int r = roots[roots_in];
      	  if (r != -1)
      	    r = ((r & 0xfff) << 1) | (0x0);
      	  else
      	    r = ((0 & 0xfff) << 1) | (0x1);
      	  root_op(gc_memory_fd, &r);
	  roots_in++;
      	  status.root_rdy = 0;
      	}
      }
    }

    if (status.read_res_rdy) {
      unsigned int node, ref_node;
      read_res(gc_memory_fd, &node);
      ref_node = read_reference(&ref, read_op_array[read_res_i]);
      read_res_array[read_res_i] = node;
      read_res_i++;
      reads_in--;
      if (node != ref_node){
  	printf("ERROR: read results don't match: %x %x\n", node, ref_node);
	return 0;
      }
      status.read_res_rdy = 0;
    }
    
    if (status.write_res_rdy) {
      unsigned int pointer, ref_pointer;
      write_res(gc_memory_fd, &pointer);
      ref_pointer = write_reference(&ref, write_op_array[write_res_i], myrand()%100+2);
      current_roots++;
      write_res_array[write_res_i] = pointer;
      write_res_i++;
      writes_in--;
      if (pointer != ref_pointer){
  	printf("Error: write results don't match: %x %x\n", pointer, ref_pointer);
	return 0;
      }
      status.write_res_rdy = 0;
    }
    
    if (status.read_rdy && (num_reads < num_ops * r_pr) && (current_roots > 0) && (gc_mode == 0)) {
      int r = myrand() % current_roots;
      unsigned int pointer = get_root(&ref, r);
      read_op_array[read_op_i] = pointer;
      read_op_i++;
      read_op(gc_memory_fd, &pointer);
      reads_in++;
      num_reads++;
      status.read_rdy = 0;
    }
    
    if (status.write_rdy && (num_writes < num_ops * w_pr) && (current_roots < num_roots) && (gc_mode == 0)) {
      unsigned int pointer;
      unsigned int node;
      if (current_roots > 0){
  	pointer = get_root(&ref, (myrand()%current_roots));
	node = ((pointer & 0xfff) << 13) | (((myrand()%12) & 0xfff) << 1) | (0x1);
      }else {
	node = ((0 & 0xfff) << 13) | ((0 & 0xfff) << 1) | (0x0);
      }
      write_op_array[write_op_i] = node;
      write_op_i++;
      write_op(gc_memory_fd, &node);
      writes_in++;
      num_writes++;
      status.write_rdy = 0;
      current_roots -= decrement_roots(&ref);

    }
  }

  free(hw_bram);
  free(roots);
  free_reference(&ref);
  printf("GC Memory userspace program terminating\n");
  return 0;
}
