module GC_Memory(input logic clk,
		 input logic 	     reset,
		 input logic [31:0]  writedata,
		 input logic 	     write,
		 input 		     chipselect,
		 input logic [3:0]   address,
		 output logic [31:0] readdata,
		 output logic [9:0]  LEDR
		 );
   
   logic [11:0] 		     set_num_roots, set_base, set_bound, set_threshold;
   logic [6:0] 			     interrupt_status;
   logic [11:0] 		     read_op, write_res;
   logic [24:0] 		     write_op, read_res;
   logic [12:0] 		     root_op;
   logic [11:0] 		     bram_addr; 
   logic [24:0] 		     bram_line;

   assign irq = |interrupt_status;

   initial begin
      LEDR = 10'b0000000000;
   end
   
   always_ff @(posedge clk) begin
      if (chipselect) begin
	 case (address)
	   4'h0 : LEDR[3:0] <= 4'h0 ; 
	   4'h1 : LEDR[3:0] <= 4'h1 ; 
      	   4'h2 : LEDR[3:0] <= 4'h2 ; 
      	   4'h3 : LEDR[3:0] <= 4'h3 ; 
      	   4'h4 : LEDR[3:0] <= 4'h4 ; 
      	   4'h5 : LEDR[3:0] <= 4'h5 ; 
	   4'h6 : LEDR[3:0] <= 4'h6 ; 
      	   4'h7 : LEDR[3:0] <= 4'h7 ; 
	   4'h8 : LEDR[3:0] <= 4'h8 ;
	   4'h9 : LEDR[3:0] <= 4'h9 ;
	   4'ha : LEDR[3:0] <= 4'ha ;
	   4'hb : LEDR[3:0] <= 4'hb ;
	   4'hc : LEDR[3:0] <= 4'hc ;
	   4'hd : LEDR[3:0] <= 4'hd ;
	   4'he : LEDR[3:0] <= 4'he ;
	   4'hf : LEDR[3:0] <= 4'hf ; 
	 endcase 
      end
   end // always_ff @

   always_ff @(posedge clk) begin
      LEDR[8:4] <= bram_addr;
   end

   logic increment;
   
   always_ff @(posedge clk) begin
      if (reset) begin
	 set_num_roots <= 12'd0;
	 set_base <= 12'd0;
	 set_bound <= 12'd0;
	 set_threshold <= 12'd0;
	 read_op <= 12'd0;
	 write_op <= 25'd0;
	 root_op <= 13'd0;
      end else if (chipselect) begin 
	 case (address)
	   4'h0 : if (!write) begin 
	      readdata[6:0] <= interrupt_status;
	   end
	   4'h1 : if (write) set_num_roots <= writedata[11:0];
      	   4'h2 : if (write) set_base <= writedata[11:0];
      	   4'h3 : if (write) set_bound <= writedata[11:0];
      	   4'h4 : if (write) set_threshold <= writedata[11:0];
      	   4'h5 : if (write) read_op <= writedata[11:0]; else if (!write) readdata <= {7'b0, read_res};
	   4'h6 : if (write) write_op <= writedata[24:0]; else if (!write) readdata <= {20'd0, write_res};
      	   4'h7 : if (write) root_op <= writedata[12:0];
	   4'h8 : if (!write) readdata <= {7'b0, bram_line};
	   default : ;
	 endcase 
      end // if (chipselect)

      if (chipselect && write && (address == 4'h2))
	bram_addr <= writedata[11:0];
      else if (chipselect && !write && (address == 4'h8)) begin
	 if (increment) begin
	    bram_addr <= bram_addr + 12'd1;
	    increment <= ~increment;
	 end else begin
	    increment <= ~increment;
	 end
      end
      else if (bram_addr >= set_bound)
	bram_addr <= set_base;
   end 

   logic num_roots_rdy, base_rdy, bound_rdy, threshold_rdy;
   initial begin
      num_roots_rdy = 1'b0;
      base_rdy = 1'b0;
      bound_rdy = 1'b0;
      threshold_rdy = 1'b0;
   end

   always_ff @(posedge clk) begin
      if (chipselect && write && (address == 4'h1)) begin
	 num_roots_rdy <= 1'b1;
      end else if (num_roots_r && num_roots_rdy) begin
	 num_roots_rdy <= 1'b0;
      end

      if (chipselect && write && (address == 4'h2)) begin
	 base_rdy <= 1'b1;
      end else if (base_r && base_rdy) begin
	 base_rdy <= 1'b0;
      end

      if (chipselect && write && (address == 4'h3)) begin
	 bound_rdy <= 1'b1;
      end else if (bound_r && bound_rdy) begin
	 bound_rdy <= 1'b0;
      end

      if (chipselect && write && (address == 4'h4)) begin
	 threshold_rdy <= 1'b1;
      end else if (threshold_r && threshold_rdy) begin
	 threshold_rdy <= 1'b0;
      end
   end

   assign num_roots_d = {set_num_roots, 1'b1} & {13{num_roots_rdy}};
   assign base_d = {set_base, 1'b1} & {13{base_rdy}};
   assign bound_d = {set_bound, 1'b1} & {13{bound_rdy}};
   assign threshold_d = {set_threshold, 1'b1} & {13{threshold_rdy}};

   logic write_rdy, read_rdy, write_res_rdy, read_res_rdy, root_rdy;
   initial begin
      write_rdy = 1'b0;
      read_rdy = 1'b0;
      write_res_rdy = 1'b0;
      read_res_rdy = 1'b0;
      root_rdy = 1'b0;
   end
		 
   always_ff @(posedge clk) begin
      if (chipselect && write && (address == 4'h6)) begin
	 write_rdy <= 1'b1;
      end else if (wr_r && write_rdy) begin
	 write_rdy <= 1'b0;
	 interrupt_status[1] <= 1'b1;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[1] <= 1'b0;
      end

      if (chipselect && write && (address == 4'h5)) begin
	 read_rdy <= 1'b1;
      end else if (rd_r && read_rdy) begin
	 read_rdy <= 1'b0;
	 interrupt_status[0] <= 1'b1;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[0] <= 1'b0;
      end

      if (~write_res_rdy && wr_ptr_dout[0]) begin
	 write_res_rdy <= 1'b1;
	 interrupt_status[3] <= 1'b1;
	 write_res <= wr_ptr_dout[12:1];
      end else if (chipselect && !write && (address == 4'h6)) begin
	 write_res_rdy <= 1'b0;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[3] <= 1'b0;
      end

      if (~read_res_rdy && rd_node_dout[0]) begin
	 read_res_rdy <= 1'b1;
	 interrupt_status[2] <= 1'b1;
	 read_res <= rd_node_dout[25:1];
      end else if (chipselect && !write && (address == 4'h5)) begin
	 read_res_rdy <= 1'b0;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[2] <= 1'b0;
      end

      if (start_gc_dout[0]) begin
	 interrupt_status[4] <= 1'b1;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[4] <= 1'b0;
      end

      if (gc_done_d[0]) begin
	 interrupt_status[5] <= 1'b1;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[5] <= 1'b0;
      end

      if (chipselect && write && (address == 4'h7)) begin
	 root_rdy <= 1'b1;
      end else if (root_r && root_rdy) begin
	 root_rdy <= 1'b0;
	 interrupt_status[6] <= 1'b1;
      end else if (chipselect && !write && (address == 4'h0)) begin
	 interrupt_status[6] <= 1'b0;
      end
   end

   assign wr_d = {write_op, 1'b1} & {26{write_rdy}};
   assign rd_d = {read_op, 1'b1} & {13{read_rdy}};
   assign wr_ptr_rout = (chipselect && !write && (address == 4'h6));
   assign rd_node_rout = (chipselect && !write && (address == 4'h5));
   assign start_gc_rout = (chipselect && !write && (address == 4'h0));
   assign gc_done_r = (chipselect && !write && (address == 4'h0));
   assign root_d = {root_op, 1'b1} & {14{root_rdy}};

   \Word#_t num_roots_d;
   logic 		      num_roots_r;
   Root_t root_d;
   logic 		      root_r;
   \Word#_t base_d;
   logic 		      base_r;
   \Word#_t bound_d;
   logic 		      bound_r;
   Go_t gc_done_d;
   logic 		      gc_done_r;
   Pointer_t rd_d;
   logic 		      rd_r;
   Node_t rd_node_dout;
   logic 		      rd_node_rout;
   Node_t wr_d;
   logic 		      wr_r;
   Pointer_t wr_ptr_dout;
   logic 		      wr_ptr_rout;
   Go_t force_gc_d;
   logic 		      force_gc_r;
   Go_t start_gc_dout;
   logic 		      start_gc_rout;
   Raise_Err_t gc_error_dout;
   logic 		      gc_error_rout;
   Node_t mem_error_d;
   logic 		      mem_error_r;
   assign mem_error_r = 1'b1;
   \Word#_t threshold_d;
   logic 		      threshold_r;

   Pointer_t gc_read_d;
   logic 		      gc_read_r;
   Node_t gc_node_d;
   logic 		      gc_node_r;
   Pointer_t gc_free_d;
   logic 		      gc_free_r;
   Go_t gc_ack_d;
   logic 		      gc_ack_r;
   Pointer_t mem_read_d;
   logic 		      mem_read_r;
   Node_t mem_node_d;
   logic 		      mem_node_r;
   Go_t mem_ack_d;
   logic 		      mem_ack_r;
   \Word#_t mem_base_d;
   logic 		      mem_base_r;
   \Word#_t mem_bound_d;
   logic 		      mem_bound_r;
   \Word#_t gc_base_d;
   logic 		      gc_base_r;
   \Word#_t gc_bound_d;
   logic 		      gc_bound_r;
   Pointer_t mem_ref_dout;
   logic 		      mem_ref_rout;
   Pointer_t gc_ref_d;
   logic 		      gc_ref_r;
   Go_t no_gc_dout;
   logic 		      no_gc_rout;
   
   parameter POINTER_BITS = 13;
   parameter NODE_BITS = 26;
   
   WordFork2 #(.POINTER_BITS(POINTER_BITS),
	       .NODE_BITS(NODE_BITS)) base_fork(.clk(clk),
			 .input_d(base_d),
			 .input_r(base_r),
			 .output1_dout(mem_base_d),
			 .output1_rout(mem_base_r),
			 .output2_dout(gc_base_d),
			 .output2_rout(gc_base_r)
			 );
   
   WordFork2 #(.POINTER_BITS(POINTER_BITS),
	       .NODE_BITS(NODE_BITS)) bound_fork(.clk(clk),
			  .input_d(bound_d),
			  .input_r(bound_r),
			  .output1_dout(mem_bound_d),
			  .output1_rout(mem_bound_r),
			  .output2_dout(gc_bound_d),
			  .output2_rout(gc_bound_r)
			  );

   RefFork2 #(.POINTER_BITS(POINTER_BITS),
	      .NODE_BITS(NODE_BITS)) ref_fork(.clk(clk),
		     .input_d(mem_ref_dout),
		     .input_r(mem_ref_rout),
		     .output1_dout(wr_ptr_dout),
		     .output1_rout(wr_ptr_rout),
		     .output2_dout(gc_ref_d),
		     .output2_rout(gc_ref_r)
		     );
   
   
   Memory #(.POINTER_BITS(POINTER_BITS),
	    .NODE_BITS(NODE_BITS)) memory(.clk(clk),
					  .dat_d(wr_d),
					  .dat_r(wr_r),
					  .ref_dout(mem_ref_dout),
					  .ref_rout(mem_ref_rout),
					  .rd_d(mem_read_d),
					  .rd_r(mem_read_r),
					  .rd_node_dout(mem_node_d),
					  .rd_node_rout(mem_node_r),
					  .free_d(gc_free_d),
					  .free_r(gc_free_r),
					  .init_start_d(mem_base_d),
					  .init_start_r(mem_base_r),
					  .init_end_d(mem_bound_d),
					  .init_end_r(mem_bound_r),
					  .free_ack_dout(gc_ack_d),
					  .free_ack_rout(gc_ack_r),
					  .error_mem_d(mem_error_d),
					  .error_mem_r(mem_error_r),
 					  .bram_addr(bram_addr),
					  .bram_line(bram_line)
					  );

   Garbage_Collection #(.POINTER_BITS(POINTER_BITS),
			.NODE_BITS(NODE_BITS)) gc(.clk(clk),
			 .num_roots_d(num_roots_d),
			 .num_roots_r(num_roots_r),
			 .root_d(root_d),
			 .root_r(root_r),
			 .base_d(gc_base_d),
			 .base_r(gc_base_r),
			 .bound_d(gc_bound_d),
			 .bound_r(gc_bound_r),
			 .done_dout(gc_done_d),
			 .done_rout(gc_done_r),
			 .node_d(gc_node_d),
			 .node_r(gc_node_r),
			 .ack_d(gc_ack_d),
			 .ack_r(gc_ack_r),
			 .read_dout(gc_read_d),
			 .read_rout(gc_read_r),
			 .free_dout(gc_free_d),
			 .free_rout(gc_free_r),
			 .alloc_d(gc_ref_d),
			 .alloc_r(gc_ref_r),
			 .force_gc_d(force_gc_d),
			 .force_gc_r(force_gc_r),
			 .start_gc_dout(start_gc_dout),
			 .start_gc_rout(start_gc_rout),
			 .gc_error_dout(gc_error_dout),
			 .gc_error_rout(gc_error_rout),
			 .threshold_d(threshold_d),
			 .threshold_r(threshold_r)
			 );

   RefNodeMerge2 #(.POINTER_BITS(POINTER_BITS),
		   .NODE_BITS(NODE_BITS)) read_merge(.clk(clk),
			    .cmd0_d(gc_read_d),
			    .cmd0_r(gc_read_r),
			    .cmd1_d(rd_d),
			    .cmd1_r(rd_r),
			    .res_in_d(mem_node_d),
			    .res_in_r(mem_node_r),
			    .res0_dout(gc_node_d),
			    .res0_rout(gc_node_r),
			    .res1_dout(rd_node_dout),
			    .res1_rout(rd_node_rout),
			    .cmd_out_dout(mem_read_d),
			    .cmd_out_rout(mem_read_r)
			    );
   

endmodule // Toplevel
