/* * Device driver for the GC Memory module
 *
 * A Platform device implemented using the misc subsystem
 *
 * Adapted from vga_ball:
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod GC_Memory.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree GC_Memory.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "GC_Memory.h"

#define DRIVER_NAME "gc_memory"
#define GC_MEMORY_IRQ_NUM 72

/* Device registers */
#define BG_STATUS(x) (x)
#define BG_NUM_ROOTS(x) ((x)+1*4)
#define BG_BASE(x) ((x)+2*4)
#define BG_BOUND(x) ((x)+3*4)
#define BG_THRESHOLD(x) ((x)+4*4)
#define BG_READ(x) ((x)+5*4)
#define BG_WRITE(x) ((x)+6*4)
#define BG_ROOT(x) ((x)+7*4)
#define BG_BRAM(x) ((x)+8*4)

/*
 * Information about our device
 */
struct gc_memory_dev {
  struct resource res; /* Resource: our registers */
  void __iomem *virtbase; /* Where registers can be accessed in memory */
  gc_memory_status_t status;
} dev;

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long gc_memory_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
  gc_memory_arg_t vla;
  pr_info(DRIVER_NAME ": ioctl");

  switch (cmd) {
  case GC_MEMORY_NUM_ROOTS:
    pr_info(DRIVER_NAME ": set num_roots");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t)))
      return -EACCES;
    iowrite32(vla.value, BG_NUM_ROOTS(dev.virtbase));
    break;

  case GC_MEMORY_BASE:
    pr_info(DRIVER_NAME ": set base");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t)))
      return -EACCES;
    iowrite32(vla.value, BG_BASE(dev.virtbase));
    break;

  case GC_MEMORY_BOUND:
    pr_info(DRIVER_NAME ": set bound");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t)))
      return -EACCES;
    iowrite32(vla.value, BG_BOUND(dev.virtbase));
    break;

  case GC_MEMORY_THRESHOLD:
    pr_info(DRIVER_NAME ": set threshold");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t)))
      return -EACCES;
    iowrite32(vla.value, BG_THRESHOLD(dev.virtbase));
    break;

  case GC_MEMORY_READ_OP:
    pr_info(DRIVER_NAME ": read op");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t))){
      return -EACCES;
    }
    pr_info(DRIVER_NAME ": read op 2");
	
    iowrite32(vla.value, BG_READ(dev.virtbase));
    pr_info(DRIVER_NAME ": read op 3");
    break;

  case GC_MEMORY_READ_RES:
    pr_info(DRIVER_NAME ": read res");
    vla.value = ioread32(BG_READ(dev.virtbase));
    if (copy_to_user((gc_memory_arg_t *) arg, &vla,
		     sizeof(gc_memory_arg_t)))
      return -EACCES;
    break;

  case GC_MEMORY_WRITE_OP:
    pr_info(DRIVER_NAME ": write op");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t)))
      return -EACCES;
    iowrite32(vla.value, BG_WRITE(dev.virtbase));
    break;

  case GC_MEMORY_WRITE_RES:
    pr_info(DRIVER_NAME ": write res");
    vla.value = ioread32(BG_WRITE(dev.virtbase));
    if (copy_to_user((gc_memory_arg_t *) arg, &vla,
		     sizeof(gc_memory_arg_t)))
      return -EACCES;
    break;

  case GC_MEMORY_ROOT:
    pr_info(DRIVER_NAME ": root");
    if (copy_from_user(&vla, (gc_memory_arg_t *) arg,
		       sizeof(gc_memory_arg_t)))
      return -EACCES;
    iowrite32(vla.value, BG_ROOT(dev.virtbase));
    break;

  case GC_MEMORY_STATUS:
    pr_info(DRIVER_NAME ": read status reg");
    vla.value = ioread32(BG_STATUS(dev.virtbase));
    if (copy_to_user((gc_memory_arg_t *) arg, &vla,
		     sizeof(gc_memory_arg_t)))
      return -EACCES;
    break;

  case GC_MEMORY_READ_BRAM:
    vla.value = ioread32(BG_BRAM(dev.virtbase));
    if (copy_to_user((gc_memory_arg_t *) arg, &vla,
		     sizeof(gc_memory_arg_t)))
      return -EACCES;
    break;

  default:
    return -EINVAL;
  }

  return 0;
}

/* The operations our device knows how to do */
static const struct file_operations gc_memory_fops = {
  .owner		= THIS_MODULE,
  .unlocked_ioctl       = gc_memory_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice gc_memory_misc_device = {
  .minor		= MISC_DYNAMIC_MINOR,
  .name		        = DRIVER_NAME,
  .fops		        = &gc_memory_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init gc_memory_probe(struct platform_device *pdev)
{
  int ret;
  
  /* Register ourselves as a misc device: creates /dev/gc_memory */
  ret = misc_register(&gc_memory_misc_device);
  
  /* Get the address of our registers from the device tree */
  ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
  if (ret) {
    ret = -ENOENT;
    goto out_deregister;
  }
  
  /* Make sure we can use these registers */
  if (request_mem_region(dev.res.start, resource_size(&dev.res),
			 DRIVER_NAME) == NULL) {
    ret = -EBUSY;
    goto out_deregister;
  }
  
  /* Arrange access to our registers */
  dev.virtbase = of_iomap(pdev->dev.of_node, 0);
  pr_info(DRIVER_NAME ": dev.virtbase = %p", dev.virtbase);
  if (dev.virtbase == NULL) {
    ret = -ENOMEM;
    goto out_release_mem_region;
  }

  return 0;
  
 out_release_mem_region:
  release_mem_region(dev.res.start, resource_size(&dev.res));
 out_deregister:
  misc_deregister(&gc_memory_misc_device);
  return ret;
}

/* Clean-up code: release resources */
static int gc_memory_remove(struct platform_device *pdev)
{
  iounmap(dev.virtbase);
  release_mem_region(dev.res.start, resource_size(&dev.res));
  misc_deregister(&gc_memory_misc_device);
  return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id gc_memory_of_match[] = {
  { .compatible = "csee4840,gc_memory-1.0" },
  {},
};
MODULE_DEVICE_TABLE(of, gc_memory_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver gc_memory_driver = {
  .driver	= {
    .name	= DRIVER_NAME,
    .owner	= THIS_MODULE,
    .of_match_table = of_match_ptr(gc_memory_of_match),
  },
  .remove	= __exit_p(gc_memory_remove),
};

/* Called when the module is loaded: set things up */
static int __init gc_memory_init(void)
{
  pr_info(DRIVER_NAME ": init\n");
  return platform_driver_probe(&gc_memory_driver, gc_memory_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit gc_memory_exit(void)
{
  platform_driver_unregister(&gc_memory_driver);
  pr_info(DRIVER_NAME ": exit\n");
}

module_init(gc_memory_init);
module_exit(gc_memory_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Martha Barker (mmb2295)");
MODULE_DESCRIPTION("GC Memory driver");
