#include <assert.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include <functional>
#include <string>

#include "Vprocessor.h"
#include "verilated.h"
#include "verilator_test.h"
#include "utils.h"

#define RESET "\x1B[0m"
#define RED   "\x1B[31m"
#define GREEN "\x1B[32m"

#define NBRANCH_PROG(x) (x), sizeof(x), sizeof(x)
#define PROG(x) (x), sizeof(x)

#define STRINGIZE(x) #x
#define LINE(x) STRINGIZE(x)
#define ASSERT(x) \
    if (!(x)) throw (const char *)(#x " (" __FILE__ ":" LINE(__LINE__) ")");

typedef Vprocessor*        Vtp;
typedef const Vprocessor*  CVtp;

void run_test(Vprocessor *tb,
        const char *name,
        const char *prog,
        size_t prog_len,
        size_t instr_cycles,
        void (*setup)(Vtp, void*),
        void (*verifier)(CVtp, void*),
        void *data)
{
    try {
        memset(tb->processor__DOT__m__DOT__mem, 0, sizeof(tb->processor__DOT__m__DOT__mem));
        memset(tb->VCPU(R_registers), 0xCC, 32);
        tb->VCPU(alu_operand) = 0xCC;
        tb->VCPU(alu_result) = 0xCC;
        tb->VCPU(D_register) = 0xCC;

        tb->reset = 1;
        clock_cycle(tb);

        ASSERT(tb->VCPU(R_registers[0]) == 0);
        ASSERT(tb->VCPU(I_register) == 0);
        ASSERT(tb->VCPU(N_register) == 0);
        ASSERT(tb->VCPU(P_register) == 0);
        ASSERT(tb->VCPU(Q) == 0);
        ASSERT(tb->PROC(bus_from_cpu) == 0);
        ASSERT(tb->PROC(ram_we) == 0);
        ASSERT(tb->PROC(address_from_cpu) == 0);
        ASSERT(tb->VCPU(clock_counter) == 0);
        ASSERT(tb->VCPU(state) == S_FETCH);

        tb->reset = 0;
        ASSERT(tb->VCPU(clock_counter) == 0);

        memcpy(tb->processor__DOT__m__DOT__mem, prog, prog_len);
        setup(tb, data);

        for (int i = 0; i < instr_cycles; i++) {
            machine_cycle(tb);
            machine_cycle(tb);
            
            if (tb->VCPU(I_register) == 0xC) {
                machine_cycle(tb);
            }

            ASSERT(tb->VCPU(state) == S_FETCH);
        }

        verifier(tb, data);
        fprintf(stderr, RESET GREEN "[P] %s\n" RESET, name);
    }
    catch (const char *e) {
        fprintf(stderr, RESET RED "[F] %s\n" RESET, name);
        fprintf(stderr, "--> %s\n", e);
        print_state(tb);
    }
}

void tb_noop(Vtp tb, void*){}

void run_basic_test(Vprocessor *tb,
        const char *name,
        const char *prog,
        size_t prog_len,
        size_t instr_cycles,
        void (*f)(CVtp, void*),
        void *data)
{
    run_test(tb, name, prog, prog_len, instr_cycles, tb_noop, f, data);
}

int main(int argc, char *argv[])
{
    Verilated::commandArgs(argc, argv);
    Vprocessor *tb = new Vprocessor;
    char buf[32];

    //////////////////////////////////////////////////////////////////////////
    // 00 (IDL) Check that CPU stops after instruction
    //
    // We're on CentOS 6 with G++ 4.4, so no lambdas. Emulate them with
    // structs with static functions instead, which is kind of gross.

    struct IDLTest1 {
        static void verify_00_IDL(CVtp tb, void*) {
            //ASSERT(tb->VCPU(idle));
            ASSERT(tb->VCPU(R_registers[1] == 0));
            ASSERT(tb->VCPU(R_registers[0] == 2));
        }
    };

    run_basic_test(tb,
        "00 IDL",
        NBRANCH_PROG("\x00\x01\x01\x01"),
        IDLTest1::verify_00_IDL,
        NULL);

    //////////////////////////////////////////////////////////////////////////
    // 01-0F (LDN)

    struct LDNTest1 {
        static void verify_0N_LDN(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0xB3);
        }

        static void setup_0N_LDN(Vtp tb, void *data) {
            tb->VCPU(R_registers[(size_t)data]) = 0xB3;
        }
    };

    for (int x = 0x01; x <= 0x0F; x++) {
        char c = (char)x;
        sprintf(buf, "%02X LDN R(N) -> D", x);
        run_test(tb, buf, &c, 1, 1,
                LDNTest1::setup_0N_LDN,
                LDNTest1::verify_0N_LDN,
                (void*)(x & 0x0F));
    }

    //////////////////////////////////////////////////////////////////////////
    // 10-1F (INC)

    char inc_prog1[7];

    // Testing INC R(0) is hard since R(0) is the default program counter.
    // Need to do setup to switch P to R(3), reset R(0), and increment.
    char inc_prog2[] = "\xF8\x07\xA3\xF8\x00\xB3\xD3\xA0\x10\x10\x10\x10\x10";

    struct INCTest1 {
        static void setup_1N_INC(Vtp tb, void* data) {
            tb->VCPU(R_registers[(size_t)data]) = 0;
        }

        static void verify_1N_INC(CVtp tb, void *data) {
            ASSERT(tb->VCPU(R_registers[(size_t)data]) == 7);
        }
    };

    struct INCTest2 {
        static void verify_10_INC(CVtp tb, void*) {
            ASSERT(tb->VCPU(R_registers[0]) == 5);
        }
    };

    run_basic_test(tb,
        "10 INC R(0)",
        NBRANCH_PROG(inc_prog2),
        INCTest2::verify_10_INC,
        NULL);

    for (int x = 0x11; x <= 0x1F; x++) {
        sprintf(buf, "%02X INC R(N)", x);
        memset(inc_prog1, x, sizeof(inc_prog1));

        run_test(tb,
            buf,
            NBRANCH_PROG(inc_prog1),
            INCTest1::setup_1N_INC,
            INCTest1::verify_1N_INC,
            (void*)(x & 0x0F));
    }

    //////////////////////////////////////////////////////////////////////////
    // 20-2F (DEC)

    char dec_prog1[3];

    // Same issue as INC on R(0) - R(0) is the program counter
    char dec_prog2[] = "\xF8\x07\xA3\xF8\x00\xB3\xD3\xA0\x20\x20\x20";

    struct DECTest1 {
        static void setup_2N_DEC(Vtp tb, void* data) {
            tb->VCPU(R_registers[(size_t)data]) = 13;
        }

        static void verify_2N_DEC(CVtp tb, void* data) {
            tb->VCPU(R_registers[(size_t)data]) == 10;
        }
    };

    struct DECTest2 {
        static void verify_20_DEC(CVtp tb, void*) {
            tb->VCPU(R_registers[0]) == 0xFFFD;
        }
    };

    run_basic_test(tb,
        "20 DEC R(0)",
        NBRANCH_PROG(dec_prog2),
        DECTest2::verify_20_DEC,
        NULL);

    for (int x = 0x21; x <= 0x2F; x++) {
        sprintf(buf, "%02X DEC R(N)", x);
        memset(dec_prog1, x, sizeof(dec_prog1));

        run_test(tb,
            buf,
            NBRANCH_PROG(dec_prog1),
            DECTest1::setup_2N_DEC,
            DECTest1::verify_2N_DEC,
            (void*)(x & 0x0F));
    }

    //////////////////////////////////////////////////////////////////////////
    // 30-3F (SHORT BRANCH)
    
    char br_test_progs1[13][20] = {
        // 30: unconditional
        // End condition: R(0) == 0x04 after two instructions
        "\x30\x0F\x30\x06\x00\x00\x00\x00\x00\x30\x0C\x30\x05\x00\x00\x30\x04",

        // 31: if Q == 1 || 39: if Q == 0
        // End condition: R(0) == 0x08 after six instructions
        "\x31\x00\x39\x06\x30\x00\x7B\x31\x0B\x30\x00\x7A\x39\x08",

        // 32: if D == 0 || 3A: if D != 0
        // End condition: R(0) == 0x0F after six instructions
        "\xF8\x00\x32\x05\x00\xF8\x03\x3A\x0A\x00\xF8\x00\x32\x0F",

        // 33: if DF == 1 || 3B: if DF == 0
        // End condition: R(0) == 0x0C after five instructions
        "\x33\xFF\x3B\x05\x00\xF8\x01\x76\x33\x0C\x30\xFE",

        // 38: SKP
        // End condition: R(0) == 0x02 after one instruction
        "\x38\x00\x00",

        // 34-37: branch if external flag N == 1
        // 3C-3F: inverse
        // End conditions: R(0) == 0x08 after two instructions
        "\x3C\xFF\x34\x08",
        "\x3D\xFF\x35\x08",
        "\x3E\xFF\x36\x08",
        "\x3F\xFF\x37\x08",
        "\x34\xFF\x3C\x08",
        "\x35\xFF\x3D\x08",
        "\x36\xFF\x3E\x08",
        "\x37\xFF\x3F\x08"
    };

    const char *br_prog_names[13] = {
        "30 BRANCH UNCOND",
        "31 BQ + 39 BNQ",
        "32 BZ + 3A BNZ",
        "33 BDF + 3B BNF",
        "38 SKP",
        "34 B1",
        "35 B2",
        "36 B3",
        "37 B4",
        "3C BN1",
        "3D BN2",
        "3E BN3",
        "3F BN4"
    };

    int br_prog_instrs[13] = {2, 6, 6, 5, 1, 2, 2, 2, 2, 2, 2, 2, 2};
    static int br_prog_targets[13] = {0x04, 0x08, 0x0F, 0x0C, 0x02, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08};

    struct ShortBranchTest1 {
        static void setup_3N_BR(Vtp tb, void *data) {
            size_t x = (size_t)data;

            if (x >= 5 && x <= 8) {
                tb->VCPU(B_registers[x - 5]) = 1;
            }
            else if (x >= 9) {
                tb->VCPU(B_registers[x - 9]) = 0;
            }
        }

        static void verify_3N_BR(CVtp tb, void *data) {
            ASSERT(tb->VCPU(R_registers[0]) == br_prog_targets[(size_t)data]);
        }
    };

    for (int x = 0; x < 13; x++) {
        run_test(tb,
            br_prog_names[x],
            PROG(br_test_progs1[x]),
            br_prog_instrs[x],
            ShortBranchTest1::setup_3N_BR,
            ShortBranchTest1::verify_3N_BR,
            (void*)x);
    }

    //////////////////////////////////////////////////////////////////////////
    // 40-4F (LOAD & ADVANCE)

    char lda_test_prog1[] = "\x40\xE0\x00\x00\x00\x40\x09\x00\x00\xE0";

    struct LoadAdvTest1 {
        static void setup_4N_LDA(Vtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            if (x != 0) {
                tb->VCPU(R_registers[x]) = 0x09;
            }
        }

        static void verify_4N_LDA(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(D_register) == 0xE0);
            ASSERT(tb->VCPU(R_registers[x]) == (x ? 0x0A : 0x02));
        }
    };

    for (int x = 0x40; x <= 0x4F; x++) {
        lda_test_prog1[0] = lda_test_prog1[5] = x;
        sprintf(buf, "%02X LDA *R(%d) -> D", x, (x & 0x0F));
        run_test(tb, buf,
                PROG(lda_test_prog1), 1,
                LoadAdvTest1::setup_4N_LDA,
                LoadAdvTest1::verify_4N_LDA,
                (void*)x);
    }

    //////////////////////////////////////////////////////////////////////////
    // 50-5F (STORE TO R(N))

    char str_test_prog1[] = "\x50\x00\x00\x00\x00\x00";

    struct StoreTest1 {
        static void setup_5N_STR(Vtp tb, void *data) {
            tb->VCPU(D_register) = 0x2C;
            
            size_t x = (size_t)data & 0x0F;
            if (x != 0) {
                tb->VCPU(R_registers[x]) = 0x05;
            }
        }

        static void verify_5N_STR(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(D_register) == 0x2C);
            ASSERT(tb->processor__DOT__m__DOT__mem[x ? 5 : 1] == 0x2C);
        }
    };

    for (int x = 0x50; x <= 0x5F; x++) {
        str_test_prog1[0] = x;
        sprintf(buf, "%02X STR D -> *R(%d)", x, (x & 0x0F));
        run_test(tb, buf,
                PROG(str_test_prog1), 1,
                StoreTest1::setup_5N_STR,
                StoreTest1::verify_5N_STR,
                (void*)x);
    }

    //////////////////////////////////////////////////////////////////////////
    // 60: INC R(X)

    struct IRXTest1 {
        static void setup_60_IRX(Vtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            if (x != 0) {
                tb->VCPU(R_registers[x]) = 0xB9;
            }
        }

        static void verify_60_IRX(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(R_registers[x]) == x ? 0xC0 : 0x02);
        }
    };

    for (int x = 0; x < 16; x++) {
        char instr = 0x60;

        sprintf(buf, "60 IRX, X == %d", x);
        run_test(tb, buf,
                &instr, 1, 1,
                IRXTest1::setup_60_IRX,
                IRXTest1::verify_60_IRX,
                (void*)x);
    }

    //////////////////////////////////////////////////////////////////////////
    // 61-6F (BUS I/O)
    // TODO

    //////////////////////////////////////////////////////////////////////////
    // 70-7F (CONTROL)
    // TODO

    //////////////////////////////////////////////////////////////////////////
    // 80-8F (GLO)

    struct GetLowTest1 {
        static void setup_8N_GLO(Vtp tb, void *data) {
            tb->VCPU(D_register) = 0xCC;

            size_t x = (size_t)data & 0x0F;
            if (x != 0) {
                tb->VCPU(R_registers[x]) = 0x2E74;
            }
        }

        static void verify_8N_GLO(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(D_register) == (x ? 0x74 : 0x01));
        }
    };

    for (unsigned char x = 0x80; x <= 0x8F; x++) {
        sprintf(buf, "%02X GLO R(%d).0 -> D", x, (x & 0x0F));
        run_test(tb, buf,
                (char*)&x, 1, 1,
                GetLowTest1::setup_8N_GLO,
                GetLowTest1::verify_8N_GLO,
                (void*)x);
    }

    //////////////////////////////////////////////////////////////////////////
    // 90-9F (GHI)

    struct GetHighTest1 {
        static void setup_9N_GHI(Vtp tb, void *data) {
            tb->VCPU(D_register) = 0xCC;

            size_t x = (size_t)data & 0x0F;
            if (x != 0) {
                tb->VCPU(R_registers[x]) = 0x2E74;
            }
        }

        static void verify_9N_GHI(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(D_register) == (x ? 0x2E : 0x00));
        }
    };

    for (unsigned char x = 0x90; x <= 0x9F; x++) {
        sprintf(buf, "%02X GHI R(%d).1 -> D", x, (x & 0x0F));
        run_test(tb, buf,
                (char*)&x, 1, 1,
                GetHighTest1::setup_9N_GHI,
                GetHighTest1::verify_9N_GHI,
                (void*)x);
    }

    //////////////////////////////////////////////////////////////////////////
    // A0-AF (PLO)

    struct PutLowTest1 {
        static void setup_AN_PLO(Vtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            tb->VCPU(D_register) = 0xCB;
            tb->VCPU(R_registers[x]) = 0x0000;
        }

        static void verify_AN_PLO(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(R_registers[x]) == 0x00CB);
        }
    };

    for (unsigned char x = 0xA0; x <= 0xAF; x++) {
        sprintf(buf, "%02X PLO D -> R(%d).0", x, (x & 0x0F));
        run_test(tb, buf,
                (char*)&x, 1, 1,
                PutLowTest1::setup_AN_PLO,
                PutLowTest1::verify_AN_PLO,
                (void*)x);
    }
    
    //////////////////////////////////////////////////////////////////////////
    // B0-BF (PHI)

    struct PutHighTest1 {
        static void setup_BN_PHI(Vtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            tb->VCPU(D_register) = 0xCB;
            tb->VCPU(R_registers[x]) = 0x0000;
        }

        static void verify_BN_PHI(CVtp tb, void *data) {
            size_t x = (size_t)data & 0x0F;
            ASSERT(tb->VCPU(R_registers[x]) == (x ? 0xCB00 : 0xCB01));
        }
    };

    for (unsigned char x = 0xB0; x <= 0xBF; x++) {
        sprintf(buf, "%02X PHI D -> R(%d).1", x, (x & 0x0F));
        run_test(tb, buf,
                (char*)&x, 1, 1,
                PutHighTest1::setup_BN_PHI,
                PutHighTest1::verify_BN_PHI,
                (void*)x);
    }
    
    //////////////////////////////////////////////////////////////////////////
    // C0-CF (LONG BRANCH)
    
    //////////////////// Branches \\\\\\\\\\\\\\\\\\\\
    // C0: unconditional long branch
    char lbr_test_prog1[] = "\xC0\x34\x56";

    struct LBRTest {
        static void verify_taken(CVtp tb, void *data) {
            ASSERT(tb->VCPU(R_registers[0]) == 0x3456);
        }

        static void verify_not_taken(CVtp tb, void *data) {
            ASSERT(tb->VCPU(R_registers[0]) == 0x03);
        }
    };

    run_basic_test(tb, "C0 LBR NOCOND", PROG(lbr_test_prog1), 1, LBRTest::verify_taken, NULL);

    // C1: LBQ (Q == 1)
    // C9: LBNQ (Q == 0)
    char lbq_test_prog1[] = "\xC1\x34\x56";
    char lbq_test_prog2[] = "\xC9\x34\x56";

    struct LBQTest {
        static void setup_C1C9_LBQ(Vtp tb, void *data) { tb->VCPU(Q) = 1; }
        static void setup_C1C9_LBNQ(Vtp tb, void *data) { tb->VCPU(Q) = 0; }
    };

    run_test(tb, "C1 LBQ Q==0", PROG(lbq_test_prog1), 1,
            LBQTest::setup_C1C9_LBNQ, LBRTest::verify_not_taken, NULL);
    run_test(tb, "C1 LBQ Q==1", PROG(lbq_test_prog1), 1,
            LBQTest::setup_C1C9_LBQ, LBRTest::verify_taken, NULL);
    run_test(tb, "C9 LBNQ Q==0", PROG(lbq_test_prog2), 1,
            LBQTest::setup_C1C9_LBNQ, LBRTest::verify_taken, NULL);
    run_test(tb, "C9 LBNQ Q==1", PROG(lbq_test_prog2), 1,
            LBQTest::setup_C1C9_LBQ, LBRTest::verify_not_taken, NULL);

    // C2: LBZ (D == 0)
    // CA: LBNZ (D != 0)
    char lbz_test_prog1[] = "\xC2\x34\x56";
    char lbz_test_prog2[] = "\xCA\x34\x56";

    struct LBZTest {
        static void setup_C2CA_LBZ(Vtp tb, void *data) { tb->VCPU(D_register) = 0; }
        static void setup_C2CA_LBNZ(Vtp tb, void *data) { tb->VCPU(D_register) = 0xFE; }
    };

    run_test(tb, "C2 LBZ D==0", PROG(lbz_test_prog1), 1,
            LBZTest::setup_C2CA_LBZ, LBRTest::verify_taken, NULL);
    run_test(tb, "C2 LBZ D==1", PROG(lbz_test_prog1), 1,
            LBZTest::setup_C2CA_LBNZ, LBRTest::verify_not_taken, NULL);
    run_test(tb, "CA LBNZ D==0", PROG(lbz_test_prog2), 1,
            LBZTest::setup_C2CA_LBZ, LBRTest::verify_not_taken, NULL);
    run_test(tb, "CA LBNZ D==1", PROG(lbz_test_prog2), 1,
            LBZTest::setup_C2CA_LBNZ, LBRTest::verify_taken, NULL);

    // C3: LBDF (DF == 1)
    // CB: LBNF (DF == 0)
    char lbdf_test_prog1[] = "\xC3\x34\x56";
    char lbdf_test_prog2[] = "\xCB\x34\x56";

    struct LBDFTest {
        static void setup_C3CB_LBDF(Vtp tb, void *data) { tb->VCPU(DF) = 1; }
        static void setup_C3CB_LBNF(Vtp tb, void *data) { tb->VCPU(DF) = 0; }
    };

    run_test(tb, "C3 LBDF D==0", PROG(lbdf_test_prog1), 1,
            LBDFTest::setup_C3CB_LBDF, LBRTest::verify_taken, NULL);
    run_test(tb, "C3 LBDF D==1", PROG(lbdf_test_prog1), 1,
            LBDFTest::setup_C3CB_LBNF, LBRTest::verify_not_taken, NULL);
    run_test(tb, "CB LBNF D==0", PROG(lbdf_test_prog2), 1,
            LBDFTest::setup_C3CB_LBNF, LBRTest::verify_taken, NULL);
    run_test(tb, "CB LBNF D==1", PROG(lbdf_test_prog2), 1,
            LBDFTest::setup_C3CB_LBDF, LBRTest::verify_not_taken, NULL);

    // C4: NOP
    char nop_test_prog1[] = "\xC4\xC4\xC4";
    Vprocessor old_state;

    struct NOPTest {
        static void setup_C4_NOP(Vtp tb, void *data) {
            memcpy(data, tb, sizeof(Vprocessor));
        }

        static void verify_C4_NOP(CVtp tb, void *data) {
            Vprocessor *old = (Vprocessor *)data;
            old->VCPU(R_registers[0]) += 3 + 1; // + 1 since FETCH will increase PC

            ASSERT(memcmp(old->processor__DOT__m__DOT__mem,
                          tb->processor__DOT__m__DOT__mem,
                          sizeof(tb->processor__DOT__m__DOT__mem)) == 0);
            
            ASSERT(memcmp(old->VCPU(R_registers),
                          tb->VCPU(R_registers),
                          sizeof(tb->VCPU(R_registers))) == 0);
            
            ASSERT(memcmp(old->VCPU(B_registers),
                          tb->VCPU(B_registers),
                          sizeof(tb->VCPU(B_registers))) == 0);

            ASSERT(old->VCPU(mode) == tb->VCPU(mode));
            ASSERT(old->VCPU(state) == tb->VCPU(state));
            ASSERT(old->VCPU(alu_operand) == tb->VCPU(alu_operand));
            ASSERT(old->VCPU(alu_result) == tb->VCPU(alu_result));
            ASSERT(old->VCPU(alu_carry) == tb->VCPU(alu_carry));
            ASSERT(old->VCPU(Q) == tb->VCPU(Q));
            ASSERT(old->VCPU(DF) == tb->VCPU(DF));
            ASSERT(old->VCPU(IE) == tb->VCPU(IE));
            ASSERT(old->VCPU(N_register) == tb->VCPU(N_register));
            ASSERT(old->VCPU(P_register) == tb->VCPU(P_register));
            ASSERT(old->VCPU(X_register) == tb->VCPU(X_register));
            ASSERT(old->VCPU(I_register) == tb->VCPU(I_register));
            ASSERT(old->VCPU(D_register) == tb->VCPU(D_register));
        }
    };

    run_test(tb, "C4 NOP", NBRANCH_PROG(nop_test_prog1),
            NOPTest::setup_C4_NOP, NOPTest::verify_C4_NOP, &old_state);

    //////////////////// Long skips \\\\\\\\\\\\\\\\\\\\
    // C5: LNQ (Q == 0)
    // CD: LSQ (Q == 1)
    struct LSKPTest {
        static void verify_taken(CVtp tb, void *) {
            ASSERT(tb->VCPU(R_registers[0]) == 0x03);
        }

        static void verify_not_taken(CVtp tb, void *) {
            ASSERT(tb->VCPU(R_registers[0]) == 0x01);
        }
    };

    char lnq_test_prog1 = 0xC5;
    char lsq_test_prog1 = 0xCD;

    struct LNQTest {
        static void setup_C5CD_LNQ(Vtp tb, void *) { tb->VCPU(Q) = 0; }
        static void setup_C5CD_LSQ(Vtp tb, void *) { tb->VCPU(Q) = 1; }
    };

    run_test(tb, "C5 LNQ Q==0", &lnq_test_prog1, 1, 1,
            LNQTest::setup_C5CD_LNQ, LSKPTest::verify_taken, NULL);
    run_test(tb, "C5 LNQ Q==1", &lnq_test_prog1, 1, 1,
            LNQTest::setup_C5CD_LSQ, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CD LSQ Q==0", &lsq_test_prog1, 1, 1,
            LNQTest::setup_C5CD_LNQ, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CD LSQ Q==1", &lsq_test_prog1, 1, 1,
            LNQTest::setup_C5CD_LSQ, LSKPTest::verify_taken, NULL);

    // C6: LSNZ (D != 0)
    // CE: LSZ (D == 0)
    char lsnz_test_prog1 = 0xC6;
    char lsz_test_prog1 = 0xCE;

    struct LSNZTest {
        static void setup_C6CE_LSNZ(Vtp tb, void *) { tb->VCPU(D_register) = 0x65; }
        static void setup_C6CE_LSZ(Vtp tb, void *) { tb->VCPU(D_register) = 0; }
    };

    run_test(tb, "C6 LSNZ D!=0", &lsnz_test_prog1, 1, 1,
            LSNZTest::setup_C6CE_LSNZ, LSKPTest::verify_taken, NULL);
    run_test(tb, "C6 LSNZ D==0", &lsnz_test_prog1, 1, 1,
            LSNZTest::setup_C6CE_LSZ, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CE LSZ D!=0", &lsz_test_prog1, 1, 1,
            LSNZTest::setup_C6CE_LSNZ, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CE LSZ D==0", &lsz_test_prog1, 1, 1,
            LSNZTest::setup_C6CE_LSZ, LSKPTest::verify_taken, NULL);

    // C7: LSNF (DF == 0)
    // CF: LSDF (DF == 1)
    char lsnf_test_prog1 = 0xC7;
    char lsdf_test_prog1 = 0xCF;

    struct LSNFTest {
        static void setup_C7CF_LSNF(Vtp tb, void *) { tb->VCPU(DF) = 0; }
        static void setup_C7CF_LSDF(Vtp tb, void *) { tb->VCPU(DF) = 1; }
    };

    run_test(tb, "C7 LSNF DF==0", &lsnf_test_prog1, 1, 1,
            LSNFTest::setup_C7CF_LSNF, LSKPTest::verify_taken, NULL);
    run_test(tb, "C7 LSNF DF==1", &lsnf_test_prog1, 1, 1,
            LSNFTest::setup_C7CF_LSDF, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CF LSDF DF==0", &lsdf_test_prog1, 1, 1,
            LSNFTest::setup_C7CF_LSNF, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CF LSDF DF==1", &lsdf_test_prog1, 1, 1,
            LSNFTest::setup_C7CF_LSDF, LSKPTest::verify_taken, NULL);

    // C8: NLBR (uncond. long skip)
    char nlbr_test_prog1 = 0xC8;

    run_basic_test(tb, "C8 NLBR uncond. long skip", &nlbr_test_prog1, 1, 1,
            LSKPTest::verify_taken, NULL);

    // CC: LSIE (IE == 1)
    char lsie_test_prog1 = 0xCC;

    struct LSIETest {
        static void setup_CC_POS(Vtp tb, void *) { tb->VCPU(IE) = 1; }
        static void setup_CC_NEG(Vtp tb, void *) { tb->VCPU(IE) = 0; }
    };

    run_test(tb, "CC LSIE IE==0", &lsie_test_prog1, 1, 1,
            LSIETest::setup_CC_NEG, LSKPTest::verify_not_taken, NULL);
    run_test(tb, "CC LSIE IE==1", &lsie_test_prog1, 1, 1,
            LSIETest::setup_CC_POS, LSKPTest::verify_taken, NULL);

    //////////////////////////////////////////////////////////////////////////
    // D0-DF (SET P TO N)

    struct SetPTest1 {
        static void verify_DN_SEP(CVtp tb, void *data) {
            ASSERT(tb->VCPU(P_register) == (size_t)data);
        }
    };

    for (unsigned char x = 0xD0; x <= 0xDF; x++) {
        sprintf(buf, "%02X SEP", x);
        run_basic_test(tb, buf, (char*)&x, 1, 1, SetPTest1::verify_DN_SEP, (void*)(x & 0x0F));
    }

    //////////////////////////////////////////////////////////////////////////
    // E0-EF (SET X TO N)

    struct SetXTest1 {
        static void verify_EN_SEX(CVtp tb, void *data) {
            ASSERT(tb->VCPU(X_register) == (size_t)data);
        }
    };

    for (unsigned char x = 0xE0; x <= 0xEF; x++) {
        sprintf(buf, "%02X SEX", x);
        run_basic_test(tb, buf, (char*)&x, 1, 1, SetXTest1::verify_EN_SEX, (void*)(x & 0x0F));
    }
    
    //////////////////////////////////////////////////////////////////////////
    // F0-FF (ARITH)
    
    // F0: load via X
    // TODO

    // F1-FF: OR via X
    char arith_via_x_test_prog1[] = "\xF1\x54";

    struct ArithViaXTest {
        static void setup_FN_arith(Vtp tb, void *data) {
            tb->VCPU(D_register) = 0xC1;

            size_t x = (size_t)data & 0x0F;
            if (x != 0) {
                tb->VCPU(R_registers[x]) = 0x01;
            }
        }

        static void verify_IMM(CVtp tb, void *data) {
            size_t x = (size_t)data;
            if (x > 0xF8 && x != 0xFE) {
                ASSERT(tb->VCPU(R_registers[0]) == 0x03);
            }
        }

        static void verify_OR(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0xD5);
            verify_IMM(tb, data);
        }
        
        static void verify_AND(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x40);
            verify_IMM(tb, data);
        }
        
        static void verify_XOR(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x95);
            verify_IMM(tb, data);
        }

        static void verify_ADD(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x15);
            ASSERT(tb->VCPU(DF) == 1);
            verify_IMM(tb, data);
        }

        static void verify_SUB(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x93);
            ASSERT(tb->VCPU(DF) == 0);
            verify_IMM(tb, data);
        }

        static void verify_SHR(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x60);
            ASSERT(tb->VCPU(DF) == 1);
        }

        static void verify_SHL(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x82);
            ASSERT(tb->VCPU(DF) == 1);
        }

        static void verify_mem_SUB(CVtp tb, void *data) {
            ASSERT(tb->VCPU(D_register) == 0x6D);
            ASSERT(tb->VCPU(DF) == 1);
            verify_IMM(tb, data);
        }

        // TODO: F8 (LDI)
    };

    const char *arith_via_x_names[16] = {
        NULL, "OR", "AND", "XOR", "ADD", "SD", "SHR", "SM",
        NULL, "ORI", "ANI", "XRI", "ADI", "SDI", "SHL", "SMI"
    };

    for (unsigned char x = 0xF1; ; x++) {
        if (x == 0xF8) continue;

        arith_via_x_test_prog1[0] = x;
        void (*verifier)(CVtp, void*) = NULL;

        switch (x & 0x07) {
            case 0x01:
                verifier = ArithViaXTest::verify_OR;
                break;
            case 0x02:
                verifier = ArithViaXTest::verify_AND;
                break;
            case 0x03:
                verifier = ArithViaXTest::verify_XOR;
                break;
            case 0x04:
                verifier = ArithViaXTest::verify_ADD;
                break;
            case 0x05:
                verifier = ArithViaXTest::verify_SUB;
                break;
            case 0x06:
                if (x == 0xF6) verifier = ArithViaXTest::verify_SHR;
                else verifier = ArithViaXTest::verify_SHL;
                break;
            case 0x07:
                verifier = ArithViaXTest::verify_mem_SUB;
                break;
        }

        sprintf(buf, "%02X %s", x, arith_via_x_names[x & 0x0F]);
        run_test(tb, buf, PROG(arith_via_x_test_prog1), 1,
                ArithViaXTest::setup_FN_arith, verifier, NULL);

        if (x == 0xFF) break;
    }
    
    return 0;
}
