/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_ball.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define BG_RED(x) ((x))
#define BG_GREEN(x) ((x)+2)
#define BG_BLUE(x) ((x)+4)
#define PADDLE(x) ((x))
#define BALL_H(x) ((x)+2)
#define BALL_V(x) ((x)+4)
#define BRICK_STATUS0(x) ((x)+6)
#define BRICK_STATUS1(x) ((x)+8)
#define BRICK_STATUS2(x) ((x)+10)
#define BRICK_STATUS3(x) ((x)+12)
#define BRICK_STATUS4(x) ((x)+14)
#define BRICK_STATUS5(x) ((x)+16)
#define BRICK_STATUS6(x) ((x)+18)
#define HEART_STATUS(x)  ((x)+20)
#define AUDIO(x)         ((x)+22)
#define SCORE1(x)        ((x)+24)
#define SCORE2(x)        ((x)+26)
#define SCORE3(x)        ((x)+28)
#define GAMESTATUS(x)        ((x)+30)

/*
 * Information about our device
 */
struct vga_ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        hardware_p alldata;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_alldata(hardware_p *alldata)
{
	/*iowrite16(alldata->red, BG_RED(dev.virtbase) );
	iowrite16(alldata->green, BG_GREEN(dev.virtbase) );
	iowrite16(alldata->blue, BG_BLUE(dev.virtbase) );*/
	iowrite16(alldata->paddle_left, PADDLE(dev.virtbase) );
	iowrite16(alldata->ball_h, BALL_H(dev.virtbase) );
        iowrite16(alldata->ball_v, BALL_V(dev.virtbase) );
        iowrite16(alldata->brick_status0, BRICK_STATUS0(dev.virtbase) );
        iowrite16(alldata->brick_status1, BRICK_STATUS1(dev.virtbase) );
        iowrite16(alldata->brick_status2, BRICK_STATUS2(dev.virtbase) );
        iowrite16(alldata->brick_status3, BRICK_STATUS3(dev.virtbase) );
        iowrite16(alldata->brick_status4, BRICK_STATUS4(dev.virtbase) );
        iowrite16(alldata->brick_status5, BRICK_STATUS5(dev.virtbase) );
        iowrite16(alldata->brick_status6, BRICK_STATUS6(dev.virtbase) );
        iowrite16(alldata->heart_status,  HEART_STATUS(dev.virtbase));
        iowrite16(alldata->audio_choose,  AUDIO(dev.virtbase));
        iowrite16(alldata->score1,  SCORE1(dev.virtbase));
        iowrite16(alldata->score2,  SCORE2(dev.virtbase));
        iowrite16(alldata->score3,  SCORE3(dev.virtbase));
        iowrite16(alldata->gamestatus,  GAMESTATUS(dev.virtbase));

	dev.alldata = *alldata;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long vga_ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	vga_ball_arg_t vla;

	switch (cmd) {
	case VGA_BALL_WRITE_alldata:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_alldata(&vla.alldata);
		break;

	case VGA_BALL_READ_alldata:
	  	vla.alldata = dev.alldata;
		if (copy_to_user((vga_ball_arg_t *) arg, &vla,
				 sizeof(vga_ball_arg_t)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_ball_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = vga_ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_ball_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init vga_ball_probe(struct platform_device *pdev)
{
        hardware_p beige = { 0xf9, 0xe4, 0xb7 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&vga_ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        write_alldata(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_ball_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_ball_of_match),
	},
	.remove	= __exit_p(vga_ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_ball_driver, vga_ball_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit vga_ball_exit(void)
{
	platform_driver_unregister(&vga_ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_ball_init);
module_exit(vga_ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
