(*
File: SCANNER.MLL
Description: Scans input
*)

{ open Parser }

rule token = parse
  [' ' '\t' '\r' '\n'] { token lexbuf } (* Whitespace *)
| "/*"     { multicomment lexbuf }      (* Comments *)
| "//" 	   { singlecomment lexbuf}
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| ';'      { SEMI }
| ','      { COMMA }
| '.'	     { DOT }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '='      { ASSIGN }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&&"     { AND }
| "||"     { OR }
| "!"      { NOT }
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "while"  { WHILE }
| "return" { RETURN }
| "int"    { INT }
| "bool"   { BOOL }
| "double" { DOUBLE }
| "string" { STRING }
| "matrix" { MATRIX }
| "image"  { IMAGE }
| "red"    { RED }
| "green"  { GREEN }
| "blue"   { BLUE }
| "rowsize" { ROWSIZE }
| "colsize" { COLSIZE }
| "void"   { VOID }
| "true"   { TRUE }
| "false"  { FALSE }
| '['	     { LBRACK }
| ']'	     { RBRACK }
| "func"   { FUNC }
| '%'      { MOD }
| "->"	   { ARROW }

| ['0'-'9']+ as lxm { INT_LITERAL(int_of_string lxm) }
| ['0'-'9']*'.'['0'-'9']+ as lxm { DBL_LITERAL(float_of_string lxm) }
| ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']* as lxm { ID(lxm) }
| '"' ([^ '"']* as lxm) '"' { STR_LITERAL(lxm) }
| eof { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and singlecomment = parse
  "\n" { token lexbuf }
| _    { singlecomment lexbuf }

and multicomment = parse
  "*/" { token lexbuf }
| _    { multicomment lexbuf }
