(*
  Allie Costa
  Laura Smerling
  Jacob Penn
  Laura Matos
*)

(* Semantically-checked Abstract Syntax Tree and functions for printing it *)

open Ast

type sbind = typ * string

type sexpr = typ * sx
and sx =
    SLiteral of int
  | SFliteral of string
  | SBoolLit of bool
  | SSliteral of string
  | SNodeLit of sexpr * sexpr
  | SNodeData of sexpr
  | SNodeParent of sexpr
  | SNodeLevel of sexpr
  | SLNodeShift of sexpr
  | SRNodeShift of sexpr
  | SSwapNode of sexpr * sexpr
  | SFindNode of sexpr * sexpr
  | SAddNode of sexpr * sexpr
  | SRemoveNode of sexpr * sexpr
  | SId of string
  | SBinop of sexpr * op * sexpr
  | SUnop of uop * sexpr
  | SAssign of string * sexpr
  | SFunCall of string * sexpr list
  | SNoexpr

type sstmt =
    SSeq of sstmt list
  | SExpr of sexpr
  | SReturn of sexpr
  | SIf of sexpr * sstmt * sstmt
  | SVarDec of sbind * sexpr
  | SFor of sexpr * sexpr * sexpr * sstmt
  | SWhile of sexpr * sstmt

type sfunc_decl = {
    styp : typ;
    sfname : string;
    sformals : bind list;
    sbody : sstmt list;
  }
(*
type SStmt = sstmt

type SFunction = sfunc_decl
*)
type sitem =
      SStmt of sstmt
    | SFunction of sfunc_decl

type sprogram = sitem list
(*type sprogram = sfunc_decl list * sstmt list*)

(* Pretty-printing functions *)

let rec string_of_sexpr (t, e) = 
  "(" ^ string_of_typ t ^ " : " ^ (match e with
    SLiteral(l) -> string_of_int l
  | SBoolLit(true) -> "true"
  | SBoolLit(false) -> "false"
  | SFliteral(l) -> l
  | SSliteral(l) -> "\"" ^ l ^ "\""
  | SNodeLit(e1, e2) -> string_of_sexpr e1 ^ "," ^ string_of_sexpr e2 
  | SNodeData(e) -> string_of_sexpr e ^ ".data"
  | SNodeParent(e) -> string_of_sexpr e ^ ".parent"
  | SNodeLevel(e) -> string_of_sexpr e ^ ".level"
  | SLNodeShift(e) -> string_of_sexpr e ^ "<<"
  | SRNodeShift(e) -> string_of_sexpr e ^ ">>"
  | SSwapNode(e1, e2) -> string_of_sexpr e1 ^ "^" ^ string_of_sexpr e2
  | SFindNode (g,param) -> string_of_sexpr g
        ^ ".find_node(" ^ string_of_sexpr param ^ ")"
  | SAddNode (g,param) -> string_of_sexpr g
        ^ ".add_node(" ^ string_of_sexpr param ^ ")"
  | SRemoveNode (g,param) -> string_of_sexpr g
        ^ ".remove_node(" ^ string_of_sexpr param ^ ")"
  | SId(s) -> s
  | SBinop(e1, o, e2) ->
      string_of_sexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_sexpr e2
  | SUnop(o, e) -> string_of_uop o ^ string_of_sexpr e
  | SAssign(v, e) -> v ^ " = " ^ string_of_sexpr e
  | SFunCall(s,e) -> s ^ "(" ^ String.concat ", " (List.map string_of_sexpr e) ^ ")"
  | SNoexpr -> ""
				  ) ^ ")"				     

let rec string_of_sstmt = function
    SSeq(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_sstmt stmts) ^ "}\n"
  | SExpr(expr) -> string_of_sexpr expr ^ ";\n";
  | SReturn(expr) -> "return " ^ string_of_sexpr expr ^ ";\n";
  | SIf(e,s1,s2) -> "If(" ^ string_of_sexpr e ^ ") {" ^ string_of_sstmt s1 ^ "} Else {" ^ string_of_sstmt s2 ^ "}"
  | SVarDec((t, s),e) -> string_of_typ t ^ " " ^ s ^ " = " ^ string_of_sexpr e
  | SFor(e1, e2, e3, s) ->
      "for (" ^ string_of_sexpr e1  ^ " ; " ^ string_of_sexpr e2 ^ " ; " ^
      string_of_sexpr e3  ^ ") " ^ string_of_sstmt s
  | SWhile(e, s) -> "while (" ^ string_of_sexpr e ^ ") " ^ string_of_sstmt s

let string_of_sfunc_decl func_decl =
  func_decl.sfname ^ "<" ^ string_of_typ func_decl.styp ^ 
  ">(" ^ String.concat ", " (List.map snd func_decl.sformals) ^
  "){" ^
  String.concat "" (List.map string_of_sstmt func_decl.sbody) ^
  "}\n"


let string_of_sprogram stmts =
  String.concat " " (List.map string_of_sstmt stmts)
(*   String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_sfunc_decl funcs) *)
