(*
  Allie Costa
  Laura Smerling
  Jacob Penn
  Laura Matos
*)

(* Abstract Syntax Tree and functions for printing it *)

type op = Add | Sub | Mult | Mod | Div | Equal | Neq | Less | Leq | Greater | Geq |
          And | Or

type uop = Neg | Not

type typ = String | Int | Bool | Float | Void | Node of typ

type bind = typ * string

type expr =
    Literal of int
  | Fliteral of string
  | BoolLit of bool
  | Sliteral of string
  | NodeLit of expr * expr
  | NodeData of expr
  | NodeParent of expr
  | NodeLevel of expr
  | LNodeShift of expr
  | RNodeShift of expr
  | SwapNode of expr * expr
  | FindNode of expr * expr
  | AddNode of expr * expr
  | RemoveNode of expr * expr
  | Id of string
  | Binop of expr * op * expr
  | Unop of uop * expr
  | Assign of string * expr
  | FunCall of string * expr list
  | Noexpr


type stmt =
    (*Block of stmt list*)
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | VarDec of bind * expr
(*   | Assign of string * expr *)  
  | For of expr * expr * expr * stmt
  | While of expr * stmt
  | Seq of stmt list

type func_decl = {
        typ: typ;
        fname: string;
        formals: bind list;
        body: stmt list;
}    

type item =
      Stmt of stmt
    | Function of func_decl

type program = item list 

(* Pretty-printing functions *)
let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Mod -> "%"
  | Div -> "/"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"

let string_of_uop = function
    Neg -> "-"
  | Not -> "!"


let rec string_of_typ = function
    Int -> "int"
  | Bool -> "bool"
  | Float -> "float"
  | Void -> "void"
  | String -> "string"
  | Node(t) -> "node<" ^ string_of_typ t ^ ">" 


let rec string_of_expr = function
    Literal(l) -> string_of_int l
  | Fliteral(l) -> l
  | Sliteral(l) -> "\"" ^ l ^ "\""
  | BoolLit(true) -> "true"
  | BoolLit(false) -> "false"
  | NodeLit(e1, e2) -> string_of_expr e1 ^ "," ^ string_of_expr e2 
  | NodeData(e) -> string_of_expr e ^ ".data"
  | NodeParent(e) -> string_of_expr e ^ ".parent"
  | NodeLevel(e) -> string_of_expr e ^ ".level"
  | LNodeShift(e) -> string_of_expr e ^ "<<"
  | RNodeShift(e) -> string_of_expr e ^ ">>"
  | SwapNode(e1, e2) -> string_of_expr e1 ^ "^" ^ string_of_expr e2
  | FindNode (g,param) -> string_of_expr g
        ^ ".find_node(" ^ string_of_expr param ^ ")"
  | AddNode (g,param) -> string_of_expr g
        ^ ".add_node(" ^ string_of_expr param ^ ")"
  | RemoveNode (g,param) -> string_of_expr g
        ^ ".remove_node(" ^ string_of_expr param ^ ")"
  | Id(s) -> s
  | Binop(e1, o, e2) -> string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  | Unop(o, e) -> string_of_uop o ^ string_of_expr e
  | FunCall(s,e) -> s ^ "(" ^ String.concat ", " (List.map string_of_expr e) ^ ")"
  | Assign(s,e) -> s ^ " = " ^ string_of_expr e 
  | Noexpr -> ""

let rec string_of_stmt = function
    Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e,s1,s2) -> "If(" ^ string_of_expr e ^ ") {" ^ string_of_stmt s1 ^ "} Else {" ^ string_of_stmt s2 ^ "}"
  | VarDec((t, s),e) -> string_of_typ t ^ " " ^ s ^ " = " ^ string_of_expr e
  (* | Assign(s,e) -> s ^ " = " ^ string_of_expr e *)
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
      string_of_expr e3  ^ ") " ^ string_of_stmt s
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s
  | Seq(l) -> String.concat ", " (List.map string_of_stmt l)



let string_of_func_decl func_decl =  
  func_decl.fname ^ "<" ^ string_of_typ func_decl.typ ^ 
  ">(" ^ String.concat ", " (List.map snd func_decl.formals) ^
  "){" ^
  String.concat "" (List.map string_of_stmt func_decl.body) ^
  "}\n"

let string_of_vdecl (t, id) = string_of_typ t ^ " " ^ id ^ ";\n"

let string_of_fdecl fdecl =
  string_of_typ fdecl.typ ^ " " ^
  fdecl.fname ^ "(" ^ String.concat ", " (List.map snd fdecl.formals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_vdecl fdecl.formals) ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_items item = match item with
   Stmt(x) -> string_of_stmt x
  |Function(y) -> string_of_fdecl y

 let string_of_program stmts =
      String.concat " " (List.map string_of_items stmts)

