{ open Parser }

rule token = parse
  [' ' '\t' '\r' '\n'] { token lexbuf }
	(* comment *)
	| "/*" { long_comment lexbuf }
	| "//" { short_comment lexbuf }
	(* operators *)
	| '+' { PLUS }
	| '-' { MINUS }
	| '*' { TIMES }
	| '/' { DIVIDE }
	| '=' { ASSIGN }
	| '^' { POWER }
	| '|' { ABS }
	| ''' { TRANSPOSE }
	| ".*" { DOTTIMES }
	| "./" { DOTDIVIDE }
	| ".^" { DOTPOWER }
	(* logical operators *)
	| '>' { GT }
	| '<' { LT }
	| "<=" { LSEQ }
	| ">=" { GTEQ }
	| "==" { EQ }
	| "!=" { NEQ }
	| "&&" { AND }
	| "||" { OR }
	| '!' { NOT }
	(* special operators *)
	| '[' { LSQ }
	| ']' { RSQ }
	| '(' { LPA }
	| ')' { RPA }
	| '{' { LBR }
	| '}' { RBR }
	| ';' { SEMI }
	| ',' { COMMA }
	| ':' { COLON }
	(* data types *)
	| "int" { INT }
	| "double" { DOUBLE }
	| "boolean" { BOOLEAN }
	| "string" { STRING }
	| "matrix" { MATRIX }
	| "void" { VOID }
	(* control flow *)
	| "if" { IF }
	| "else" { ELSE }
	| "for" { FOR }
	| "while" { WHILE }
	| "continue" { CONTINUE }
	| "break" { BREAK }
	| "func" { FUNC }
	| "return" { RETURN }
	(* constants *)
	| "PI" { PI }
	| "True" { TRUE }
	| "False" { FALSE }
	(* literals *)
	| ['0'-'9']+ as lit { INTLIT(int_of_string lit) }
	| '"' ([^ '"']* as lit) '"' { STRLIT(lit) }
	| (['0'-'9']*)'.'(['0'-'9']+) as lit { DOUBLELIT(float_of_string lit) }
	| (['0'-'9']+)'.'(['0'-'9']*) as lit { DOUBLELIT(float_of_string lit) }
	| ['a'-'z']['a'-'z' '0'-'9' 'A'-'Z' '_']* as lit { ID(lit) }
	(* special cases *)
	| eof { EOF }
	| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and long_comment = parse 
		"*/" { token lexbuf } 
		(* Return to normal scanning *)
		| _ { long_comment lexbuf }

and short_comment = parse 
		"\n" { token lexbuf } 
		| _ { short_comment lexbuf }
