(*
Author: Chunli Fu
Copyright 2018, MathLight
*)

open Ast

type sexpr = (data_type * (int * int)) * sx
and sx =
    SIntLit of int
  | SDoubleLit of float
  | SBoolLit of bool
  | SStrLit of string
  | SRange of sexpr * sexpr
  | SMatrixLit of float array array
  | SMatrix1DElement of string * sexpr
  | SMatrix2DElement of string * sexpr * sexpr
  | SMatrix1DModify of string * sexpr * sexpr
  | SMatrix2DModify of string * (sexpr * sexpr) * sexpr
  | SId of string
  | SBinop of sexpr * operator * sexpr
  | SUnop of unary_operator * sexpr
  | SMatrixOp of string * operator * string
  | SAssign of string * sexpr
  | SCall of string * sexpr list
  | SNoexpr

type sbind = data_type * string * (int * int) * sexpr

type sstmt =
    SBlock of sstmt list
  | SExpr of sexpr
  | SReturn of sexpr
  | SIf of sexpr * sstmt * sstmt
  | SFor of sexpr * sexpr * sexpr * sstmt
  | SForRange of string * sexpr * sstmt
  | SWhile of sexpr * sstmt
  | SContinue of sexpr
  | SBreak of sexpr

type sfunc_decl = {
    styp : data_type;
    ssize : (int * int);
    sfname : string;
    sargs : sbind list;
    slocals : sbind list;
    sbody : sstmt list;
  }

type sprogram = sbind list * sfunc_decl list

(* Pretty-printing functions *)

let rec string_of_sexpr ((t, (r, c)), e) =
  "(" ^ string_of_typ t ^ " with size (" ^ string_of_int r ^ ", " ^ string_of_int c ^ ") : " ^
  (match e with
    SIntLit(l) -> string_of_int l
  | SBoolLit(true) -> "true"
  | SBoolLit(false) -> "false"
  | SStrLit(s) -> s
  | SDoubleLit(l) -> string_of_float(l)
  | SId(s) -> s
  | SBinop(e1, o, e2) ->
      string_of_sexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_sexpr e2
  | SUnop(o, e) -> string_of_uop o ^ string_of_sexpr e
  | SAssign(v, e) -> v ^ " = " ^ string_of_sexpr e
  | SCall(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_sexpr el) ^ ")"
  | SNoexpr -> ""
  | SMatrixLit(_) -> "SMatrixLit"
  | SMatrix1DElement(s, e) -> "SMatrix1DElement " ^ s ^ "[" ^ string_of_sexpr(e) ^ "]"
  | SMatrix2DElement(s, e1, e2) ->
      "SMatrix2DElement " ^ s ^ "[" ^ string_of_sexpr(e1) ^ ", " ^ string_of_sexpr(e2) ^ "]"
  | SMatrix1DModify(s, e1, e2) ->
      s ^ "[" ^ string_of_sexpr(e1) ^ "]" ^ " = " ^ string_of_sexpr(e2)
  | SMatrix2DModify(s, (e1,e2), e3) ->
      s ^ "[" ^ string_of_sexpr(e1) ^ ", " ^ string_of_sexpr(e2) ^ "]" ^ " = " ^ string_of_sexpr(e3)
  | SRange(e1, e2) -> "SRange: " ^ string_of_sexpr(e1) ^ ": " ^ string_of_sexpr(e2)
  | SMatrixOp(m1, o, m2) -> "SMatrixOp " ^ m1 ^ string_of_op(o) ^ m2
          ) ^ ")"

let rec string_of_sstmt = function
    SBlock(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_sstmt stmts) ^ "}\n"
  | SExpr(expr) -> string_of_sexpr expr ^ ";\n";
  | SReturn(expr) -> "return " ^ string_of_sexpr expr ^ ";\n";
  | SIf(e, s, SBlock([])) ->
      "if (" ^ string_of_sexpr e ^ ")\n" ^ string_of_sstmt s
  | SIf(e, s1, s2) ->  "if (" ^ string_of_sexpr e ^ ")\n" ^
      string_of_sstmt s1 ^ "else\n" ^ string_of_sstmt s2
  | SFor(e1, e2, e3, s) ->
      "for (" ^ string_of_sexpr e1  ^ " ; " ^ string_of_sexpr e2 ^ " ; " ^
      string_of_sexpr e3  ^ ") " ^ string_of_sstmt s
  | SWhile(e, s) -> "while (" ^ string_of_sexpr e ^ ") " ^ string_of_sstmt s
  | SBreak(e) -> "break " ^ string_of_sexpr e
  | SContinue(e) -> "continue " ^ string_of_sexpr e
  | SForRange(_,_,_) -> "SForRange"

let string_of_svdecl (t, id, _, sexpr) = string_of_typ t ^ " " ^ id ^ " " ^ string_of_sexpr sexpr ^";\n"

let string_of_sfdecl fdecl =
  let sndOfTriple = fun (_, y, _, _) -> y in
  string_of_typ fdecl.styp ^ " " ^
  fdecl.sfname ^ "(" ^ String.concat ", " (List.map sndOfTriple fdecl.sargs) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_svdecl fdecl.slocals) ^
  String.concat "" (List.map string_of_sstmt fdecl.sbody) ^
  "}\n"

let string_of_sprogram (vars, funcs) =
  String.concat "" (List.map string_of_svdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_sfdecl funcs)
