%{
open Ast
%}

%token PLUS MINUS TIMES DIVIDE POWER ASSIGN ABS TRANSPOSE DOTTIMES DOTDIVIDE DOTPOWER
%token GT LT LSEQ GTEQ EQ NEQ AND OR NOT
%token LSQ RSQ LPA RPA LBR RBR SEMI COMMA COLON
%token INT DOUBLE BOOLEAN STRING MATRIX VOID
%token IF ELSE FOR WHILE CONTINUE BREAK FUNC RETURN
%token PI TRUE FALSE
%token EOF

%token<int> INTLIT
%token<float> DOUBLELIT
%token<string> STRLIT ID

%nonassoc NOELSE
%nonassoc ELSE
%nonassoc COLON
%right ASSIGN
%left OR
%left AND
%left EQ NEQ
%left LT GT LSEQ GTEQ
%left PLUS MINUS
%left TIMES DIVIDE DOTTIMES DOTDIVIDE
%right NOT NEG
%left TRANSPOSE POWER DOTPOWER

%start program
%type <Ast.program> program

%%

program:
  decls EOF { $1 }

decls:
   /* nothing */ { [], [] }
 | decls vdecl { ($2 :: fst $1), snd $1 }
 | decls fdecl { fst $1, ($2 :: snd $1) }

fdecl:
   FUNC data_type ID LPA formals_opt RPA LBR vdecl_list stmt_list RBR
     { 
       { data_type = $2;
        function_name = $3;
        arguments = $5;
        local_vars = List.rev $8;
        body = List.rev $9 }
     }

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    data_type ID                   { [($1,$2, (-1,-1), Noexpr)] }
  | formal_list COMMA data_type ID { ($3,$4, (-1,-1), Noexpr) :: $1 }

data_type:
    INT { Int }
  | BOOLEAN { Bool }
  | VOID { Void }
  | MATRIX { Matrix }
  | STRING { String }
  | DOUBLE { Double }

vdecl_list:
    /* nothing */    { [] }
  | vdecl_list vdecl { $2 :: $1 }

/* In this way, data_type will be uniform (benefit the whole pipeline) and resolve shift/reduce conflict */
vdecl:
    data_type ID SEMI { check_size_normal_return_bind $1 $2 Noexpr }
  | data_type ID matrix_size SEMI { check_size_matrix_return_bind $1 $2 $3 Noexpr }
  | data_type ID ASSIGN expr SEMI { check_size_normal_return_bind $1 $2 $4 }
  | data_type ID matrix_size ASSIGN expr SEMI { check_size_matrix_return_bind $1 $2 $3 $5 }

matrix_size:
    LT INTLIT GT { (1, $2) }
  | LT INTLIT COMMA INTLIT GT { ($2, $4) }

stmt_list:
    /* nothing */  { [] }
  | stmt_list statement { $2 :: $1 }

statement:
    expr SEMI { Expr $1 }
  | BREAK SEMI { Break Noexpr }
  | CONTINUE SEMI { Continue Noexpr }
  | RETURN SEMI { Return Noexpr }
  | RETURN expr SEMI { Return $2 }
  | LBR stmt_list RBR { Block(List.rev $2) }
  | IF LPA expr RPA statement %prec NOELSE { If($3, $5, Block([])) }
  | IF LPA expr RPA statement ELSE statement    { If($3, $5, $7) }
  | FOR LPA expr_opt SEMI expr SEMI expr_opt RPA statement
     { For($3, $5, $7, $9) }
  | FOR LPA INT ID ASSIGN expr RPA statement
     { ForRange($4, $6, $8) }
  | WHILE LPA expr RPA statement { While($3, $5) }

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

expr:
    INTLIT           { IntLit($1) }
  | DOUBLELIT        { DoubleLit($1) }
  | STRLIT           { StrLit($1) }
  | TRUE             { BoolLit(true) }
  | FALSE            { BoolLit(false) }
  | ID               { Id($1) }
  | PI               { DoubleLit(3.1415926535) }
  | expr PLUS   expr { Binop($1, Add,   $3) }
  | expr MINUS  expr { Binop($1, Sub,   $3) }
  | expr TIMES  expr { Binop($1, Mult,  $3) }
  | expr DIVIDE expr { Binop($1, Div,   $3) }
  | expr POWER  expr { Binop($1, Pow,   $3) }
  | expr DOTTIMES  expr { Binop($1, Dotmul,  $3) }
  | expr DOTDIVIDE expr { Binop($1, Dotdiv,  $3) }
  | expr DOTPOWER  expr { Binop($1, Dotpow,  $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq,   $3) }
  | expr LT     expr { Binop($1, Less,  $3) }
  | expr LSEQ   expr { Binop($1, Leq,   $3) }
  | expr GT     expr { Binop($1, Greater, $3) }
  | expr GTEQ   expr { Binop($1, Geq,   $3) }
  | expr AND    expr { Binop($1, And,   $3) }
  | expr OR     expr { Binop($1, Or,    $3) }
  | expr COLON  expr { Range($1, $3) }
  | LSQ ID COMMA ID RSQ           { MatrixOp($2, Comma, $4) }
  | LSQ ID SEMI ID RSQ            { MatrixOp($2, Semi,  $4) }
  | LSQ matrix_lit RSQ            { MatrixLit($2) }
  | ID LSQ expr RSQ               { Matrix1DElement($1, $3) }
  | ID LSQ expr COMMA expr RSQ    { Matrix2DElement($1, $3, $5) }
  | MINUS expr %prec NEG { Unop(Neg, $2) }
  | NOT expr         { Unop(Not, $2) }
  | ABS expr ABS     { Unop(Abs, $2) }
  | expr TRANSPOSE   { Unop(Transpose, $1) }
  | ID ASSIGN expr   { Assign($1, $3) }
  | ID LSQ expr COMMA expr RSQ ASSIGN expr   { Matrix2DModify($1, ($3,$5), $8) }
  | ID LSQ expr RSQ ASSIGN expr              { Matrix1DModify($1, $3, $6) }
  | ID LPA actuals_opt RPA { Call($1, $3) }
  | LPA expr RPA { $2 }

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
    expr                    { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }

matrix_lit:
    matrix_lit_1D                   { [|$1|] }
  | matrix_lit_2D                   { $1 }

matrix_primitive:
    DOUBLELIT                       { $1 }
  | INTLIT                          { float_of_int $1 }
  | MINUS DOUBLELIT                 { -. $2 }
  | MINUS INTLIT                    { float_of_int (-$2) }

matrix_lit_1D:
    matrix_primitive { [|$1|] }
  | matrix_lit_1D COMMA matrix_primitive { Array.append $1 [|$3|] }

matrix_lit_2D:
  | matrix_lit_1D SEMI matrix_lit_1D { [|$1; $3|]}
  | matrix_lit_2D SEMI matrix_lit_1D {Array.append $1 [|$3|]}
