(* Ocamllex scanner for Matrx *)
{
    module Lex = Lexing
    module Buf = Buffer

let sprintf  = Printf.sprintf

let position lexbuf =
    let p = lexbuf.Lex.lex_curr_p in
        sprintf "%s:%d:%d"
        p.Lex.pos_fname p.Lex.pos_lnum (p.Lex.pos_cnum - p.Lex.pos_bol)

let set_filename (fname:string) (lexbuf:Lex.lexbuf)  =
    ( lexbuf.Lex.lex_curr_p <-
        { lexbuf.Lex.lex_curr_p with Lex.pos_fname = fname }
    ; lexbuf
    )

open Parser
}

let digit = ['0' - '9']
let digits = digit+

rule token = parse
  [' ' '\t' '\r' '\n'] { token lexbuf } (* Whitespace *)
| "/*"     { comment lexbuf }           (* Comments *)
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| '['      { LBRACK }
| ']'      { RBRACK }
| ';'      { SEMI }
| ','      { COMMA }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '='      { ASSIGN }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&&"     { AND }
| "||"     { OR }
| "!"      { NOT }
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "while"  { WHILE }
| "return" { RETURN }
| "int"    { INT }
| "bool"   { BOOL }
| "float"  { FLOAT }
| "string" { STRING }
| "void"   { VOID }
| "matrix" { MATRIX }
| '"'      { STRINGLIT ( string (Buf.create 100) lexbuf ) }
| '\''     { CHARLIT ( char (Buf.create 100) lexbuf) }
| "true"   { BLIT(true)  }
| "false"  { BLIT(false) }
| digits as lxm { LITERAL(int_of_string lxm) }
| digits '.'  digit* ( ['e' 'E'] ['+' '-']? digits )? as lxm { FLOATLIT(lxm) }
| ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']*     as lxm { ID(lxm) }
| eof { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and string buf = parse
| [^'"' '\n' '\\']+ as content { Buf.add_string buf content ; string buf lexbuf }
| '\n'      { Buf.add_string buf "\n"; Lex.new_line lexbuf; string buf lexbuf }
| '\\' '"'  { Buf.add_char buf '"'; string buf lexbuf }
| '\\'      { Buf.add_char buf '\\'; string buf lexbuf }
| '"'       { Buf.contents buf } (* return *)

and char buf = parse
| [^'\\'] as content '\'' { content }
| '\\' '\\' '\'' { '\\' }
| '\\' '?' '\'' { '?' }
| '\\' '\'' '\'' { '\'' }
| '\\' '\"' '\'' { '\"' }
| '\\' 'a' '\'' { Char.chr 7 }
| '\\' 'b' '\'' { Char.chr 8 }
| '\\' 'e' '\'' { Char.chr 27 }
| '\\' 'f' '\'' { Char.chr 12 }
| '\\' 'n' '\'' { Char.chr 10 }
| '\\' 'r' '\'' { Char.chr 13 }
| '\\' 't' '\'' { Char.chr 9 }
| '\\' 'v' '\'' { Char.chr 11 }


and comment = parse
  "*/" { token lexbuf }
| _    { comment lexbuf }
