(*
    Semantically-checked Abstract Syntax Tree for Casper
    with functions for printing it
    Based on MicroC
    File: sast.ml
    Michael Makris, mm3443
    PLT Fall 2018
*)

open Ast

type sCasperExpression = casperType * sx
and sx =
      SEpsilon
    | SIntLIT of int
    | SFltLIT of string
    | SStrLIT of string
    | SChrLIT of char
    | SBoolLIT of bool
    | SVoidLIT
    | SNullLIT
    | SIdentifier of string
    | SBinOP of sCasperExpression * casperBinaryOperator * sCasperExpression
    | SUnrOP of casperUnaryOperator * sCasperExpression
    | SAsgnOP of string * casperAssignment * sCasperExpression
    | SFunctionCall of string * sCasperExpression list

type sCasperStatement =
      SStatementBlock of sCasperStatement list
    | SExpression of sCasperExpression
    | SIfCondition of sCasperExpression * sCasperStatement * sCasperStatement
    | SForLoop of sCasperExpression * sCasperExpression * sCasperExpression * sCasperStatement
    | SWhileLoop of sCasperExpression * sCasperStatement
    | SDoUntilLoop of sCasperStatement * sCasperExpression
    | SDoWhileLoop of sCasperStatement * sCasperExpression
    | SBreak
    | SContinue
    | SReturn of sCasperExpression

type sCasperFunction = {
    sFunctionType       : casperType;
    sFunctionName       : string;
    sFunctionFormals    : casperVariable list;
    sFunctionLocals     : casperVariable list;
    sFunctionStatements : sCasperStatement list;
  }

type sCasperProgram = casperVariable list * sCasperFunction list




(* Pretty-printing functions *)

let rec string_of_sCasperExpression (t, e) =
  "(" ^ string_of_casperType t ^ " : " ^ (match e with
      SEpsilon -> ""
    | SIntLIT(i) -> string_of_int i
    | SFltLIT(s) -> s
    | SStrLIT(s) -> s
    | SChrLIT(_) -> "" (* TODO fix Core module to print char c*)
    | SBoolLIT(true) -> "true"
    | SBoolLIT(false) -> "false"
    | SVoidLIT -> ""
    | SNullLIT -> ""
    | SIdentifier(s) -> s
    | SBinOP(e1, o, e2) -> string_of_sCasperExpression e1 ^ " " ^ string_of_casperBinaryOperator o ^ " " ^ string_of_sCasperExpression e2
    | SUnrOP(o, e) -> string_of_casperUnaryOperator o ^ string_of_sCasperExpression e
    | SAsgnOP(s, o, e) -> s ^ " " ^ string_of_casperAssignment o ^ " " ^ string_of_sCasperExpression e
    | SFunctionCall(f, el) -> f ^ "(" ^ String.concat ", " (List.map string_of_sCasperExpression el) ^ ")"
) ^ ")"

let rec string_of_sCasperStatement = function
      SStatementBlock (s) -> "{\n" ^ String.concat "" (List.map string_of_sCasperStatement s) ^ "}\n"
    | SExpression(e) -> string_of_sCasperExpression e ^ ";\n";
    | SIfCondition(e, s, SStatementBlock([])) -> "if (" ^ string_of_sCasperExpression e ^ ") " ^ string_of_sCasperStatement s
    | SIfCondition(e, s1, s2) ->  "if (" ^ string_of_sCasperExpression e ^ ")\n" ^
        string_of_sCasperStatement s1 ^ "else \n" ^ string_of_sCasperStatement s2
    | SForLoop(e1, e2, e3, s) -> "for (" ^ string_of_sCasperExpression e1 ^ "; " ^ string_of_sCasperExpression e2 ^ "; " 
        ^ string_of_sCasperExpression e3 ^ ")" ^ string_of_sCasperStatement s
    | SWhileLoop(e, s) -> "while (" ^ string_of_sCasperExpression e ^ ") " ^ string_of_sCasperStatement s
    | SDoUntilLoop(s, e) -> "do " ^ string_of_sCasperStatement s  ^ "    until (" ^ string_of_sCasperExpression e ^ ");\n"
    | SDoWhileLoop(s, e) -> "do " ^ string_of_sCasperStatement s ^ "    while (" ^ string_of_sCasperExpression e ^ ");\n"
    | SBreak     -> "break;\n";
    | SContinue  -> "continue;\n";
    | SReturn(e) -> "return " ^ string_of_sCasperExpression e ^ ";\n"

let string_of_sCasperFunction casperFunction =
    string_of_casperType casperFunction.sFunctionType ^ " " ^ casperFunction.sFunctionName ^
    "(" ^ String.concat ", " (List.map string_of_casperVariable casperFunction.sFunctionFormals) ^ ") {\n" ^
    String.concat "" (List.map string_of_casperVariable casperFunction.sFunctionLocals) ^
    String.concat "" (List.map string_of_sCasperStatement casperFunction.sFunctionStatements) ^ "}\n"

let string_of_sCasperProgram (programGlobals, programFunctions) =
    String.concat ";\n" (List.map string_of_casperVariable programGlobals) ^ "\n" ^
    String.concat "\n" (List.map string_of_sCasperFunction programFunctions)

(* END Pretty-printing functions*)
