/*  Ocamlyacc parser for Casper
    Based on MicroC
    File: parser.mly
    Michael Makris, mm3443
    PLT Fall 2018
*/

%{ open Ast %}

/* blocks, delimiters, terminators */
%token LPAREN RPAREN LBRACKET RBRACKET LBRACE RBRACE SEMICOLON COMMA EOF
/* string operators */
%token CONCAT CHARAT
/* arithmetic operators */
%token PLUS MINUS TIMES DIVIDE MODULUS EXPONENT
/* relational operators */
%token GT GTE LT LTE EQ NEQ
/* inc/decrement operators */
%token INC DEC
/* assignment operators */
%token ASSIGN CON_ASSIGN ADD_ASSIGN SUB_ASSIGN
/* cast operators */
%token ITOF
/* logical operators */
%token AND OR NOT
/* conditional keywords */
%token IF ELSE
/* loop keywords */
%token FOR WHILE DO UNTIL BREAK CONTINUE
/* function keywords */
%token RETURN
/* type keywords */
%token INT FLOAT STRING CHAR BOOL VOID
/* null keyword */
%token NULL

/* literals */
%token <int> INTLITERAL
%token <string> FLTLITERAL
%token <string> STRLITERAL
%token <char> CHRLITERAL
%token <bool> BOOLLITERAL
%token VOIDLITERAL

/* identifier for variable and function names */
%token <string> IDENTIFIER

/* precedence */
%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN CON_ASSIGN ADD_ASSIGN SUB_ASSIGN
%left OR
%left AND
%left EQ NEQ
%left GT GTE LT LTE
%left CONCAT
%left PLUS MINUS
%left TIMES DIVIDE MODULUS
%right EXPONENT
%right CHARAT
%right ITOF
%right INC DEC
%right NOT NEG

%start casper
%type <Ast.casperProgram> casper

%%

/****** program declaration ******/
casper: declarations EOF { $1 }

declarations:
    /* epsilon */               { ([], []) }
  | declarations casperVariable { (($2 :: fst $1), snd $1) }
  | declarations casperFunction { (fst $1, ($2 :: snd $1)) }

/****** variable declaration ******/
/* variable declaration: <type> <variable_name>; */
casperVariable: casperType IDENTIFIER SEMICOLON {$1, $2}

/* optional array size [n]
optArraySize: /* epsilon *                  { -1 }
              | LBRACKET INTLITERAL RBRACKET { $2 } */

/****** type declaration ******/
casperType:
    INT     { Int }
  | FLOAT   { Float }
  | STRING  { String }
  | CHAR    { Char }
  | BOOL    { Bool }
  | VOID    { Void }
/*  | casperType LBRACKET RBRACKET  { CArray $1 }  */

/****** function declaration ******/
/* <type> <function_name> (<arg1>, <arg2>, ...) { <statements> } */
casperFunction: casperType IDENTIFIER formalsBlock 
                LBRACE localsBlock statementsList RBRACE {{
                    functionType       = $1;
                    functionName       = $2;
                    functionFormals    = List.rev $3;
                    functionLocals     = List.rev $5;
                    functionStatements = List.rev $6;
                }}

/* formals Block*/
formalsBlock: LPAREN optFormals RPAREN          {$2}

/* optional formals list */
optFormals: /* epsilon */  { [] }
            | formalsList { $1 }

/* formals list with elements */
formalsList: casperType IDENTIFIER                      { [($1, $2)] }
             | formalsList COMMA casperType IDENTIFIER  { ($3,$4) :: $1 }

/* optional locals list */
localsBlock:  /* epsilon */              { [] }
            | localsBlock casperVariable { $2::$1 }

/* optional statement list */
statementsList: /* epsilon */                   { [] }
                | statementsList casperStatement { $2::$1 }

casperStatement:
  /* { <statement>; ... } */
    LBRACE statementsList RBRACE            { StatementBlock(List.rev $2) }
  /* <expression>; */
  | casperExpression SEMICOLON              { Expression $1 }
  /* if (<expression>) { <statements> } */
  | IF LPAREN casperExpression RPAREN casperStatement %prec NOELSE 
                                            { IfCondition($3, $5, StatementBlock([])) }
  /* if (<expression>) { <statements> } else { <statements> } */
  | IF LPAREN casperExpression RPAREN casperStatement
    ELSE casperStatement                    { IfCondition($3, $5, $7) }
  /* for (<expression>; <expression>; <expression>) { <statements> } */
  | FOR LPAREN optExpression SEMICOLON casperExpression SEMICOLON optExpression RPAREN
    casperStatement                         { ForLoop($3, $5, $7, $9) }
  /* while (<expression>) { <statements> } */
  | WHILE LPAREN casperExpression RPAREN casperStatement
                                            { WhileLoop($3, $5) }
  /* do { <statements> } until (<expression>);*/
  | DO casperStatement UNTIL LPAREN casperExpression RPAREN SEMICOLON 
                                            { DoUntilLoop($2, $5) }
  /* do { <statements> } while (<expression>);*/
  | DO casperStatement WHILE LPAREN casperExpression RPAREN SEMICOLON 
                                            { DoWhileLoop($2, $5) }  
  /* break; */
  | BREAK SEMICOLON                         { Break }
  /* continue; */
  | CONTINUE SEMICOLON                      { Continue }
  /* return <expression>; */
  | RETURN optExpression SEMICOLON          { Return $2 }

/****** expression declaration ******/
/* optional expression */
optExpression: { Epsilon }
               | casperExpression { $1 }

casperExpression:
    INTLITERAL                                   { IntLIT  ($1) }
  | FLTLITERAL                                   { FltLIT  ($1) }
  | STRLITERAL                                   { StrLIT  ($1) }
  | CHRLITERAL                                   { ChrLIT  ($1) }
  | BOOLLITERAL                                  { BoolLIT ($1) }
  | VOIDLITERAL                                  { VoidLIT }
  | NULL                                         { NullLIT }
  | IDENTIFIER                                   { Identifier ($1) }
  | casperExpression  PLUS      casperExpression { BinOP($1, Add, $3) }
  | casperExpression  MINUS     casperExpression { BinOP($1, Sub, $3) }
  | casperExpression  TIMES     casperExpression { BinOP($1, Mul, $3) }
  | casperExpression  DIVIDE    casperExpression { BinOP($1, Div, $3) }
  | casperExpression  MODULUS   casperExpression { BinOP($1, Mod, $3) }
  | casperExpression  EXPONENT  casperExpression { BinOP($1, Exp, $3) }  
  | casperExpression  CONCAT    casperExpression { BinOP($1, Con, $3) }
  | casperExpression  CHARAT    casperExpression { BinOP($1, Cat, $3) }
  | casperExpression  GT        casperExpression { BinOP($1, Grt, $3) }
  | casperExpression  GTE       casperExpression { BinOP($1, Gre, $3) }
  | casperExpression  LT        casperExpression { BinOP($1, Lst, $3) }
  | casperExpression  LTE       casperExpression { BinOP($1, Lse, $3) }
  | casperExpression  EQ        casperExpression { BinOP($1, Eql, $3) }
  | casperExpression  NEQ       casperExpression { BinOP($1, Neq, $3) }
  | casperExpression  AND       casperExpression { BinOP($1, And, $3) }
  | casperExpression  OR        casperExpression { BinOP($1, Or , $3) }
  | INC casperExpression                         { BinOP($2, Add, IntLIT(1)) }
  | DEC casperExpression                         { BinOP($2, Sub, IntLIT(1)) }
  | NOT casperExpression                         { UnrOP(Not, $2) }
  | MINUS casperExpression %prec NEG             { UnrOP(Neg, $2) }
  | ITOF casperExpression                        { UnrOP(ItoF, $2) }  
  | IDENTIFIER  ASSIGN      casperExpression     { AsgnOP($1, Asgn, $3) }
  | IDENTIFIER  CON_ASSIGN  casperExpression     { AsgnOP($1, ConAsgn, $3) }
  | IDENTIFIER  ADD_ASSIGN  casperExpression     { AsgnOP($1, AddAsgn, $3) }
  | IDENTIFIER  SUB_ASSIGN  casperExpression     { AsgnOP($1, SubAsgn, $3) }
  | IDENTIFIER LPAREN elements RPAREN            { FunctionCall($1, $3) }
/*  | LBRACKET elements RBRACKET                   { ArrayLIT ($2) }  */
  | LPAREN casperExpression RPAREN               { $2 }

/* optional element list */
elements: /* epsilon */   { [] }
          | elementsList  { List.rev $1 }

/* argument list with elements */
elementsList: casperExpression                      { [$1] }
              | elementsList COMMA casperExpression { $3::$1 }
