(*
    Code generation for Casper
    Translation takes a semantically checked AST and produces LLVM IR
    Based on MicroC
    File: codegen.ml
    Michael Makris, mm3443
    PLT Fall 2018
*)
(*
LLVM tutorial: Make sure to read the OCaml version of the tutorial

http://llvm.org/docs/tutorial/index.html

Detailed documentation on the OCaml LLVM library:

http://llvm.moe/
http://llvm.moe/ocaml/

*)

module C = Char
module L = Llvm
module A = Ast
open Sast

module StringMap = Map.Make(String)

(* translate : Sast.program -> Llvm.module *)
let translate (globals, functions) =
  let context    = L.global_context () in

  (* Create the LLVM compilation module into which we will generate code *)
  let the_module = L.create_module context "Casper" in

  (* Get types from the context *)
  let i32_t      = L.i32_type    context
  and i8_t       = L.i8_type     context
  and float_t    = L.double_type context
  and str_t      = L.pointer_type (L.i8_type context)
  and i1_t       = L.i1_type     context
  and void_t     = L.void_type   context
  and cnull      = L.const_null (L.i32_type context)
  in
(*
  and ptr_t = str_t
  and size_t = L.type_of (L.size_of i8_t)
  and void_p = L.pointer_type size_t
  and null_str = L.const_null str_t
*)

  (* Return the LLVM type for a Casper type *)
  let ltype_of_casperType = function
      A.Int    -> i32_t
    | A.Float  -> float_t
    | A.String -> str_t
    | A.Char   -> i8_t
    | A.Bool   -> i1_t
    | A.Void   -> void_t
  in

  (* pointer to while loop merge_bb, pred_bb for break and continue statement branch *)
  (* TODO: expand to list of pointers for nested loops *)
  let break_block = ref (cnull) in
  let continue_block = ref (cnull) in

  (* Create a map of global variables after creating each *)
  let global_vars : L.llvalue StringMap.t =
    let global_var m (t, n) =
      if (ltype_of_casperType t = str_t)
      then (
        let init = L.const_null str_t in
        StringMap.add n (L.define_global n init the_module) m
      )
      else (
        let init = match t with
          A.Float -> L.const_float (ltype_of_casperType t) 0.0
          | _ -> L.const_int (ltype_of_casperType t) 0
          in StringMap.add n (L.define_global n init the_module) m
      ) in
    List.fold_left global_var StringMap.empty globals in

  let printf_t : L.lltype = L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
  let printf_func : L.llvalue = L.declare_function "printf" printf_t the_module in
  let printb_t : L.lltype = L.function_type i32_t [| i1_t; i32_t |] in
  let printb_func : L.llvalue = L.declare_function "printb" printb_t the_module in
  let printbig_t : L.lltype = L.function_type i32_t [| i32_t |] in
  let printbig_func : L.llvalue = L.declare_function "printbig" printbig_t the_module in

  (* expose string.h strcat function for the _ operator *)
  let concats_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let concats_func : L.llvalue = L.declare_function "concats" concats_t the_module in  
  let concati_t : L.lltype = L.function_type str_t [| str_t; i32_t |] in
  let concati_func : L.llvalue = L.declare_function "concati" concati_t the_module in    
  let concatf_t : L.lltype = L.function_type str_t [| str_t; float_t |] in
  let concatf_func : L.llvalue = L.declare_function "concatf" concatf_t the_module in  
  let concatb_t : L.lltype = L.function_type str_t [| str_t; i1_t |] in
  let concatb_func : L.llvalue = L.declare_function "concatb" concatb_t the_module in   
  
  (* function in stdlib.c for ? operator *)
  let charat_t : L.lltype = L.function_type str_t [| str_t; i32_t |] in
  let charat_func : L.llvalue = L.declare_function "charat" charat_t the_module in
  (* function in stdlib.c for string comparison operators *)
  let scomp_t : L.lltype = L.function_type i1_t [| str_t; str_t; i32_t |] in
  let scomp_func : L.llvalue = L.declare_function "scomp" scomp_t the_module in
  (* expose ctype.h functions as build-in functions *)
  let isalnum_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isalnum_func : L.llvalue = L.declare_function "isalnum" isalnum_t the_module in
  let isalpha_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isalpha_func : L.llvalue = L.declare_function "isalpha" isalpha_t the_module in
  let iscntrl_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let iscntrl_func : L.llvalue = L.declare_function "iscntrl" iscntrl_t the_module in
  let isdigit_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isdigit_func : L.llvalue = L.declare_function "isdigit" isdigit_t the_module in
  let isgraph_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isgraph_func : L.llvalue = L.declare_function "isgraph" isgraph_t the_module in
  let islower_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let islower_func : L.llvalue = L.declare_function "islower" islower_t the_module in  
  let isprint_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isprint_func : L.llvalue = L.declare_function "isprint" isprint_t the_module in    
  let ispunct_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let ispunct_func : L.llvalue = L.declare_function "ispunct" ispunct_t the_module in  
  let isspace_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isspace_func : L.llvalue = L.declare_function "isspace" isspace_t the_module in    
  let isupper_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isupper_func : L.llvalue = L.declare_function "isupper" isupper_t the_module in
  let isxdigit_t : L.lltype = L.function_type i32_t [| i8_t |] in
  let isxdigit_func : L.llvalue = L.declare_function "isxdigit" isxdigit_t the_module in  
  (* expose string.h functions as build-in functions *)
  let strlen_t : L.lltype = L.function_type i32_t [| str_t |] in
  let strlen_func : L.llvalue = L.declare_function "strlen" strlen_t the_module in
  let strcpy_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strcpy_func : L.llvalue = L.declare_function "strcpy" strcpy_t the_module in
  let strncpy_t : L.lltype = L.function_type str_t [| str_t; str_t; i32_t |] in
  let strncpy_func : L.llvalue = L.declare_function "strncpy" strncpy_t the_module in
  let strcat_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strcat_func : L.llvalue = L.declare_function "strcat" strcat_t the_module in
  let strncat_t : L.lltype = L.function_type str_t [| str_t; str_t; i32_t |] in
  let strncat_func : L.llvalue = L.declare_function "strcat" strncat_t the_module in
  let strcmp_t : L.lltype = L.function_type i32_t [| str_t; str_t |] in
  let strcmp_func : L.llvalue = L.declare_function "strcmp" strcmp_t the_module in
  let strncmp_t : L.lltype = L.function_type i32_t [| str_t; str_t; i32_t |] in
  let strncmp_func : L.llvalue = L.declare_function "strcmp" strncmp_t the_module in
  let strchr_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strchr_func : L.llvalue = L.declare_function "strchr" strchr_t the_module in
  let strrchr_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strrchr_func : L.llvalue = L.declare_function "strrchr" strrchr_t the_module in
  let strspn_t : L.lltype = L.function_type i32_t [| str_t; str_t |] in
  let strspn_func : L.llvalue = L.declare_function "strspn" strspn_t the_module in
  let strcspn_t : L.lltype = L.function_type i32_t [| str_t; str_t |] in
  let strcspn_func : L.llvalue = L.declare_function "strcspn" strcspn_t the_module in
  let strpbrk_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strpbrk_func : L.llvalue = L.declare_function "strrchr" strpbrk_t the_module in
  let strstr_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strstr_func : L.llvalue = L.declare_function "strstr" strstr_t the_module in
  let strerror_t : L.lltype = L.function_type str_t [| i32_t |] in
  let strerror_func : L.llvalue = L.declare_function "strerror" strerror_t the_module in
  let strtok_t : L.lltype = L.function_type str_t [| str_t; str_t |] in
  let strtok_func : L.llvalue = L.declare_function "strtok" strtok_t the_module in

  (* expose math.h pow function for the ^ operator *)
  let pow_t : L.lltype = L.function_type float_t [| float_t; float_t |] in
  let pow_func : L.llvalue = L.declare_function "pow" pow_t the_module in
  (* expose math.h functions as build-in functions *)
  let sin_t : L.lltype = L.function_type float_t [| float_t |] in
  let sin_func : L.llvalue = L.declare_function "sin" sin_t the_module in
  let cos_t : L.lltype = L.function_type float_t [| float_t |] in
  let cos_func : L.llvalue = L.declare_function "cos" cos_t the_module in
  let tan_t : L.lltype = L.function_type float_t [| float_t |] in
  let tan_func : L.llvalue = L.declare_function "tan" tan_t the_module in
  let asin_t : L.lltype = L.function_type float_t [| float_t |] in
  let asin_func : L.llvalue = L.declare_function "asin" asin_t the_module in
  let acos_t : L.lltype = L.function_type float_t [| float_t |] in
  let acos_func : L.llvalue = L.declare_function "acos" acos_t the_module in
  let atan_t : L.lltype = L.function_type float_t [| float_t |] in
  let atan_func : L.llvalue = L.declare_function "atan" atan_t the_module in
  let sinh_t : L.lltype = L.function_type float_t [| float_t |] in
  let sinh_func : L.llvalue = L.declare_function "sinh" sinh_t the_module in
  let cosh_t : L.lltype = L.function_type float_t [| float_t |] in
  let cosh_func : L.llvalue = L.declare_function "cosh" cosh_t the_module in
  let tanh_t : L.lltype = L.function_type float_t [| float_t |] in
  let tanh_func : L.llvalue = L.declare_function "tanh" tanh_t the_module in
  let exp_t : L.lltype = L.function_type float_t [| float_t |] in
  let exp_func : L.llvalue = L.declare_function "exp" exp_t the_module in
  let log_t : L.lltype = L.function_type float_t [| float_t |] in
  let log_func : L.llvalue = L.declare_function "log" log_t the_module in
  let log10_t : L.lltype = L.function_type float_t [| float_t |] in
  let log10_func : L.llvalue = L.declare_function "log10" log10_t the_module in
  let sqrt_t : L.lltype = L.function_type float_t [| float_t |] in
  let sqrt_func : L.llvalue = L.declare_function "sqrt" sqrt_t the_module in
  let floor_t : L.lltype = L.function_type float_t [| float_t |] in
  let floor_func : L.llvalue = L.declare_function "floor" floor_t the_module in
  let ceil_t : L.lltype = L.function_type float_t [| float_t |] in
  let ceil_func : L.llvalue = L.declare_function "ceil" ceil_t the_module in
  let fabs_t : L.lltype = L.function_type float_t [| float_t |] in
  let fabs_func : L.llvalue = L.declare_function "fabs" fabs_t the_module in
  let srand_t : L.lltype = L.var_arg_function_type i32_t [| i32_t |] in
  let srand_func : L.llvalue = L.declare_function "srand" srand_t the_module in
  let rand_t : L.lltype = L.function_type i32_t [|  |] in
  let rand_func : L.llvalue = L.declare_function "rand" rand_t the_module in
  let fmod_t : L.lltype = L.function_type float_t [| float_t; float_t |] in
  let fmod_func : L.llvalue = L.declare_function "fmod" fmod_t the_module in

  (* Define each function (arguments and return type) so we can
     call it even before we've created its body *)
  let function_decls : (L.llvalue * sCasperFunction) StringMap.t =
    let function_decl m fdecl =
      let name = fdecl.sFunctionName
      and formal_types =
    Array.of_list (List.map (fun (t,_) -> ltype_of_casperType t) fdecl.sFunctionFormals)
      in let ftype = L.function_type (ltype_of_casperType fdecl.sFunctionType) formal_types in
      StringMap.add name (L.define_function name ftype the_module, fdecl) m in
    List.fold_left function_decl StringMap.empty functions in

  (* Fill in the body of the given function *)
  let build_function_body fdecl =
    let (the_function, _) = StringMap.find fdecl.sFunctionName function_decls in
    let builder = L.builder_at_end context (L.entry_block the_function) in

    let int_format_str_nl = L.build_global_stringptr "%d\n" "fmt" builder
    and int_format_str = L.build_global_stringptr "%d" "fmt" builder
    and float_format_str_nl = L.build_global_stringptr "%.10f\n" "fmt" builder
    and float_format_str = L.build_global_stringptr "%g" "fmt" builder
    and str_format_str_nl = L.build_global_stringptr "%s\n" "fmt" builder
    and str_format_str = L.build_global_stringptr "%s" "fmt" builder
    and chr_format_str_nl = L.build_global_stringptr "%c\n" "fmt" builder
    and chr_format_str = L.build_global_stringptr "%c" "fmt" builder    
    and bool_format_str_ln = L.build_global_stringptr "%d\n" "fmt" builder in

    (* Construct the function's "locals": formal arguments and locally
       declared variables.  Allocate each on the stack, initialize their
       value, if appropriate, and remember their values in the "locals" map *)
    let local_vars =
      let add_formal m (t, n) p = L.set_value_name n p;
    let local = L.build_alloca (ltype_of_casperType t) n builder in
        ignore (L.build_store p local builder);
    StringMap.add n local m

      (* Allocate space for any locally declared variables and add the
       * resulting registers to our map *)
      and add_local m (t, n) =
    let local_var = L.build_alloca (ltype_of_casperType t) n builder
    in StringMap.add n local_var m
      in

      let formals = List.fold_left2 add_formal StringMap.empty fdecl.sFunctionFormals
          (Array.to_list (L.params the_function)) in
      List.fold_left add_local formals fdecl.sFunctionLocals
    in

    (* Return the value for a variable or formal argument.
       Check local names first, then global names *)
    let lookup n = try StringMap.find n local_vars
                   with Not_found -> StringMap.find n global_vars
    in

    (* Construct code for an expression; return its value *)
    let rec expr builder ((_, e) : sCasperExpression) = match e with
        SEpsilon    -> L.const_int i32_t 0
    | SIntLIT i   -> L.const_int i32_t i
      | SFltLIT f   -> L.const_float_of_string float_t f
      | SStrLIT s   -> L.build_global_stringptr s "str" builder
      | SChrLIT c   -> L.const_int i8_t (C.code c)
      | SBoolLIT b  -> L.const_int i1_t (if b then 1 else 0)
      | SVoidLIT    -> L.const_int i32_t 0
      | SNullLIT    -> L.const_int i32_t 0
      | SIdentifier s -> L.build_load (lookup s) s builder
      | SBinOP (((A.String,_ ) as e1), op, ((t, _) as e2)) ->
        let e1' = expr builder e1
        and e2' = expr builder e2
        in
        (match op with
            A.Con when t = A.Int   -> L.build_call concati_func [| e1' ; e2' |] "coni" builder
          | A.Con when t = A.Float -> L.build_call concatf_func [| e1' ; e2' |] "conf" builder
          | A.Con when t = A.Bool  -> L.build_call concatb_func [| e1' ; e2' |] "conb" builder
          | A.Con                  -> L.build_call concats_func [| e1' ; e2' |] "cons" builder
          | A.Cat -> L.build_call charat_func [| e1' ; e2' |] "cat" builder
          | A.Eql -> L.build_call scomp_func [| e1' ; e2' ; (expr builder (A.Int, SIntLIT 1)) |] "scomp" builder
          | A.Neq -> L.build_call scomp_func [| e1' ; e2' ; (expr builder (A.Int, SIntLIT 2)) |] "scomp" builder
          | A.Lst -> L.build_call scomp_func [| e1' ; e2' ; (expr builder (A.Int, SIntLIT 3)) |] "scomp" builder
          | A.Lse -> L.build_call scomp_func [| e1' ; e2' ; (expr builder (A.Int, SIntLIT 4)) |] "scomp" builder
          | A.Grt -> L.build_call scomp_func [| e1' ; e2' ; (expr builder (A.Int, SIntLIT 5)) |] "scomp" builder
          | A.Gre -> L.build_call scomp_func [| e1' ; e2' ; (expr builder (A.Int, SIntLIT 6)) |] "scomp" builder
          | A.Add | A.Sub | A.Mul | A.Div | A.Mod | A.Exp | A.And | A.Or ->
            raise (Failure "internal error: semant should have rejected these ops on string")
        )
      | SBinOP ((A.Float,_ ) as e1, op, e2) ->
        let e1' = expr builder e1
        and e2' = expr builder e2 in
        (match op with
            A.Add     -> L.build_fadd e1' e2' "fadd"
          | A.Sub     -> L.build_fsub e1' e2' "fsub"
          | A.Mul     -> L.build_fmul e1' e2' "fmul"
          | A.Div     -> L.build_fdiv  e1' e2' "fdiv"
          | A.Mod     -> L.build_call fmod_func [| e1' ; e2' |] "fmod"
          | A.Exp     -> L.build_call pow_func [| e1' ; e2' |] "fpow"
          | A.Eql     -> L.build_fcmp L.Fcmp.Oeq e1' e2' "feq"
          | A.Neq     -> L.build_fcmp L.Fcmp.One e1' e2' "fneq"
          | A.Lst     -> L.build_fcmp L.Fcmp.Olt e1' e2' "flst"
          | A.Lse     -> L.build_fcmp L.Fcmp.Ole e1' e2' "flse"
          | A.Grt     -> L.build_fcmp L.Fcmp.Ogt e1' e2' "fgrt"
          | A.Gre     -> L.build_fcmp L.Fcmp.Oge e1' e2' "fgre"
          | A.And | A.Or | A.Con | A.Cat ->
            raise (Failure "internal error: semant should have rejected and/or/con/cat on float")
        ) builder
      | SBinOP (e1, op, e2) ->
        let e1' = expr builder e1
        and e2' = expr builder e2 in
        (match op with
            A.Add     -> L.build_add e1' e2' "tmp"
          | A.Sub     -> L.build_sub e1' e2' "tmp"
          | A.Mul     -> L.build_mul e1' e2' "tmp"
          | A.Div     -> L.build_sdiv e1' e2' "tmp"
          | A.Mod     -> L.build_srem e1' e2' "tmp"
          | A.And     -> L.build_and e1' e2' "tmp"
          | A.Or      -> L.build_or e1' e2' "tmp"
          | A.Eql     -> L.build_icmp L.Icmp.Eq e1' e2' "tmp"
          | A.Neq     -> L.build_icmp L.Icmp.Ne e1' e2' "tmp"
          | A.Lst     -> L.build_icmp L.Icmp.Slt e1' e2' "tmp"
          | A.Lse     -> L.build_icmp L.Icmp.Sle e1' e2' "tmp"
          | A.Grt     -> L.build_icmp L.Icmp.Sgt e1' e2' "tmp"
          | A.Gre     -> L.build_icmp L.Icmp.Sge e1' e2' "tmp"
          | A.Exp | A.Con | A.Cat ->
              raise (Failure "internal error: semant should have rejected pow/con/cat on int or bool")
        ) builder
      | SUnrOP(op, ((t, _) as e)) -> let e' = expr builder e in
          (match op with
              A.Neg when t = A.Float -> L.build_fneg   e'           "tmp" builder
            | A.Neg                  -> L.build_neg    e'           "tmp" builder  
            | A.Not                  -> L.build_not    e'           "tmp" builder
            | A.ItoF when t = A.Float -> L.build_fptosi e' i32_t    "tmp" builder 
            | A.ItoF                  -> L.build_sitofp e' float_t  "tmp" builder
          ) 
      | SAsgnOP (s, op, ((t, _) as e)) -> let e' = expr builder e in
          (match op with
              A.Asgn when t = A.String   -> ignore(L.build_store (L.build_call concats_func [| (expr builder (A.String, SStrLIT ""))   ; e' |] "cons" builder) (lookup s) builder); e'
            | A.ConAsgn when t = A.Int   -> ignore(L.build_store (L.build_call concati_func [| (L.build_load (lookup s) "tmp" builder) ; e' |] "coni" builder) (lookup s) builder); e'
            | A.ConAsgn when t = A.Float -> ignore(L.build_store (L.build_call concatf_func [| (L.build_load (lookup s) "tmp" builder) ; e' |] "conf" builder) (lookup s) builder); e'
            | A.ConAsgn when t = A.Bool  -> ignore(L.build_store (L.build_call concatb_func [| (L.build_load (lookup s) "tmp" builder) ; e' |] "conb" builder) (lookup s) builder); e'
            | A.ConAsgn                  -> ignore(L.build_store (L.build_call concats_func [| (L.build_load (lookup s) "tmp" builder) ; e' |] "cons" builder) (lookup s) builder); e'
            | A.Asgn                     -> ignore(L.build_store e' (lookup s) builder); e'
            | A.AddAsgn when t = A.Float -> ignore(L.build_store (L.build_fadd (L.build_load (lookup s) "tmp" builder) e' "tmp" builder) (lookup s) builder); e'
            | A.AddAsgn                  -> ignore(L.build_store (L.build_add (L.build_load (lookup s) "tmp" builder) e' "tmp" builder) (lookup s) builder); e'
            | A.SubAsgn when t = A.Float -> ignore(L.build_store (L.build_fsub (L.build_load (lookup s) "tmp" builder) e' "tmp" builder) (lookup s) builder); e'
            | A.SubAsgn                  -> ignore(L.build_store (L.build_sub (L.build_load (lookup s) "tmp" builder) e' "tmp" builder) (lookup s) builder); e'
           )
      | SFunctionCall ("printi",   [e]) -> L.build_call printf_func [| int_format_str      ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printinl", [e]) -> L.build_call printf_func [| int_format_str_nl   ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printf",   [e]) -> L.build_call printf_func [| float_format_str    ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printfnl", [e]) -> L.build_call printf_func [| float_format_str_nl ; (expr builder e) |] "printf" builder
      | SFunctionCall ("prints",   [e]) -> L.build_call printf_func [| str_format_str      ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printsnl", [e]) -> L.build_call printf_func [| str_format_str_nl   ; (expr builder e) |] "printf" builder      
      | SFunctionCall ("printc",   [e]) -> L.build_call printf_func [| chr_format_str      ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printcnl", [e]) -> L.build_call printf_func [| chr_format_str_nl   ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printbasi", [e])-> L.build_call printf_func [| bool_format_str_ln  ; (expr builder e) |] "printf" builder
      | SFunctionCall ("printb",   [e]) -> L.build_call printb_func [| (expr builder e) ; (expr builder (A.Int, SIntLIT 0)) |] "printb" builder
      | SFunctionCall ("printbnl", [e]) -> L.build_call printb_func [| (expr builder e) ; (expr builder (A.Int, SIntLIT 1)) |] "printb" builder
      | SFunctionCall ("printbig", [e]) -> L.build_call printbig_func [| (expr builder e) |] "printbig" builder
      | SFunctionCall ("sin",   [e]) -> L.build_call sin_func   [| (expr builder e) |] "sin"   builder
      | SFunctionCall ("cos",   [e]) -> L.build_call cos_func   [| (expr builder e) |] "cos"   builder
      | SFunctionCall ("tan",   [e]) -> L.build_call tan_func   [| (expr builder e) |] "tan"   builder
      | SFunctionCall ("asin",  [e]) -> L.build_call asin_func  [| (expr builder e) |] "asin"  builder
      | SFunctionCall ("acos",  [e]) -> L.build_call acos_func  [| (expr builder e) |] "acos"  builder
      | SFunctionCall ("atan",  [e]) -> L.build_call atan_func  [| (expr builder e) |] "atan"  builder
      | SFunctionCall ("sinh",  [e]) -> L.build_call sinh_func  [| (expr builder e) |] "sinh"  builder
      | SFunctionCall ("cosh",  [e]) -> L.build_call cosh_func  [| (expr builder e) |] "cosh"  builder
      | SFunctionCall ("tanh",  [e]) -> L.build_call tanh_func  [| (expr builder e) |] "tanh"  builder
      | SFunctionCall ("exp",   [e]) -> L.build_call exp_func   [| (expr builder e) |] "exp"   builder
      | SFunctionCall ("log",   [e]) -> L.build_call log_func   [| (expr builder e) |] "log"   builder
      | SFunctionCall ("log10", [e]) -> L.build_call log10_func [| (expr builder e) |] "log10" builder
      | SFunctionCall ("sqrt",  [e]) -> L.build_call sqrt_func  [| (expr builder e) |] "sqrt"  builder
      | SFunctionCall ("floor", [e]) -> L.build_call floor_func [| (expr builder e) |] "floor" builder
      | SFunctionCall ("ceil",  [e]) -> L.build_call ceil_func  [| (expr builder e) |] "ceil"  builder
      | SFunctionCall ("abs",   [e]) -> L.build_call fabs_func  [| (expr builder e) |] "abs"   builder
      | SFunctionCall ("srand", [e]) -> L.build_call srand_func [| (expr builder e) |] "srand" builder
      | SFunctionCall ("rand",  []) -> L.build_call rand_func   [|  |] "rand"  builder
      | SFunctionCall ("fmod",  [e1; e2])   -> L.build_call fmod_func [| (expr builder e1); (expr builder e2) |] "fmod"  builder
      | SFunctionCall ("strlen", [e1])     -> L.build_call strlen_func  [| (expr builder e1) |]                    "strlen" builder
      | SFunctionCall ("strcpy", [e1; e2]) -> L.build_call strcpy_func  [| (expr builder e1); (expr builder e2) |] "strcpy" builder
      | SFunctionCall ("strncpy",[e1; e2; e3]) -> L.build_call strncpy_func [| (expr builder e1); (expr builder e2); (expr builder e3) |] "strncpy" builder
      | SFunctionCall ("strcat", [e1; e2]) -> L.build_call strcat_func  [| (expr builder e1); (expr builder e2) |] "strcat" builder
      | SFunctionCall ("strncat",[e1; e2; e3]) -> L.build_call strncat_func [| (expr builder e1); (expr builder e2); (expr builder e3) |] "strncat" builder
      | SFunctionCall ("strcmp", [e1; e2]) -> L.build_call strcmp_func  [| (expr builder e1); (expr builder e2) |] "strcmp" builder
      | SFunctionCall ("strncmp",[e1; e2; e3]) -> L.build_call strncmp_func [| (expr builder e1); (expr builder e2); (expr builder e3) |] "strncmp" builder
      | SFunctionCall ("strchr", [e1; e2]) -> L.build_call strchr_func  [| (expr builder e1); (expr builder e2) |] "strchr" builder
      | SFunctionCall ("strrchr",[e1; e2]) -> L.build_call strrchr_func [| (expr builder e1); (expr builder e2) |] "strrchr" builder
      | SFunctionCall ("strspn", [e1; e2]) -> L.build_call strspn_func  [| (expr builder e1); (expr builder e2) |] "strspn" builder
      | SFunctionCall ("strcspn",[e1; e2]) -> L.build_call strcspn_func [| (expr builder e1); (expr builder e2) |] "strcspn" builder
      | SFunctionCall ("strpbrk",[e1; e2]) -> L.build_call strpbrk_func [| (expr builder e1); (expr builder e2) |] "strpbrk" builder
      | SFunctionCall ("strstr", [e1; e2]) -> L.build_call strstr_func  [| (expr builder e1); (expr builder e2) |] "strstr" builder
      | SFunctionCall ("strerror",[e1])    -> L.build_call strerror_func [| (expr builder e1) |]                   "strerror" builder
      | SFunctionCall ("strtok",  [e1; e2])-> L.build_call strtok_func  [| (expr builder e1); (expr builder e2) |] "strtok" builder

      | SFunctionCall ("isalnum",  [e]) -> L.build_call isalnum_func  [| (expr builder e) |] "isalnum"  builder
      | SFunctionCall ("isalpha",  [e]) -> L.build_call isalpha_func  [| (expr builder e) |] "isalpha"  builder
      | SFunctionCall ("iscntrl",  [e]) -> L.build_call iscntrl_func  [| (expr builder e) |] "iscntrl"  builder
      | SFunctionCall ("isdigit",  [e]) -> L.build_call isdigit_func  [| (expr builder e) |] "isdigit"  builder
      | SFunctionCall ("isgraph",  [e]) -> L.build_call isgraph_func  [| (expr builder e) |] "isgraph"  builder
      | SFunctionCall ("islower",  [e]) -> L.build_call islower_func  [| (expr builder e) |] "islower"  builder
      | SFunctionCall ("isprint",  [e]) -> L.build_call isprint_func  [| (expr builder e) |] "isprint"  builder
      | SFunctionCall ("ispunct",  [e]) -> L.build_call ispunct_func  [| (expr builder e) |] "ispunct"  builder
      | SFunctionCall ("isspace",  [e]) -> L.build_call isspace_func  [| (expr builder e) |] "isspace"  builder
      | SFunctionCall ("isupper",  [e]) -> L.build_call isupper_func  [| (expr builder e) |] "isupper"  builder
      | SFunctionCall ("isxdigit", [e]) -> L.build_call isxdigit_func [| (expr builder e) |] "isxdigit" builder      
      
      | SFunctionCall (f, args) -> let (fdef, fdecl) = StringMap.find f function_decls in
    let llargs = List.rev (List.map (expr builder) (List.rev args)) in
    let result = (match fdecl.sFunctionType with
                        A.Void -> ""
                      | _ -> f ^ "_result") in
         L.build_call fdef (Array.of_list llargs) result builder
    in

    (* LLVM insists each basic block end with exactly one "terminator"
       instruction that transfers control.  This function runs "instr builder"
       if the current block does not already have a terminator.  Used,
       e.g., to handle the "fall off the end of the function" case. *)
    let add_terminal builder instr =
      match L.block_terminator (L.insertion_block builder) with
        Some _ -> ()
      | None -> ignore (instr builder) in

    (* Build the code for the given statement; return the builder for
       the statement's successor (i.e., the next instruction will be built
       after the one generated by this call) *)

    let rec stmt builder = function
        SStatementBlock sl -> List.fold_left stmt builder sl
      | SExpression e -> ignore(expr builder e); builder
      | SBreak ->    ignore(L.build_br (L.block_of_value !break_block) builder); builder    (* TODO: expand to list of pointers for nested loops *)
      | SContinue -> ignore(L.build_br (L.block_of_value !continue_block) builder); builder (* TODO: expand to list of pointers for nested loops *)
      | SReturn e -> ignore(match fdecl.sFunctionType with
                     (* Special "return nothing" instr *)
                     A.Void -> L.build_ret_void builder
                     (* Build return statement *)
                     | _ -> L.build_ret (expr builder e) builder ); builder
      | SIfCondition (predicate, then_stmt, else_stmt) ->
         let bool_val = expr builder predicate in
        let merge_bb = L.append_block context "merge" the_function in
             let build_br_merge = L.build_br merge_bb in (* partial function *)

                let then_bb = L.append_block context "then" the_function in
                   add_terminal (stmt (L.builder_at_end context then_bb) then_stmt) build_br_merge;

                let else_bb = L.append_block context "else" the_function in
                   add_terminal (stmt (L.builder_at_end context else_bb) else_stmt) build_br_merge;

                ignore(L.build_cond_br bool_val then_bb else_bb builder);
                L.builder_at_end context merge_bb

      | SWhileLoop (predicate, body) ->
          let pred_bb = L.append_block context "while" the_function in
            let body_bb = L.append_block context "while_body" the_function in
              let merge_bb = L.append_block context "merge" the_function in
                let _ = L.build_br pred_bb builder in
                  let _ = break_block := L.value_of_block merge_bb in
                  let _ = continue_block := L.value_of_block pred_bb in
                    let pred_builder = L.builder_at_end context pred_bb in
                      let bool_val = expr pred_builder predicate in
                        let _ = L.build_cond_br bool_val body_bb merge_bb pred_builder in
                          add_terminal (stmt (L.builder_at_end context body_bb) body) (L.build_br pred_bb);
                          L.builder_at_end context merge_bb

      (* Implement for loops as while loops *)
      | SForLoop (e1, e2, e3, body) -> stmt builder
        ( SStatementBlock [SExpression e1 ; SWhileLoop (e2, SStatementBlock [body ; SExpression e3]) ] )
      (* Implement do until loops as while loops *)
      | SDoUntilLoop (body, (t, e)) -> stmt builder
        ( SStatementBlock [body ; SWhileLoop ( ( t, SUnrOP(A.Not, (t,e)) ) , body) ] )
      (* Implement do while loops as while loops *)
      | SDoWhileLoop (body, predicate) -> stmt builder
        ( SStatementBlock [body ; SWhileLoop (predicate, body) ] )
    in

    (* Build the code for each statement in the function *)
    let builder = stmt builder (SStatementBlock fdecl.sFunctionStatements) in

    (* Add a return if the last block falls off the end *)
    add_terminal builder (match fdecl.sFunctionType with
        A.Void -> L.build_ret_void
      | A.Float -> L.build_ret (L.const_float float_t 0.0)
      | t -> L.build_ret (L.const_int (ltype_of_casperType t) 0))
  in

  List.iter build_function_body functions;
  the_module
