(*
    Abstract Syntax Tree for Casper
    with functions for printing it
    Based on MicroC
    File: ast.ml
    Michael Makris, mm3443
    PLT Fall 2018
*)

type casperBinaryOperator =
      Add | Sub | Mul | Div | Mod | Exp
    | Con | Cat
    | Grt | Gre | Lst | Lse | Eql | Neq
    | And | Or

type casperUnaryOperator = Not | Neg | ItoF

type casperAssignment = Asgn | ConAsgn | AddAsgn | SubAsgn

type casperExpression =
      Epsilon
    | IntLIT of int
    | FltLIT of string
    | StrLIT of string
    | ChrLIT of char
    | BoolLIT of bool
    | VoidLIT
    | NullLIT
    | Identifier of string
    | BinOP of casperExpression * casperBinaryOperator * casperExpression
    | UnrOP of casperUnaryOperator * casperExpression
    | AsgnOP of string * casperAssignment * casperExpression
    | FunctionCall of string * casperExpression list

type casperStatement =
      StatementBlock of casperStatement list
    | Expression of casperExpression
    | IfCondition of casperExpression * casperStatement * casperStatement
    | ForLoop of casperExpression * casperExpression * casperExpression * casperStatement
    | WhileLoop of casperExpression * casperStatement
    | DoUntilLoop of casperStatement * casperExpression
    | DoWhileLoop of casperStatement * casperExpression    
    | Break
    | Continue
    | Return of casperExpression

type casperType = Int | Float| String | Char | Bool | Void

type casperVariable = casperType * string

type casperFunction = {
    functionType       : casperType;
    functionName       : string;
    functionFormals    : casperVariable list;
    functionLocals     : casperVariable list;
    functionStatements : casperStatement list;
}

type casperProgram = casperVariable list * casperFunction list

(* Pretty-printing functions *)
let string_of_casperBinaryOperator = function
      Add -> "+"
    | Sub -> "-"
    | Mul -> "*"
    | Div -> "/"
    | Mod -> "%"
    | Exp -> "^"    
    | Con -> "_"
    | Cat -> "?"
    | Grt -> ">"
    | Gre -> ">="
    | Lst -> "<"
    | Lse -> "<="
    | Eql -> "=="
    | Neq -> "!="
    | And -> "&&"
    | Or  -> "||"

let string_of_casperUnaryOperator = function
      Not -> "!"
    | Neg -> "-"
    | ItoF -> "~"

let string_of_casperAssignment = function
      Asgn     -> "="
    | ConAsgn  -> "_="
    | AddAsgn  -> "+="
    | SubAsgn  -> "-="

let rec string_of_casperExpression = function
      Epsilon -> ""
    | IntLIT(i) -> string_of_int i
    | FltLIT(s) -> s
    | StrLIT(s) -> s
    | ChrLIT(_) -> "" (* TODO fix Core module to print char c*)
    | BoolLIT(true) -> "true"
    | BoolLIT(false) -> "false"
    | VoidLIT -> ""
    | NullLIT -> ""
    | Identifier(s) -> s
    | BinOP(e1, o, e2) -> string_of_casperExpression e1 ^ " " ^ string_of_casperBinaryOperator o ^ " " ^ string_of_casperExpression e2
    | UnrOP(o, e) -> string_of_casperUnaryOperator o ^ string_of_casperExpression e
    | AsgnOP(s, o, e) -> s ^ " " ^ string_of_casperAssignment o ^ " " ^ string_of_casperExpression e
    | FunctionCall(f, el) -> f ^ "(" ^ String.concat ", " (List.map string_of_casperExpression el) ^ ")"
   
let string_of_casperType = function
      Int    -> "int"
    | Float  -> "float"
    | String -> "str"
    | Char   -> "chr"
    | Bool   -> "bool"
    | Void   -> "void"

let string_of_casperVariable (variableType, variableName) = string_of_casperType variableType ^ " " ^ variableName

let rec string_of_casperStatement = function
      StatementBlock (s) -> "{\n" ^ String.concat "" (List.map string_of_casperStatement s) ^ "}\n"
    | Expression(e) -> string_of_casperExpression e ^ ";\n";
    | IfCondition(e, s, StatementBlock([])) -> "if (" ^ string_of_casperExpression e ^ ") " ^ string_of_casperStatement s
    | IfCondition(e, s1, s2) ->  "if (" ^ string_of_casperExpression e ^ ")\n" ^
        string_of_casperStatement s1 ^ "else \n" ^ string_of_casperStatement s2
    | ForLoop(e1, e2, e3, s) -> "for (" ^ string_of_casperExpression e1 ^ "; " ^ string_of_casperExpression e2 ^ "; " 
        ^ string_of_casperExpression e3 ^ ")" ^ string_of_casperStatement s
    | WhileLoop(e, s) -> "while (" ^ string_of_casperExpression e ^ ") " ^ string_of_casperStatement s
    | DoUntilLoop(s, e) -> "do " ^ string_of_casperStatement s ^ "    until (" ^ string_of_casperExpression e ^ ");\n"
    | DoWhileLoop(s, e) -> "do " ^ string_of_casperStatement s ^ "    while (" ^ string_of_casperExpression e ^ ");\n"
    | Break     -> "break;\n";
    | Continue  -> "continue;\n";
    | Return(e) -> "return " ^ string_of_casperExpression e ^ ";\n"

let string_of_casperFunction casperFunction =
    string_of_casperType casperFunction.functionType ^ " " ^ casperFunction.functionName ^
    "(" ^ String.concat ", " (List.map string_of_casperVariable casperFunction.functionFormals) ^ ") {\n" ^
    String.concat ", " (List.map string_of_casperVariable casperFunction.functionLocals) ^
    String.concat "" (List.map string_of_casperStatement casperFunction.functionStatements) ^ "}\n"

let string_of_casperProgram (programGlobals, programFunctions) =
    String.concat ";\n" (List.map string_of_casperVariable programGlobals) ^ "\n" ^
    String.concat "\n" (List.map string_of_casperFunction programFunctions)

(* END Pretty-printing functions*)
