module L = Llvm
module A = Ast
open To_string

module StringMap = Map.Make(String)

type tw_val = {var_llval: L.llvalue; var_astype: A.datatype};;
type loc_arg = {arg_lltype: L.lltype; arg_name: string; arg_astype: A.datatype};;
type fun_type = {fun_llval: L.llvalue; fun_decl: A.vardecl; fun_args: loc_arg list};;
type fun_scope = {loc_map: fun_type StringMap.t; loc_args: loc_arg list; glob_map: tw_val StringMap.t};;
type scope = {dec_funs: fun_type StringMap.t; loc_vars: tw_val StringMap.t; glob_vars: tw_val StringMap.t};;
type fun_env = {loc_scope: scope; fun_name: string; fun_buil: L.llbuilder; fun_ll: L.llvalue};;

let translate (globals) =
    let context = L.global_context () in
    let the_module = L.create_module context "Twister" in  
    let builder = L.builder context  
    and i32_t = L.i32_type context
    and i8_t = L.i8_type context 
    and i1_t = L.i1_type context  (* for booleans*)
    and float_t   = L.double_type context
    and ptr_t = L.pointer_type (L.i8_type context) in

(* Matrices: n-d arrays *)
let vec_type t = L.packed_struct_type context [| i32_t ; (L.pointer_type t) |]
and mat_type t = L.packed_struct_type context [| i32_t ; i32_t ; (L.pointer_type t) |]

in
let counter = ref 0 in
let get_var_val v = v.var_llval
and get_arg_lltype arg = arg.arg_lltype
and get_arg_name arg = arg.arg_name in

let printf_t = L.var_arg_function_type i32_t [| L.pointer_type i8_t |]
in let printf_func = L.declare_function "printf" printf_t the_module in

let printbig_t = L.function_type i32_t [| i32_t |] in
let _ = L.declare_function "printbig" printbig_t the_module in

let open_file_t = L.function_type ptr_t [| L.pointer_type i8_t ; L.pointer_type i8_t|] in
let _ = L.declare_function "fopen" open_file_t the_module in

let read_t = L.function_type i32_t [| ptr_t ; i32_t ; i32_t ; ptr_t |] in
let _ = L.declare_function "fread" read_t the_module in

let get_lookup_val n lm gm = try StringMap.find n lm with Not_found -> StringMap.find n gm

in

let lookup n lm gm = let v = get_lookup_val n lm gm in (v.var_llval, v.var_astype)

in

let rec lltype_of_astype dt = match dt with
    | A.Int -> i32_t
    | A.Float -> float_t
    | A.Char -> i8_t
    | A.Bool -> i1_t
    | A.String -> vec_type i8_t
    | A.Tup(dt) -> vec_type (lltype_of_astype dt)
    | A.List(dt) -> vec_type (lltype_of_astype dt)
    | A.Matrix(dt) -> mat_type (lltype_of_astype dt)
    | A.Fun -> raise (Invalid_argument "Cannot determine type of function type without signature ")

in

(* allocates a local variable and returns a new locals map (fmap) *)
let allocate_loc vl ty asty name fmap builder=
    let _ = L.set_value_name name (vl) in
    let local = L.build_alloca ty name builder in
    let _ = ignore(L.build_store (vl) local builder ) in
    StringMap.add name {var_llval = local; var_astype = asty} fmap 

in

let add_terminal builder f = match L.block_terminator (L.insertion_block builder) with
   | Some _ -> ()
   | None -> ignore (f builder) 

in

let get_extra_args (fmap, li, builder) lasl = match (get_arg_lltype lasl) with
    | _ -> (fmap, (L.build_load (fst (lookup (get_arg_name lasl) fmap StringMap.empty)) (get_arg_name lasl) builder) :: li, builder)
    

in

let rec fill_arr arr vals index builder = match vals with
    | [] -> true
    | h :: t -> let index_const = L.const_int i32_t index in
        let index_ptr = L.build_gep arr [| index_const |] "%tmp_ptr" builder in
        let _ = L.build_store h index_ptr builder in
        fill_arr arr t (index + 1) builder

in

let extract_env env =
    (env.loc_scope.dec_funs, env.loc_scope.loc_vars, env.loc_scope.glob_vars, env.fun_name, env.fun_buil, env.fun_ll)

in

let store_env (funmap, fmap, gsm, cfn, builder, mai) =
    let s = {dec_funs = funmap; loc_vars = fmap; glob_vars = gsm} in
    {loc_scope = s; fun_name = cfn; fun_buil = builder; fun_ll = mai}

in

let eval_float_binop_w_type val1 val2 op = let res =
    (match op with 
        | A.Add -> L.build_fadd 
        | A.Sub -> L.build_fsub
        | A.Mul -> L.build_fmul
        | A.Div -> L.build_fdiv
        | A.Mod -> L.build_frem
        | A.Gt -> L.build_fcmp L.Fcmp.Ogt
        | A.Eq -> L.build_fcmp L.Fcmp.Oeq
        | A.Neq -> L.build_fcmp L.Fcmp.One 
        | A.Lt -> L.build_fcmp L.Fcmp.Olt 
        | A.Geq -> L.build_fcmp L.Fcmp.Oge
        | A.Leq -> L.build_fcmp L.Fcmp.Ole
    ) val1 val2 "%tmp" builder in
    let new_type = if List.mem op [A.Gt; A.Eq; A.Neq; A.Lt; A.Geq; A.Leq] then A.Bool else A.Float in
    {var_llval = res; var_astype = new_type}

in

(*evalues expressions, not statements*)
let rec eval_w_type env expr = 
    let (funmap, fmap, gmap, _, builder, _) = extract_env env in
    match expr with 
    | A.NumLit(sc) -> (match sc with 
        | A.LitInt(i) -> {var_llval = L.const_int i32_t i; var_astype = A.Int}
        | A.LitFloat(fl) -> {var_llval = L.const_float float_t fl; var_astype = A.Float}
    )
    | A.CharLit(c) -> {var_llval = L.const_int i8_t (Char.code c); var_astype = A.Char}
    | A.StringLit(str) -> {var_llval = L.build_global_stringptr str "%tmp" builder; var_astype = A.String}
    | A.BoolLit(b) -> {var_llval = L.const_int i1_t (if b then 1 else 0); var_astype = A.Bool}
    | A.TupLit(e_list) -> eval_w_type env (A.ListLit(e_list))
        (* Tuples are just lists restricted to scalars *)
    | A.ListLit(e_list) -> let entries = (List.map (eval_w_type env) e_list) in
        let entry_vals = List.map get_var_val entries in
        let (llvm_etype, ast_etype) = (match entries with
            | [] -> (i1_t, A.Bool)
            | h :: _ -> (L.type_of h.var_llval, h.var_astype)
        ) in
        let list_len = L.const_int i32_t (List.length entry_vals) in
        (* Build entries array *)
        let llvm_entry_arr = L.build_array_malloc llvm_etype list_len "%tmp_list_ents" builder in
        let _ = fill_arr llvm_entry_arr entry_vals 0 builder in
        (* Get type and init*)
        let l_type = vec_type llvm_etype in
        let list_struct = L.build_malloc l_type "%tmp_list_struct" builder in
        (* Fill values in struct *)
        let len_pos = L.build_struct_gep list_struct 0 "%tmp" builder in
        let _ = L.build_store list_len len_pos builder in
        let vals_pos = L.build_struct_gep list_struct 1 "%tmp" builder in
        let _ = L.build_store llvm_entry_arr vals_pos builder in
        {var_llval = L.build_load list_struct "%tmp" builder; var_astype = A.List(ast_etype)}
    | A.MatrixLit(row_list) -> let entries = List.flatten (List.map (List.map (eval_w_type env)) row_list) in
        let entry_vals = List.map get_var_val entries in
        let row_lengths = List.map List.length row_list in 
        if not (Semant.all_equal row_lengths) then
            begin
                let expr_str = To_string.string_of_expr(expr) in 
                let error_details = "Inconsistent row lengths found in " ^ expr_str in
                raise (Invalid_argument error_details)
            end
        else let num_cols = (match row_lengths with
                | [] -> 0
                | h :: _ -> h
            ) in
            let (llvm_etype, ast_etype) = (match entries with 
                | [] -> i32_t, A.Int
                | h :: _ -> (L.type_of h.var_llval, h.var_astype)
            ) in
            (* Get size constants*)
            let num_entries = List.length entry_vals in
            let llvm_num_rows = L.const_int i32_t (List.length row_list) in
            let llvm_num_cols = L.const_int i32_t (num_cols) in
            let llvm_num_ent = L.const_int i32_t num_entries in
            (* Build entries array *)
            let llvm_entry_arr = L.build_array_malloc llvm_etype llvm_num_ent "%tmp_mat_ents" builder in
            let _ = fill_arr llvm_entry_arr entry_vals 0 builder in
            (* Get type and init *)
            let l_type = mat_type llvm_etype in
            let mat_struct = L.build_malloc l_type "%tmp_mat_struct" builder in
            (* Fill values in struct *)
            let num_row_pos = L.build_struct_gep mat_struct 0 "%tmp" builder in
            let _ = L.build_store llvm_num_rows num_row_pos builder in
            let num_col_pos = L.build_struct_gep mat_struct 1 "%tmp" builder in
            let _ = L.build_store llvm_num_cols num_col_pos builder in
            let vals_pos = L.build_struct_gep mat_struct 2 "%tmp" builder in
            let _ = L.build_store llvm_entry_arr vals_pos builder in
            {var_llval = L.build_load mat_struct "tmp" builder; var_astype = A.Matrix(ast_etype)}
    | A.BinOp(e1, e2, op) -> let v1 = eval_w_type env e1 in let v2 = eval_w_type env e2 in
        (match v1.var_astype, v2.var_astype with
            | A.Int, A.Int -> let res = (match op with 
                | A.Add -> L.build_add 
                | A.Sub -> L.build_sub
                | A.Mul -> L.build_mul
                | A.Div -> L.build_sdiv
                | A.Mod -> L.build_srem
                | A.Gt -> L.build_icmp L.Icmp.Sgt
                | A.Eq -> L.build_icmp L.Icmp.Eq
                | A.Neq -> L.build_icmp L.Icmp.Ne 
                | A.Lt -> L.build_icmp L.Icmp.Slt
                | A.Geq -> L.build_icmp L.Icmp.Sge
                | A.Leq -> L.build_icmp L.Icmp.Sle
                ) v1.var_llval v2.var_llval "%tmp" builder in
                let new_type = if List.mem op [A.Gt; A.Eq; A.Neq; A.Lt; A.Geq; A.Leq] then A.Bool else A.Int in
                {var_llval = res; var_astype = new_type}
            | A.Float, A.Float -> eval_float_binop_w_type v1.var_llval v2.var_llval op
            | A.Int, A.Float -> let fv1 = L.build_sitofp v1.var_llval float_t "%tmp" builder in
                eval_float_binop_w_type fv1 v2.var_llval op
            | A.Float, A.Int -> eval_w_type env (A.BinOp(e2, e1, op))
            | A.Bool, A.Bool -> let res = 
                (match op with
                    | A.And -> L.build_and
                    | A.Or -> L.build_and
                    | A.Xor -> L.build_xor
                    | A.Eq -> L.build_icmp L.Icmp.Eq
                    | A.Neq -> L.build_icmp L.Icmp.Ne
                ) v1.var_llval v2.var_llval "%tmp" builder in
                {var_llval = res; var_astype = A.Bool}
        )
    | A.UnOp(e, op) -> let v = eval_w_type env e in
        let res = 
        (match op with
            | A.Not -> L.build_not 
            | A.Neg -> 
                (match v.var_astype with
                    | A.Int -> L.build_neg
                    | A.Float -> L.build_fneg
                )
        ) v.var_llval "%tmp" builder in
        {var_llval = res; var_astype = v.var_astype}
    | A.Id(id) -> let v = (lookup id fmap gmap) in {var_llval =  L.build_load (fst v) "%tmp" builder ; var_astype = snd v}
    | A.VecAcc(name, mat_index) -> (match mat_index with
        | A.MatIndex(i) -> let index = eval_w_type env i in
            let index_val = index.var_llval in
             
            let vec = lookup name fmap gmap in
            
            let vec_struc = (fst vec)  in
            let entry_ptr = L.build_struct_gep vec_struc 1 "%tmp" builder in
            let entries = L.build_load entry_ptr "%tmp" builder in
            let index_ptr = L.build_gep entries [| index_val |] "%tmp" builder in
            let ast_etype = (match (snd vec) with
                | A.List(et) -> et
                | A.Tup(et) -> et
                | _ -> let error_details = "Cannot index once into type " ^ To_string.string_of_datatype (snd vec) in
                    raise (Invalid_argument error_details)
            ) in
            {var_llval = L.build_load index_ptr "%tmp" builder; var_astype = ast_etype}
    )
    | A.MatAcc(name, ind_1, ind_2) -> (match ind_1, ind_2 with
        | A.MatIndex(i1), A.MatIndex(i2) -> let i1' = eval_w_type env i1 in
            let i2' = eval_w_type env i2 in
            let (i1_val, i2_val) = (i1'.var_llval, i2'.var_llval) in
            let mat = lookup name fmap gmap in
            let mat_struc =  (fst mat)  in
            let num_cols_ptr = L.build_struct_gep mat_struc 1 "%tmp" builder in
            let num_cols = L.build_load num_cols_ptr "%tmp" builder in
            let rows_offset = L.build_mul num_cols i1_val "%tmp" builder in
            let offset = L.build_add rows_offset i2_val "%tmp" builder in
            let mat_vals_ptr = L.build_struct_gep mat_struc 2 "%tmp" builder in
            let mat_vals = L.build_load mat_vals_ptr "%tmp" builder in
            let this_entry_ptr = L.build_gep mat_vals [| offset |] "%tmp" builder in
            let ast_etype = (match (snd mat) with
                | Matrix(etype) -> etype
            ) in
            {var_llval =  L.build_load this_entry_ptr "%tmp" builder ; var_astype = ast_etype}
    )
    | A.Call(id, o) -> (match id with
        | "println_int" -> let tbp = eval_w_type env (List.hd o) in
            let v = L.build_global_stringptr "%d\n" "fmt" builder in
            let _ = L.build_call printf_func [|v ; tbp.var_llval |]  "print" builder in
            {var_llval = L.const_int i32_t 1; var_astype = Int}
        | "print_int" -> let tbp = eval_w_type env (List.hd o) in
            let v = L.build_global_stringptr "%d" "fmt" builder in
            let _ = L.build_call printf_func [|v ; tbp.var_llval |]  "print" builder in
            {var_llval = L.const_int i32_t 1; var_astype = Int}

        | "print_fl" -> let tbp = eval_w_type env (List.hd o) in
            let v = L.build_global_stringptr "%f\n" "fmt" builder in
            let _ = L.build_call printf_func [|v ; tbp.var_llval |]  "print" builder in
            {var_llval = L.const_int i32_t 1; var_astype = Int}
        | "println" -> let tbp = eval_w_type env (List.hd o) in
            let v = L.build_global_stringptr "%s\n" "fmt" builder in
            let _ = L.build_call printf_func [|v ; tbp.var_llval |]  "print" builder in
            {var_llval = L.const_int i32_t 1; var_astype = Int}

        | "print" -> let tbp = eval_w_type env (List.hd o) in
            let v = L.build_global_stringptr "%s" "fmt" builder in
            let _ = L.build_call printf_func [|v ; tbp.var_llval |]  "print" builder in
            {var_llval = L.const_int i32_t 1; var_astype = Int}

        | "print_c" -> let tbp = eval_w_type env (List.hd o) in
            let v = L.build_global_stringptr "%c\n" "fmt" builder in
            let _ = L.build_call printf_func [|v ; tbp.var_llval |]  "print" builder in
            {var_llval = L.const_int i32_t 1; var_astype = Int}       
            
        
        | "tofl" -> (match (List.hd o) with
            | A.NumLit(sc) -> match sc with
                | A.LitInt(i) -> {var_llval = L.build_sitofp (L.const_int i32_t i) float_t "%tmp" builder;
                    var_astype = A.Float}
        )
        | "toint" -> let rhs = eval_w_type env (List.hd o) in
            let newlhs = L.build_fptosi rhs.var_llval i32_t "%tmp" builder in
            {var_llval = newlhs; var_astype = A.Int}
        | _ -> let fun_val = StringMap.find id funmap in
            
            let (fdef, fdecl, more_list) = (fun_val.fun_llval, fun_val.fun_decl, fun_val.fun_args) in
            let gll v = v.var_llval in
            let argl =  (List.map gll ( List.map (eval_w_type env) o )) in
             
            let (_, extra_args, _) = List.fold_left get_extra_args (fmap, [], builder) more_list in
            
            let all_arls = argl @ extra_args in
            let res = L.build_call fdef (Array.of_list all_arls) ( id ^ "_result" ) builder in
            {var_llval = res; var_astype = fdecl.A.return_type}
    )
    | A.Attribute(id, atid) -> (match atid with
        | "length" -> let list_struct = L.build_load (fst (lookup id fmap gmap))  "%tmp" builder in
            let entry_ptr = L.build_struct_gep list_struct 0 "%tmp" builder in
            let _ = L.build_gep entry_ptr [| L.const_int i32_t 0 |] "%tmp" builder in
            {var_llval = L.build_load entry_ptr "%tmp" builder; var_astype = A.Int}
        | "num_rows" -> let mat = lookup id fmap gmap in
            let mat_struc =  (fst mat)  in
            let num_cols_ptr = L.build_struct_gep mat_struc 0 "%tmp" builder in
            let num_cols = L.build_load num_cols_ptr "%tmp" builder in {var_llval = num_cols ; var_astype = A.Int}
        | "num_cols" -> let mat = lookup id fmap gmap in
            let mat_struc =  (fst mat)  in
            let num_cols_ptr = L.build_struct_gep mat_struc 1 "%tmp" builder in
            let num_cols = L.build_load num_cols_ptr "%tmp" builder in {var_llval = num_cols ; var_astype = A.Int}
        | _ -> raise (Invalid_argument ("Unsupported attribute: " ^ atid))
    )


in

let eval env expr = let v = eval_w_type env expr in v.var_llval

in

let get_range env cal  =
    let (funmap, fmap, gmap, cfn, builder, mai) = extract_env env in
    let the_function = try let tf = (StringMap.find cfn funmap) in tf.fun_llval with Not_found -> mai in
    match cal with
    | A.ItCall(id, o) -> (let i = eval env (List.hd o) and j = eval env (List.nth o 1) in
        let len = (L.build_sub j i "%tmp" builder) in
        let fmap = allocate_loc (len) (i32_t) A.Int "len_var" fmap builder in
        let fmap = allocate_loc (i) (i32_t) A.Int "i_var" fmap builder in
        let fmap = allocate_loc (j) (i32_t) A.Int "j_var" fmap builder in
        let llvm_entry_arr = L.build_array_malloc i32_t len "%tmp_list_ents" builder in
        let fmap = allocate_loc (L.const_int i32_t 0) (i32_t) A.Int "for_loop_var" fmap builder in
        let pred_bb = L.append_block context "while" the_function in
        let _ = ignore (L.build_br pred_bb builder) in

        let body_bb = L.append_block context "while_body" the_function in
        (* assign id to an element of the list*)
        let flv = L.build_load (fst (lookup "for_loop_var" fmap gmap)) "for_loop_var" (L.builder_at_end context body_bb) in
        let iv = L.build_load (fst (lookup "i_var" fmap gmap)) "i_var" (L.builder_at_end context body_bb) in
        let index_ptr = L.build_gep llvm_entry_arr [| flv |] "%tmp_ptr" (L.builder_at_end context body_bb) in
        let _ = L.build_store iv index_ptr (L.builder_at_end context body_bb) in

        (*increment iv*)
        let add_one = L.build_add iv (L.const_int i32_t 1) "%tmp" (L.builder_at_end context body_bb) in
        let _ = L.build_store add_one (fst (lookup "i_var" fmap gmap)) (L.builder_at_end context body_bb) in
        (* increment flv*)
        let add_one = L.build_add flv (L.const_int i32_t 1) "%tmp" (L.builder_at_end context body_bb) in
        let _ = L.build_store add_one (fst (lookup "for_loop_var" fmap gmap)) (L.builder_at_end context body_bb) in
        let _ = add_terminal (L.builder_at_end context body_bb) (L.build_br pred_bb) in

        let pred_builder = L.builder_at_end context pred_bb in

        let new_scope = {env.loc_scope with loc_vars = fmap} in
        let new_env = {env with loc_scope = new_scope; fun_buil = pred_builder} in

        let flv = eval new_env (A.Id("for_loop_var")) in
        let bool_val = L.build_icmp L.Icmp.Slt flv len "%tmp" pred_builder in
        let merge_bb = L.append_block context "merge" the_function in
        let merge_buil = L.builder_at_end context merge_bb  in

        let l_type = vec_type i32_t in
        let list_struct = L.build_malloc l_type "%tmp_list_struct" (merge_buil) in
        (* Fill values in struct *)
        let len_pos = L.build_struct_gep list_struct 0 "%tmp" (merge_buil) in
        let _ = L.build_store len len_pos (merge_buil) in
        let vals_pos = L.build_struct_gep list_struct 1 "%tmp" (merge_buil) in
        let _ = L.build_store llvm_entry_arr vals_pos (merge_buil) in
        let _ = ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder) in
        (list_struct,  L.builder_at_end context merge_bb)
    )

(* TODO: don't forget to move the built in functions here as well*)

in

(*looks at and builds out the statements inside a function definition*)
(*fmap is the formals map which will also hold the local variables, gsm is the global map*)
(* cfn is the current function name; mai is ? *)
(*bd is a statement*)
let rec bexpr env bd =
 
    let (funmap, fmap, gsm, cfn, builder, mai) = extract_env env in
     
    let the_function = try let tf = (StringMap.find cfn funmap) in tf.fun_llval with Not_found -> mai
    in match bd with 
    | A.Decl(vd) -> (match vd.A.return_type with 
        | A.Fun -> env
        (*| A.Matrix(dt) -> (let e2 = eval env vd.A.body in
            let fmap = StringMap.add vd.A.var_name {var_llval = e2; var_astype = vd.A.return_type} fmap in
            let cur_scope = {env.loc_scope with loc_vars = fmap} in
            {env with loc_scope = cur_scope}
        )
        | A.List(dt) -> ( let e2 = eval env vd.A.body in
            
            let fmap = StringMap.add vd.A.var_name {var_llval = e2; var_astype = vd.A.return_type} fmap in
            (*let local = L.build_alloca (L.type_of e2) name builder in*)           
            (*let _ = L.build_store   e2 local  builder in*)
            let cur_scope = {env.loc_scope with loc_vars = fmap} in
            {env with loc_scope = cur_scope}
        )*)
        |_ ->   (let rhs = eval env vd.A.body in
            let typ = L.type_of rhs in
            let fmap = allocate_loc rhs typ vd.A.return_type vd.A.var_name fmap builder in
            let cur_scope = {env.loc_scope with loc_vars = fmap} in
            {env with loc_scope = cur_scope}
        )
    )
    | A.MatAssign(asnm) -> (let matr = (fst (lookup asnm.A.mat_name fmap gsm )) in
           match asnm.A.row_index with
        | A.MatIndex(ie) -> (match asnm.A.col_index with
            | A.MatIndex(je) -> (let i = eval env  ie in
                let j = eval env je in
                let newval = eval env asnm.A.mat_el_val in
                let num_cols_ptr = L.build_struct_gep matr 1 "%tmp" builder in
                let num_cols = L.build_load num_cols_ptr "%tmp" builder in
                let rows_offset = L.build_mul num_cols i "%tmp" builder in
                let offset = L.build_add rows_offset j "%tmp" builder in
                let mat_vals_ptr = L.build_struct_gep matr 2 "%tmp" builder in
                let mat_vals = L.build_load mat_vals_ptr "%tmp" builder in
                let this_entry_ptr = L.build_gep mat_vals [| offset |] "%tmp" builder in
                let _ = L.build_store newval this_entry_ptr builder in
                env
            )
        )
    )                      
    | A.Return (e) -> (let e' = eval env e in
        let _ = L.build_ret ( e') builder in
        env
    )
    | A.If(expr, ifst, els) -> let pred = eval env expr in 
        let merge_bb = L.append_block context "merge" the_function in 
        let then_bb = L.append_block context "then" the_function in
        let buil= L.builder_at_end context then_bb in
        let nenv = {env with fun_buil = buil} in 
        let nnenv = List.fold_left bexpr nenv ifst in
        let builth = nnenv.fun_buil in
        add_terminal builth (L.build_br merge_bb);

        let else_bb = L.append_block context "else" the_function in
        let buil = L.builder_at_end context else_bb in
        let nenv = {env with fun_buil = buil} in 
        let nnenv = List.fold_left bexpr nenv els in
        let buil = nnenv.fun_buil in
        add_terminal buil (L.build_br merge_bb);
        let _ = ignore(L.build_cond_br pred then_bb else_bb builder) in
        {env with fun_buil = L.builder_at_end context merge_bb}
    | A.For(id, it, stms) -> (let (a, builder, fty) = 
        (match it with 
            | A.ItId(str) -> ( (fst (lookup str fmap gsm) )  , builder, (snd (lookup str fmap gsm)))
            | A.ItCall(id, o) -> let r = get_range env it in ((fst r), (snd r), A.Int)
        ) in

        let fet = (match fty with
        | A.Int -> A.Int
        | A.List(dt) -> (dt) ) in
        (* allocate space for the id in the loop*)
        (*get the first element of the list*)
        let _ = counter := !counter + 1 in 
        let ep = L.build_struct_gep a 1 "%tmp" builder in
        let es = L.build_load ep "%tmp" builder in
        let ip = L.build_gep es [| L.const_int i32_t 0 |] "%tmp" builder in
        let felem = L.build_load ip "%tmp" builder in
        (* put that first element in id*) 

        
        let fmap = allocate_loc (felem) (L.type_of felem) fet id fmap builder in
        (* get the length of the list*)
        let ep = L.build_struct_gep a 0 "%tmp" (builder) in
        let len = L.build_load ep "%tmp" builder in

        (* allocate a for loop var that we will increment at the end of each loop and check against len*) 
        let fmap = allocate_loc (L.const_int i32_t 0) (i32_t) A.Int ("for_loop_var" ^ (string_of_int !counter)) fmap builder in 

        let pred_bb = L.append_block context "while" the_function in
        let _ = ignore (L.build_br pred_bb builder) in
        let body_bb = L.append_block context "while_body" the_function in
        (* assign id to an element of the list*)
        let i = L.build_load (fst (lookup ("for_loop_var" ^ string_of_int !counter) fmap gsm)) ("for_loop_var" ^ string_of_int !counter) (L.builder_at_end context body_bb) in

        let entry_ptr = L.build_struct_gep a 1 "%tmp" (L.builder_at_end context body_bb) in
        let entries = L.build_load entry_ptr "%tmp" (L.builder_at_end context body_bb) in
        let index_ptr = L.build_gep entries [| i |] "%tmp" (L.builder_at_end context body_bb) in

        let list_val = L.build_load index_ptr "%tmp" (L.builder_at_end context body_bb) in
        
        let _ = L.build_store list_val (fst (lookup id fmap gsm)) (L.builder_at_end context body_bb) in
        let builder = L.builder_at_end context body_bb in

        let cur_scope = {env.loc_scope with loc_vars = fmap} in
        let env = {env with loc_scope = cur_scope; fun_buil = builder} in
        let env = List.fold_left bexpr env stms in
        let (funmap, fmap , gsm, cfn, buil, mai) = extract_env env in
        (* increment the counter variable*)
        let pold = L.build_load (fst (lookup ("for_loop_var" ^ string_of_int !counter) fmap gsm)) "for_loop_var" buil in                       
        let add_one = L.build_add pold (L.const_int i32_t 1) "%tmp" buil in
        let _ = L.build_store add_one (fst (lookup ("for_loop_var" ^ (string_of_int !counter)) fmap gsm)) buil in  
        let _ = add_terminal buil (L.build_br pred_bb) in

        let pred_builder = L.builder_at_end context pred_bb in

        let new_scope = {env.loc_scope with loc_vars = fmap} in
        let env = {env with loc_scope = new_scope; fun_buil = pred_builder} in
        let flv = eval env (A.Id("for_loop_var" ^ (string_of_int !counter))) in
        let bool_val = L.build_icmp L.Icmp.Slt flv len "%tmp" pred_builder in
        let merge_bb = L.append_block context "merge" the_function in
        let _ =  counter := !counter -1 in
        (* CHANGED*)
        let fmap = StringMap.remove id fmap in 
        let _ = ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder) in

        (* while the for loop var in less than the length of the array*)
        let builder = L.builder_at_end context merge_bb in
        let cur_scope = {env.loc_scope with loc_vars = fmap} in
        {env with loc_scope = cur_scope; fun_buil = builder}
    )   
    | A.Assign(asn) -> let rhs = eval env asn.A.new_val in
        let _ = L.build_store rhs (fst (lookup asn.A.assign_name fmap gsm)) builder in
        env
    | A.VecAssign(v) -> let rhs = eval env v.A.vec_el_val in
        let index = (match v.A.index with
            | A.MatIndex(e) -> eval env e
        ) in
        let vec = (fst (lookup v.A.vec_name fmap gsm)) in
        (* Get array *)
        let vals_ptr = L.build_struct_gep vec 1 "%tmp_vec_vals" builder in
        let vals_arr = L.build_load vals_ptr "%tmp_ve_arr" builder in
        (* Store value *)
        let val_ptr = L.build_gep vals_arr [| index |] "%tmp_el_ptr" builder in
        let _ = L.build_store rhs val_ptr builder in
        env
    | A.MatAssign(m) -> let rhs = eval env m.A.mat_el_val in
        (* Get loc *)
        let row_index = (match m.A.row_index with
            | A.MatIndex(e) -> eval env e
        ) in
        let col_index = (match m.A.col_index with
            | A.MatIndex(e) -> eval env e
        ) in
        (* Get array *)
        let vec = (fst (lookup m.A.mat_name fmap gsm)) in
        let vals_ptr = L.build_struct_gep vec 2 "%tmp_vec_vals" builder in
        let vals_arr = L.build_load vals_ptr "%tmp_ve_arr" builder in
        (* Get loc *)
        let num_cols_ptr = L.build_struct_gep vec 1 "%tmp" builder in
        let num_cols = L.build_load num_cols_ptr "%tmp" builder in
        let offset = L.build_mul row_index num_cols "%tmp_offset" builder in
        let val_ptr = L.build_gep vals_arr [| offset |] "%tmp_el_ptr" builder in
        let _ = L.build_store rhs val_ptr builder in
        env

in

(*helps make the argument list types for the function prototype*)
let make_argl_helper arg_li arg = match arg with 
    | A.ArgId (str, dt) -> {arg_lltype = lltype_of_astype dt; arg_name = str; arg_astype = dt} :: arg_li 

in

let argl_names lnames ale = match ale with 
    | A.ArgId(str, dt) -> str :: lnames

in

let make_argl al = match al with 
    | hd :: tl ->   List.rev (List.fold_left make_argl_helper [] al)
    | [] ->  []

in

let add_formal  form_map form_val form_param =
    match form_val.arg_astype with
   | _ -> ( 
    let (t, n, asty) = (form_val.arg_lltype, form_val.arg_name, form_val.arg_astype) in
    let _ = L.set_value_name n form_param in
    let local = L.build_alloca t n builder in
    let _ = ignore (L.build_store form_param local builder) in 
    StringMap.add n {var_llval = local; var_astype = asty} form_map)

in

(*recursively builds the inner functions first*)
(*processes statements*)
(*lasl will contain the extra arguments we have to add to the inner functions*)
(* cfn is the current function name*)
let rec bfn scope bd = let (fsm, lasl, gsm) = (scope.loc_map, scope.loc_args, scope.glob_map) in
    match bd with 
    | A.Decl (vd) -> ( match vd.A.return_type with 
        | A.Fun -> ( match vd.A.body with 
            | A.FunLit(ft, bd) -> ( match ft with 
                | A.FunType(al, frt) -> ( match frt with 
                    | A.ReturnData( dt) ->(
                                          let final_scope = List.fold_left bfn scope bd in
                                          let fsm = final_scope.loc_map in
                                          let argl = make_argl al in
                                          let append = Array.of_list ((List.map get_arg_lltype argl) @ (List.rev (List.map get_arg_lltype lasl))) in
                                          let rt = lltype_of_astype dt in
                                          let t = L.function_type rt append in
                                          let tf = L.declare_function (vd.A.var_name  ) t the_module in
                                          let fsm = StringMap.add vd.A.var_name {fun_llval = tf; fun_decl = vd; fun_args = lasl} fsm in
                                          let bb = L.append_block context "entry" tf in
                                          let _ = L.position_at_end bb builder in
                                          let formals = (argl @ (List.rev lasl)) in
                                          let fmap = List.fold_left2 ( add_formal  ) StringMap.empty formals (Array.to_list (L.params tf)) in
                                          let curfn_scope = {dec_funs = fsm; loc_vars = fmap; glob_vars = gsm} in
                                          let env = {loc_scope = curfn_scope; fun_name = vd.A.var_name; fun_buil = builder; fun_ll = tf} in
                                          let env = List.fold_left bexpr env bd in
                                         
                                          {loc_map = env.loc_scope.dec_funs; loc_args = lasl; glob_map = env.loc_scope.glob_vars}
                    ) 
                )
            )
        )
        | _ ->  (let arg = {arg_lltype = lltype_of_astype vd.A.return_type; arg_name = vd.A.var_name; arg_astype = vd.A.return_type} in
            let args = arg :: lasl in
            {loc_map = fsm; loc_args = args; glob_map = gsm}) 
      

    )
    | A.Return(r) -> {loc_map = fsm; loc_args = lasl; glob_map = gsm}    
    | A.Assign(asn) -> {loc_map = fsm; loc_args = lasl; glob_map = gsm}    
    | _ -> {loc_map = fsm; loc_args = lasl; glob_map = gsm}    

in

(*first step in building function definitions*)
let build_fun (sm, gsm, eal) g = match g.A.body with 
    | A.FunLit(ft, sts) -> ( match ft with
        | A.FunType(al, frt) -> ( match frt with
            | A.ReturnData(dt) -> (let new_scope = {loc_map = sm; loc_args = eal; glob_map = gsm} in
               (* generate the inner function*)
               let new_scope = List.fold_left bfn new_scope sts in
               (* sm is the function map*)
               let (sm, _) = (new_scope.loc_map, new_scope.loc_args) in
               let argl =  (List.map get_arg_lltype (make_argl al)) in
               let tot_argl = Array.of_list ( argl @ ( List.map get_arg_lltype(List.rev eal))) in 
               let rt = lltype_of_astype dt in
               let t = L.function_type rt tot_argl in
               let tf = L.declare_function g.A.var_name t the_module in
               let bb = L.append_block context "entry" tf in
               let _ = L.position_at_end bb builder in
               let formals = make_argl al in
               let tot_args = (formals @ ( List.rev eal )) in
               let fmap = List.fold_left2 (add_formal ) StringMap.empty tot_args (Array.to_list (L.params tf)) in
               let smn = StringMap.add g.A.var_name {fun_llval = tf; fun_decl = g; fun_args = (List.rev eal)} sm in
               let new_fn_scope = {dec_funs = smn; loc_vars = fmap; glob_vars = gsm} in
               let env = {loc_scope = new_fn_scope; fun_name = g.A.var_name; fun_buil = builder; fun_ll = tf} in
               let env = List.fold_left bexpr env sts in
            
               env.loc_scope.dec_funs
            )
        )
    )

in 					


(*funsm in check_globs in the function definintions map, which we will need for calling*)
(*globsm is the global string map*)

let check_globs (funsm, globsm, eal) g = match g with
    | A.Decl(dl) -> ( match dl.A.return_type with
        |A.Fun -> let nsm = build_fun (funsm, globsm, eal) dl in (nsm, globsm, eal)
        | A.List(dt) -> ( (* add it to the extra argument list*)
                         (*let nm = StringMap.add dl.A.var_name var globsm in*)
                         let arg = {arg_lltype = lltype_of_astype dl.A.return_type; arg_name = dl.A.var_name; arg_astype = dl.A.return_type} in
                         let eal = arg :: eal in
                         (funsm, globsm, eal)
        )
        | _ -> ( match dl.A.body with
            (*| A.Call(id, o) -> (funsm, globsm, eal)*)
            | _ ->(
                (* just putting print_f here as a dummy*)
               

                let arg = {arg_lltype = lltype_of_astype dl.A.return_type; arg_name = dl.A.var_name; arg_astype = dl.A.return_type} in
                         let eal = arg :: eal in
                (funsm, globsm, eal))
        )
        | _ -> (funsm, globsm, eal)
    ) 
    | _ -> (funsm, globsm, eal)

in

let (fdecls, _ ,  _) = List.fold_left check_globs (StringMap.empty, StringMap.empty, [])  globals

in

let m_arr = [| |] in
let m_t = L.function_type i32_t m_arr in
let mai = L.declare_function "main" m_t the_module in


let bb = L.append_block context "entry" mai in
let _ = L.position_at_end bb builder in 

(*builds the calls for functions*)
let build_call_help (sm, builder) g = match  g with
    | A.Decl(dl) -> (  match dl.A.return_type with

        | A.Fun -> (sm, builder)
        | _ -> (let env = store_env (fdecls, sm, sm, "main", builder, mai) in
            let e2 = eval env dl.A.body in
           
            let sm = allocate_loc e2 (L.type_of e2) dl.A.return_type dl.A.var_name sm env.fun_buil in
          
            (sm, builder))
)
    | A.Assign(asn) -> (let lhs = (fst (lookup asn.A.assign_name sm sm)) in
        let env = store_env (fdecls, sm, sm, "main", builder, mai) in
        let rhs = eval env asn.A.new_val in
        let _ =  L.build_store rhs (lhs) builder in (env.loc_scope.loc_vars, builder)
    )
     | A.If(pred, thn, els) -> let cur_scope = {dec_funs = fdecls; loc_vars = sm; glob_vars = sm} in
        let loc_env = {loc_scope = cur_scope; fun_name = "main"; fun_buil = builder; fun_ll = mai} in
        let res_env = bexpr loc_env g in
        (res_env.loc_scope.loc_vars, res_env.fun_buil)
    | A.For(id, itid, sts) -> (let cur_scope = {dec_funs = fdecls; loc_vars = sm; glob_vars = sm} in
        let loc_env = {loc_scope = cur_scope; fun_name = "main"; fun_buil = builder; fun_ll = mai} in
        let res_env = bexpr loc_env g in
        (res_env.loc_scope.loc_vars, res_env.fun_buil))
    | A.MatAssign(asnm) ->(let matr = (fst (lookup asnm.A.mat_name sm sm )) in
           match asnm.A.row_index with
        | A.MatIndex(ie) -> (match asnm.A.col_index with
            | A.MatIndex(je) -> (let env = store_env (fdecls, sm, sm, "main", builder, mai) in let i = eval env  ie in
                let j = eval env je in
                let newval = eval env asnm.A.mat_el_val in
                let num_cols_ptr = L.build_struct_gep matr 1 "%tmp" builder in
                let num_cols = L.build_load num_cols_ptr "%tmp" builder in
                let rows_offset = L.build_mul num_cols i "%tmp" builder in
                let offset = L.build_add rows_offset j "%tmp" builder in
                let mat_vals_ptr = L.build_struct_gep matr 2 "%tmp" builder in
                let mat_vals = L.build_load mat_vals_ptr "%tmp" builder in
                let this_entry_ptr = L.build_gep mat_vals [| offset |] "%tmp" builder in
                let _ = L.build_store newval this_entry_ptr builder in
                (env.loc_scope.loc_vars, env.fun_buil)
            )
        )
    )
(*let lhs = L.build_load (fst (lookup asnm.A.mat_name sm sm)) "%tmp" builder in
        let env = store_env (fdecls, sm, sm, "main", builder, mai) in
        let rhs = eval env asnm.A.mat_el_val in
        let _ =  L.build_store rhs (lhs) builder in (env.loc_scope.loc_vars, builder)*) 
    | A.VecAssign(va) ->( 
         let env = store_env (fdecls, sm, sm, "main", builder, mai) in
        let index = (match va.A.index with
            | A.MatIndex(e) -> eval env e
        ) in
        let vec =  (fst (lookup va.A.vec_name sm sm)) in let vals_ptr = L.build_struct_gep vec 1 "%tmp_vec_vals" builder in
        let rhs = eval env va.A.vec_el_val in
        let vals_arr = L.build_load vals_ptr "%tmp_ve_arr" builder in
        (* Store value *)
        let val_ptr = L.build_gep vals_arr [| index |] "%tmp_el_ptr" builder in
        let _ = L.build_store rhs val_ptr builder in (env.loc_scope.loc_vars, builder)) (*let lhs =  (fst (lookup va.A.vec_name sm sm)) in
                          let env = store_env (fdecls, sm, sm, "main", builder, mai) in
                          let rhs = eval env va.A.vec_el_val in
                     let _ =  L.build_store rhs (lhs) builder in (env.loc_scope.loc_vars, builder)


*)

    |_ -> (sm, builder)
in

(* now we actually call the functions in main *)
let (_, nb) = List.fold_left build_call_help (StringMap.empty, builder) globals

in

let _ = L.build_ret (L.const_int i32_t 0) nb in

the_module
