(* Semantic checking for the MicroC compiler *)

open Ast
open Sast
module E = Exceptions

module StringMap = Map.Make(String)


type env = {
    env_fmap : fdecl StringMap.t;
    env_fname : string;
    env_return_type : Ast.datatype;
    env_globals : Ast.datatype StringMap.t;
    env_flocals : Ast.datatype StringMap.t;
    env_in_while: bool;
    env_set_return : bool; 
    env_sfmap : sfdecl StringMap.t
}

let rec expr_to_sexpr e env = match e with    
    IntLit(i)           -> (SIntLit(i, Datatype(Int)), env)
  | FloatLit(b)         -> (SFloatLit(b, Datatype(Float)), env)
  | StrLit(s)           -> (SStrLit(s, Datatype(String)), env)
  | BoolLit(b)          -> (SBoolLit(b, Datatype(Bool)), env)
  | Id(s)               -> (check_variable_access s env, env) 
  | Noexpr              -> (SNoexpr, env)
  | Unop(op, e)         -> (check_unop op e env)
  | Binop(e1, op, e2)   -> (check_binop e1 op e2 env)
  | Call("print", e_l)  -> (check_print e_l env)
  | Call("set", e_l)    -> (check_call_set e_l env)
  (* Library functions *)
  | Call("str", e_l)    -> (check_call_built "str" e_l env)
  | Call("read", e_l)   -> (check_call_built "read" e_l env)
  | Call("open", e_l)   -> (check_call_built "open" e_l env)
  | Call("write", e_l)  -> (check_call_built "write" e_l env)
  | Call("split", e_l)  -> (check_call_built "split" e_l env)
  | Call("close", e_l)  -> (check_call_built "close" e_l env)
  | Call(s, e_l)        -> (check_call s e_l env)
  | Set(e_l)            -> (check_set e_l env)
  | SetAccess(s, e)     -> (check_access s e env)
  | Slice(s, e1, e2)    -> (check_slice s e1 e2 env)


and check_slice s e1 e2 env = 
    let ss = check_variable_access s env in 
    let s_typ = try StringMap.find s env.env_flocals with 
    Not_found -> StringMap.find s env.env_globals in 
    let s_e_typ = match s_typ with Settype(a) -> a 
            | _ -> (raise E.InvalidSetAccess) in
    let se1, env = expr_to_sexpr e1 env in
    let se2, env = expr_to_sexpr e2 env in
    let typ1 = sexpr_to_type se1 in 
    let typ2 = sexpr_to_type se2 in 
    if typ1 <> Datatype(Int) && typ2 <> Datatype(Int)
    then raise E.InvalidSliceOperation
    else match s_e_typ with 
        Int -> (SCall("slicei", [ss; se1; se2], s_typ)), env 
        | String -> (SCall("slices", [ss; se1; se2;], s_typ)), env
        | Float ->  (SCall("slicef", [ss; se1; se2;], s_typ)), env
        | Bool -> (SCall("sliceb", [ss; se1; se2;], s_typ)), env
        | _ -> raise E.InvalidSliceOperation 



and check_access s e env = 
    let _ = check_variable_access s env in 
    let s_typ = try StringMap.find s env.env_flocals with 
    Not_found -> StringMap.find s env.env_globals in 
    let s_e_typ = match s_typ with Settype(a) -> a
            | _ -> (raise E.InvalidSetAccess) in
    let se, env = expr_to_sexpr e env in 
    let typ = sexpr_to_type se in 
    if typ <> Datatype(Int) then raise E.InvalidSetAccess 
    else (SSetAccess(s, se, Datatype(s_e_typ)), env)


and check_set el env =
    if (List.length el = 0) then SSet([], Settype(Void)), env 
    else  
        let se_first, env = expr_to_sexpr (List.hd el) env in 
        let e_typ = sexpr_to_type se_first in 
        let set_typ = match e_typ with 
            Datatype(t) -> t | _ -> (raise E.Invalid) in
        let env_ref = ref (env) in 
        let se_l = List.rev(List.fold_left
        (fun l e -> 
            let se, env = expr_to_sexpr e !env_ref in
            env_ref := env;
            let typ = sexpr_to_type se in
            if (typ <> e_typ) then raise E.DifferentSetElementType
            else (se :: l) ) [] (List.tl el)) in
        let se_l_final = se_first :: se_l in 
        ((SSet(se_l_final, Settype(set_typ))), !env_ref)



and check_variable_access s env = 
    try let typ =  StringMap.find s env.env_flocals 
    in SId(s, typ) with Not_found -> 
        (try let typ = StringMap.find s
    env.env_globals in SId(s, typ) with 
    Not_found -> (raise (E.UndefinedId s)))
  

and expr_list_to_sexpr_list e_l env = 
   let env_ref = ref(env) in 
   match e_l with 
   hd :: tl -> 
       let (se, env) = expr_to_sexpr hd !env_ref in
       env_ref := env; 
       let (l, env) = expr_list_to_sexpr_list tl !env_ref in 
       env_ref := env;
       (se :: l, !env_ref)
  | [] -> ([], !env_ref) 


and check_unop op e env =  
    let check_set_unop op = match op with 
        Card -> Datatype(Int)  
        | _  -> raise (E.InvalidUnaryOperation)
    in
    let check_bool_unop op = match op with 
        Not -> Datatype(Bool)
        | _ -> raise (E.InvalidUnaryOperation)
    in
    let check_int_unop op = match op with
        Neg -> Datatype(Int) 
        | _ -> raise (E.InvalidUnaryOperation)
    in
    let check_float_unop op = match op with 
        Neg -> Datatype(Float)
        | _ -> raise (E.InvalidUnaryOperation)
    in
    let (se, env) = expr_to_sexpr e env in 
    let typ = sexpr_to_type se in 
    match typ with
        Datatype(Int)        -> SUnop(op, se, check_int_unop op), env
        | Datatype(Float)    -> SUnop(op, se, check_float_unop op), env 
        | Datatype(Bool)     -> SUnop(op, se, check_bool_unop op), env
        | Settype(Void)   -> SIntLit(0, check_set_unop op), env
        | Settype(_)         -> SUnop(op, se, check_set_unop op), env
        | _ -> raise (E.InvalidUnaryOperation)


and check_binop e1 op e2 env = 
    let (se1, env) = expr_to_sexpr e1 env in 
    let (se2, env) = expr_to_sexpr e2 env in
    let typ1 = sexpr_to_type se1 in 
    let typ2 = sexpr_to_type se2 in
    match typ1, typ2 with Settype(t1), Settype(t2) -> 
            check_binop_set se1 op se2 t1 typ1 t2 env
    | _ -> 
    match op with  
    Equal | Neq -> 
        if typ1 = typ2 || typ1 = Datatype(Void) || typ2 = Datatype(Void) then 
                    if typ1 = Datatype(String) 
                    then (SCall("strcomp", [se1;se2], Datatype(Bool)), env) 
                    else SBinop(se1, op, se2, Datatype(Bool)), env 
                   else raise(E.InvalidBinaryOperation)
    | And | Or -> 
            if typ1 = Datatype(Bool) && typ2 = Datatype(Bool) 
            then SBinop(se1, op, se2, Datatype(Bool)), env
            else raise(E.InvalidBinaryOperation)
    | Less | Leq | Greater | Geq -> 
            if typ1 = typ2 && 
            (typ1 = Datatype(Int) || typ1 = Datatype(Float)) 
                then SBinop(se1, op, se2, Datatype(Bool)), env
                else raise(E.InvalidBinaryOperation)
    | Add | Mult | Sub | Div | Mod -> if typ1 = typ2 && 
            (typ1 = Datatype(Int) || typ1 = Datatype(Float))
                then SBinop(se1, op, se2, typ1), env
                else raise(E.InvalidBinaryOperation)
    | Qmark -> 
            let t = (match typ2 with 
                Settype(t) -> Datatype(t) 
                | _ ->  (raise E.InvalidBinaryOperation)) in
            if typ1 = t || t = Datatype(Void) then
                let new_se, env = 
                    match typ1 with 
                        Datatype(Int) -> expr_to_sexpr (Call("foundi", [e1; e2])) env 
                        | Datatype(Float) -> expr_to_sexpr (Call("foundf", [e1; e2])) env 
                        | Datatype(Bool) -> expr_to_sexpr (Call("foundb", [e1; e2])) env 
                        | Datatype(String) ->  expr_to_sexpr (Call("founds", [e1; e2])) env
                        | _ ->  raise E.InvalidBinaryOperation
                        in 
                        new_se, env  
                    else 
                        (raise E.InvalidBinaryOperation)
    | _ ->  raise(E.InvalidBinaryOperation)
 

and check_binop_set se1 op se2 t1 typ1 t2  env =
    if not (t1 = Void || t2 = Void || t1 = t2) then raise E.InvalidBinaryOperation
    else
        (* Determine the type of the binop expr *) 
        let env, se1, se2, return, t1 =  match (t1, t2) with 
        (Void, a) -> 
            let se1, env = change_sexpr_type se1 (Settype(a)) env in env, se1, se2, a, a
        | (a, Void) -> let se2, env = change_sexpr_type se2 (Settype(a)) env in env, se1, se2, a, a
        | (a, _) -> env, se1, se2, a, a
        in
        (* Convert binop to a call *) 
        match t1 with 
        Int -> (match op with 
            Add ->  SCall("append", [se1; se2], Settype(return)), env
            | Sub -> SCall("diffi", [se1; se2], typ1), env 
            | Mult -> SCall("intersecti",[se1; se2], typ1), env
            | Union -> SCall("unioni", [se1; se2], typ1), env
            | _ -> raise E.InvalidBinaryOperation) 
        | Float -> (match op with 
            Add ->  SCall("append", [se1; se2], Settype(return)), env
            | Sub -> SCall("difff", [se1; se2], typ1), env 
            | Mult -> SCall("intersectf",[se1; se2], typ1), env
            | Union -> SCall("unionf", [se1; se2], typ1), env
            | _ -> raise E.InvalidBinaryOperation) 
        | String -> (match op with 
            Add ->  SCall("append", [se1; se2], Settype(return)), env
            | Sub -> SCall("diffs", [se1; se2], typ1), env 
            | Mult -> SCall("intersects",[se1; se2], typ1), env
            | Union -> SCall("unions", [se1; se2], typ1), env
            | _ -> raise E.InvalidBinaryOperation)
        | Bool -> (match op with 
            Add ->  SCall("append", [se1; se2], Settype(return)), env
            | Sub -> SCall("diffb", [se1; se2], typ1), env 
            | Mult -> SCall("intersectb",[se1; se2], typ1), env
            | Union -> SCall("unionb", [se1; se2], typ1), env
            | _ -> raise E.InvalidBinaryOperation)
        | _ -> raise E.InvalidBinaryOperation 
 

(* Check the set built in function *)
and check_call_set e env =
    if List.length e <> 1 then
    raise(E.WrongNumberOfArguments)
    else 
        let (se, env) = expr_to_sexpr (List.hd e) env in 
        let typ = sexpr_to_type se in 
        let new_s, it = match typ with 
        | Settype(Int) -> "seti", Int
        | Settype(Float) -> "setf", Float
        | Settype(String) -> "sets", String
        | Settype(Bool) -> "setb", Bool
        | Settype(Void) -> "seti", Int
        | _ -> (raise (E.IncorrectArgumentType("expected: settype",
                         "found: " ^ string_of_typ typ)))
        in
        (SCall(new_s, [se], Settype(it)), env)

    
(* Check built in calls *)
and check_call_built str e env = 
    let fd = StringMap.find str env.env_fmap in
    if List.length e <> List.length fd.formals then
        raise(E.WrongNumberOfArguments)
    else 
        let (se, env) = expr_list_to_sexpr_list e env in 
        (* Get the return type of the built in *)
        let typ = match str with 
        "str" -> Datatype(String)
        | "read" -> Datatype(String)
        | "open" -> Datatype(Int)
        | "write" -> Datatype(Void)
        | "split" -> Settype(String)
        | "str_to_int" -> Datatype(Int)
        | "close" -> Datatype(Void)
        | _ -> raise E.Invalid in
        (SCall(str, se, typ), env)


and check_print e env = 
    if ((List.length e) <> 1) then raise (E.WrongNumberOfArguments)
    else 
        let (se, env) = expr_to_sexpr (List.hd e) env in 
        let typ = sexpr_to_type se in 
        let new_s = match typ with 
        Datatype(Int) -> "print"
        | Datatype(String) -> "prints"
        | Datatype(Bool) -> "printb"
        | Datatype(Float) -> "printf"
        | Settype(t) -> (match t with 
                | Int -> "printseti"
                | String -> "printsets"
                | Float -> "printsetf"
                | Bool -> "printsetb"
                | _ -> raise E.CannotPrintType)
        | _ -> raise E.CannotPrintType
        in
        (SCall(new_s, [se], Datatype(Int)), env)


and check_call s e_l env =
    (* Check if the function exists *)
    if not (StringMap.mem s env.env_fmap) 
    then (print_string("fname: " ^ s); raise E.FunctionNotDefined)
    else 
        (* Check that main wasn't called *)
        if s = "main" then (raise E.CannotCallMain) 
        else  
            let env_ref = ref (env) in

            (* List of semantically checked arguments  *) 
            let _ = List.rev(List.fold_left 
            (fun l expr -> 
                let (se, env) = expr_to_sexpr expr env in 
                env_ref := env;  se :: l) [] e_l )
            in

            (* List of the types of the arguments  *)
            let arg_type_list = List.rev(List.fold_left 
            (fun l expr -> 
                let (se, _) = expr_to_sexpr expr env in 
                let typ = sexpr_to_type se in typ :: l) [] e_l )
             in
             
            (* Check for correct number of arguments *)
            let fd = StringMap.find s env.env_fmap in
            if List.length e_l <> List.length fd.formals then
            raise(E.WrongNumberOfArguments)
            else
                (* if this function is called by itself *)
                if s = env.env_fname then
                    let expr_list, env = expr_list_to_sexpr_list e_l env in  
                    ((SCall(s, expr_list, env.env_return_type), env))
                else 
                    (* If the function is called by another function 
                     * and the function has already  been called/ already defined(lib), 
                     * make sure types are correct *)
                    if StringMap.mem s env.env_sfmap then 
                        let called_fdecl = StringMap.find s env.env_sfmap in 
                        List.iter2  (fun nt (_, ot) -> if nt <> ot && nt <> Settype(Void)
                        then raise (E.IncorrectArgumentType
                            ("expected: " ^ string_of_typ ot,
                         "found: " ^ string_of_typ nt)) else ())
                        arg_type_list called_fdecl.sformals;
                        let expr_list, env = expr_list_to_sexpr_list e_l env in  
                        (SCall(s, expr_list, 
                            called_fdecl.styp), env)
                    else
                        (* Called for the first time: semantically check *)
                        let expr_list, env = expr_list_to_sexpr_list e_l env in
                        let new_env = convert_fdecl_to_sfdecl s arg_type_list env in
                        let env =  {
                            env_globals = env.env_globals;
                            env_fmap = env.env_fmap;
                            env_fname = env.env_fname;
                            env_return_type = env.env_return_type;
                            env_flocals = env.env_flocals;
                            env_in_while = env.env_in_while;
                            env_set_return = env.env_set_return; 
                            env_sfmap = new_env.env_sfmap;
                         } in
                        let called_fdecl = StringMap.find s env.env_sfmap in
                        (SCall(s, expr_list,
                         called_fdecl.styp), env)



and sexpr_to_type sexpr = match sexpr with  
    SIntLit(_, typ)                  -> typ
  | SFloatLit(_, typ)                -> typ
  | SStrLit(_, typ)                  -> typ
  | SBoolLit(_, typ)                 -> typ
  | SId(_, typ)                      -> typ
  | SBinop(_, _, _, typ)             -> typ
  | SUnop(_, _, typ)                 -> typ
  | SCall(_, _, typ)                 -> typ
  | SSet(_, typ)                     -> typ
  | SSetAccess(_,_,typ)              -> typ
  | SNoexpr                          -> Datatype(Void)

and change_sexpr_type sexpr typ env = match sexpr with 
   SId(s,_)                      -> 
       (let flocals = StringMap.add s typ env.env_flocals in
       let new_env = {
            env_globals = env.env_globals;
            env_fmap = env.env_fmap;
            env_fname = env.env_fname;
            env_return_type = env.env_return_type;
            env_flocals = flocals;
            env_in_while = env.env_in_while;
            env_set_return = env.env_set_return;
            env_sfmap = env.env_sfmap;
        } in
       (SId(s, typ), new_env))
  | _ -> (sexpr, env) 


    
 and check_sblock sl env =
     (* Make sure nothing follows a return *)
     let rec check_block_return sl = match sl with 
         Return _ :: _ :: _ -> raise E.NothingAfterReturn
        | Block sl :: ss -> check_block_return (sl @ ss)   
        | _ :: ss -> check_block_return ss 
        | [] -> ()
     in check_block_return sl;
     
     (* Check all statements within a block *)
     match sl with 
     | [] ->   (SBlock([SExpr(SNoexpr, Datatype(Void))]), env)
     | _ ->  let env_ref = ref(env) in 
        let block, env = (List.rev( List.fold_left 
        (fun l stmt -> 
            let new_stmt, env = convert_stmt_to_sstmt stmt !env_ref 
            in env_ref := env;
            new_stmt :: l) [] sl), !env_ref) 
        in (SBlock(block), env)


and check_expr_stmt e env = 
    let se, env = expr_to_sexpr e env in
    let typ = sexpr_to_type se in
    (SExpr(se, typ), env)


and check_assign s e env =
 
    let se, env = expr_to_sexpr e env in
    let typ = sexpr_to_type se in

    (* If the variable has already been declared, check new type *)
    if StringMap.mem s env.env_flocals || 
    StringMap.mem s env.env_globals then 
        let old_typ = try StringMap.find s env.env_flocals 
        with Not_found -> StringMap.find s env.env_globals in
        (* Check set types *) 
        match old_typ, typ with 
        Settype(t1), Settype(t2) ->
            (* If type mismatch *)
            if not (t1 = Void || t2 = Void || t1 = t2) then (raise (E.AssignmentTypeMismatch
                ("expected set of: " ^ string_of_typ old_typ,
                "found set of : " ^ string_of_typ typ)))
            else
                (* If new type was Void return the old_typ *) 
                if t2 = Void then (SAssign(s, se, old_typ), env) else 
                
                (* Add new settype to map *) 
                let flocals = StringMap.add s typ env.env_flocals in
                let new_env = {
                    env_globals = env.env_globals;
                    env_fmap = env.env_fmap;
                    env_fname = env.env_fname;
                    env_return_type = env.env_return_type;
                    env_flocals = flocals;
                    env_in_while = env.env_in_while;
                    env_set_return = env.env_set_return;
                    env_sfmap = env.env_sfmap;
                } 
                in
                (SAssign(s, se, typ), new_env)

        (* Check all other types *)
        | _ -> 
            if (old_typ <> typ) then (raise 
            (E.AssignmentTypeMismatch("expected: " ^ string_of_typ old_typ,
            "found: " ^ string_of_typ typ)))
        else (SAssign(s, se, typ), env)
    else 
        (* Variable not declared yet, bind that type to the variable *)
        let flocals = StringMap.add s typ env.env_flocals in
        let new_env = {
            env_globals = env.env_globals;
            env_fmap = env.env_fmap;
            env_fname = env.env_fname;
            env_return_type = env.env_return_type;
            env_flocals = flocals;
            env_in_while = env.env_in_while;
            env_set_return = env.env_set_return;
            env_sfmap = env.env_sfmap;
        } 
        in
        (SAssign(s, se, typ), new_env)


and check_return e env = 
    let se, env  = expr_to_sexpr e env in 
    let typ = sexpr_to_type se in 
    
    (* If the return type has been set, check new return type *) 
    if env.env_set_return then 
        if env.env_return_type = typ 
        then (SReturn(se, typ), env) 
        else raise (E.ReturnTypeMismatch 
        ("expected: " ^ string_of_typ env.env_return_type, 
        "found: " ^ string_of_typ typ))
    else 
        (* If no return type yet, make this the return type *)
        let new_env = {
            env_globals = env.env_globals;
            env_fmap = env.env_fmap;
            env_fname = env.env_fname;
            env_return_type = typ;
            env_flocals = env.env_flocals;
            env_in_while = env.env_in_while;
            env_set_return = true;
            env_sfmap = env.env_sfmap;
        }
        in 
        (SReturn(se, typ), new_env)


and check_while e s env =
    (* Create new env for in while *)
    let old_env_in_while = env.env_in_while in
    let env = {
        env_globals = env.env_globals;
        env_fmap = env.env_fmap;
        env_fname = env.env_fname;
        env_return_type = env.env_return_type;
        env_flocals = env.env_flocals;
        env_in_while = true;
        env_set_return = env.env_set_return; 
        env_sfmap = env.env_sfmap;
    }
    in
    
    (* Semantically check predicate and body *)
    let (se, env) = expr_to_sexpr e env in 
    let typ = sexpr_to_type se in 
    let (while_body, env) = convert_stmt_to_sstmt s env in

    (* Revert env variable *)
    let env =  {
        env_globals = env.env_globals;
        env_fmap = env.env_fmap;
        env_fname = env.env_fname;
        env_return_type = env.env_return_type;
        env_flocals = env.env_flocals;
        env_in_while = old_env_in_while;
        env_set_return = env.env_set_return; 
        env_sfmap = env.env_sfmap;
    }
    in

    (* Make sure the predicate is of type Bool *)
    if typ = Datatype(Bool) then
        (SWhile(se, SBlock([while_body])), env)
    else raise (E.InvalidWhileStatementType)



and check_if e s1 s2 env = 
    let (se, env) = expr_to_sexpr e env in 
    let typ = sexpr_to_type se in 
    let (if_body, env) =  convert_stmt_to_sstmt s1 env in
    let (else_body, env) = convert_stmt_to_sstmt s2 env in
    if typ = Datatype(Bool) then
        (SIf(se, SBlock([if_body]), SBlock([else_body])), env)
    else raise (E.InvalidIfStatementType)


and check_break env = 
    if env.env_in_while then
        (SBreak, env)
    else raise E.BreakOutsideOfLoop


(* Semantically check constraints and format them 
 * to be the structure of a for loop: 
     * initialization, predicate, increment *)
and check_constraints cl env =
    
    (* check the bounds of the constraint *) 
    let check_expression e env = 
        let (se, env) = expr_to_sexpr e env in
        let typ = sexpr_to_type se in 
        if typ = Datatype(Int) then (se, typ, env) 
        else raise E.InvalidIterStatementConstraintBounds
    in
    (* add the constraint variable to local variables *)
    let add_iter_var s typ env = 
        let flocals = StringMap.add s typ env.env_flocals in
        {
            env_globals = env.env_globals;
            env_fmap = env.env_fmap;
            env_fname = env.env_fname;
            env_return_type = env.env_return_type;
            env_flocals = flocals;
            env_in_while = env.env_in_while;
            env_set_return = env.env_set_return;
            env_sfmap = env.env_sfmap;
        } 
    (* semantically check/change format of the constraint *)
    in match cl with 
      (e1, Less, s, op, e2) -> 
          let se1, typ1, env = check_expression e1 env in 
          let env = add_iter_var s typ1 env in
          let se2, typ2, env = check_expression e2 env in
          ((SAssign(s, SBinop(se1, Add, 
                SIntLit(1, Datatype(Int)), typ1), typ1), 
          SBinop(SId(s, typ1), op, se2, typ2),
          SAssign(s, SBinop(SId(s, typ1), Add, 
                SIntLit(1, Datatype(Int)), typ1), typ1)), env)
      |(e1, Leq, s, op, e2) -> 
          let se1, typ1, env = check_expression e1 env in 
          let env = add_iter_var s typ1 env in
          let se2, typ2, env = check_expression e2 env in
          ((SAssign(s, se1, typ1), 
          SBinop(SId(s, typ1), op, se2, typ2),
          SAssign(s, SBinop(SId(s, typ1), Add, 
                SIntLit(1, Datatype(Int)), typ1), typ1)), env)
      |(e1, Greater, s, op, e2) -> 
          let se1, typ1, env = check_expression e1 env in 
          let env = add_iter_var s typ1 env in
          let se2, typ2, env = check_expression e2 env in
          ((SAssign(s, SBinop(se1, Add, 
                SIntLit(1, Datatype(Int)), typ1), typ2), 
          SBinop(SId(s, typ1), op, se2, typ2),
          SAssign(s, SBinop(SId(s, typ1), Sub, 
                SIntLit(1, Datatype(Int)), typ1), typ1)), env)
      |(e1, Geq, s, op, e2) -> 
          let se1, typ1, env = check_expression e1 env in 
          let env = add_iter_var s typ1 env in
          let se2, typ2, env = check_expression e2 env in
          ((SAssign(s, se1, typ2), 
          SBinop(SId(s, typ1), op, se2, typ2),
          SAssign(s, SBinop(SId(s, typ1), 
                Sub, SIntLit(1, Datatype(Int)), typ1), typ1)), env)
      |(_, _, _, _, _) -> raise E.InvalidIterStatement


and check_iter cl boole body env =
    (* Create new env in loop *)
    let old_env = env in 
    let env = {
        env_globals = env.env_globals;
        env_fmap = env.env_fmap;
        env_fname = env.env_fname;
        env_return_type = env.env_return_type;
        env_flocals = env.env_flocals;
        env_in_while = true;
        env_set_return = env.env_set_return; 
        env_sfmap = env.env_sfmap;
    } 
    in

    (* Semantically check the constraints *)
    let env_ref = ref(env) in 
    let constraints = List.rev(List.fold_left 
    (fun l c -> 
        let new_c, env = check_constraints c !env_ref 
        in env_ref := env;
        (new_c :: l)) [] cl) 
    in

    (* Semantically check the boolean statement *)
    let (se, env) = expr_to_sexpr boole !env_ref  in 
    env_ref := env;
    let typ = sexpr_to_type se in
    let se, _ = match typ with 
        Datatype(Bool) -> se, typ
        | Datatype(Void) -> SBoolLit(true, Datatype(Bool)), Datatype(Bool)
        | _ -> raise E.InvalidIterStatementBoolExpression
    in

    (* Semantically check the body of the iter stmt *)
    let (iter_body, env) = convert_stmt_to_sstmt body !env_ref in 
    env_ref := env;

    (* Create a nested body for the siter stmt *)
    let se_first_c = List.hd (List.rev constraints) in 
    let body = SIter(se_first_c, SIf(se, iter_body, SBlock([]))) in 
    let body_ref = ref(body) 
    in
    env_ref := {
        env_globals = env.env_globals;
        env_fmap = env.env_fmap;
        env_fname = env.env_fname;
        env_return_type = env.env_return_type;
        env_flocals = env.env_flocals;
        env_in_while = old_env.env_in_while;
        env_set_return = env.env_set_return; 
        env_sfmap = env.env_sfmap;
    };
    let helper cl = match cl with  
        [] -> raise E.InvalidIterStatementConstraintBounds
        | _ :: tl ->
             (List.iter (fun cnstrnt -> 
                 let body = SIter(cnstrnt, SBlock([!body_ref])) in 
                 body_ref := body;) tl)
    in helper (List.rev constraints); (!body_ref, !env_ref)


and check_set_assign s e1 e2 env = 
    (* Check if the variables has been created *)
    let _ = check_variable_access s env in 
    let s_typ = try StringMap.find s env.env_flocals with 
    Not_found -> StringMap.find s env.env_globals in 

    (* Make sure it is of Settype *)
    let s_e_typ = match s_typ with Settype(a) -> a 
        | _ -> raise E.InvalidSetAssign in
    let se1, env = expr_to_sexpr e1 env in 
    let se2, env = expr_to_sexpr e2 env in
    let typ1 = sexpr_to_type se1 in
    let typ2 = sexpr_to_type se2 in 
    
    (* Make sure the access variable is an int and the 
     * element being assigned is of the set element type *)
    if typ1 <> Datatype(Int) then raise E.InvalidSetAssign
    else if typ2 <> Datatype(s_e_typ) then raise E.InvalidSetAssign
    else SSetElementAssign(s, se1, se2, typ2), env


 
(* Semantically checks a statement *)
and convert_stmt_to_sstmt stmt env = match stmt with 
   Block sl             -> check_sblock sl env
  | Expr e              -> check_expr_stmt e env
  | Assign (s, e)       -> check_assign s e env
  | Return e            -> check_return e env
  | If(e, s1, s2)       -> check_if e s1 s2 env
  | While(e, s)         -> check_while e s env
  | Break               -> check_break env
  | Iter(cl, e, s)      -> check_iter cl e s env 
  | SetElementAssign(s, e1, e2) -> check_set_assign s e1 e2 env
  

(* Converts each fdecl to sfdecl--
 * fname: function name
 * arg_type_list: types of the parameters
 * env: env variable *)
and convert_fdecl_to_sfdecl fname arg_type_list env =
   
    let fdecl = StringMap.find fname env.env_fmap in

    (* Make the parameters and their types local variables *)
    let flocals = List.fold_left2 (fun m n t -> StringMap.add n t m) 
        StringMap.empty fdecl.formals arg_type_list
    in

    (* Starting env variable *) 
    let env =  {
        env_globals = env.env_globals;
        env_fmap = env.env_fmap; 
        env_fname = fdecl.fname; 
        env_return_type = Ast.Datatype(Int); (* placeholder until set *)
        env_flocals = flocals; 
        env_in_while = false;
        env_set_return = false;
        env_sfmap = env.env_sfmap; 
    } 
    in 

    (* Semantically check all statements of the body *)
    let (sstmts, env) = convert_stmt_to_sstmt (Block fdecl.body) env
    in

    (* Create semantically checked formals for fname *)
    (* if a void type formal changed, change its type *)
    report_duplicate(fdecl.formals);
    let sformals = List.rev(List.fold_left2 
        (fun l name typ -> try (name, StringMap.find name env.env_flocals) :: l with Not_found-> (name, typ) :: l) 
        [] fdecl.formals arg_type_list)
    in


    (* Create semantically checked locals for fname *)
    let formals_map = List.fold_left (fun m (n, t) -> StringMap.add n t m) 
    StringMap.empty sformals
    in
    let locals_map = List.fold_left (fun m (n,t)  -> 
        match t with Settype(Void) -> m | _ -> 
        if StringMap.mem n formals_map then StringMap.remove n m
        else m) env.env_flocals (StringMap.bindings env.env_flocals)
    in
    let locals = StringMap.bindings locals_map
    in
    let locals = List.rev(List.fold_left (fun l (n, t) -> (n, t) :: l) [] locals) in
    
    (* Create a semantically checked main function *)
    let sfdecl = {
        styp =  env.env_return_type;
        sfname = fdecl.fname;
        slocals = locals;
        sformals = sformals;
        sbody = match sstmts with SBlock(sl) ->  sl | _ -> [] ;
    }
    in

    (* Return the env variable with this information *)
    let env =  {
        env_globals = env.env_globals;
        env_fmap = env.env_fmap;
        env_fname = env.env_fname;
        env_return_type = env.env_return_type; 
        env_flocals = env.env_flocals; 
        env_in_while = env.env_in_while;
        env_set_return = env.env_set_return;
        env_sfmap = StringMap.add fname sfdecl env.env_sfmap; 
    } 
    in env


(* Reports duplicate variables *)
and report_duplicate list = 
    let rec helper = function
        n1 :: n2 :: _ when n1 = n2 -> raise (E.DuplicateVariable n1)
    | _ :: t -> helper t 
    | [] -> () 
    in helper (List.sort compare list)

(* Converts lib to semantically checked lib *) 
and convert_lib_fdecl_to_sfdecl env =
   
    (* Library functions *) 
    let built_in_decls =
        (StringMap.add "printb" [Datatype(Bool)]
        (StringMap.add "printsetf" [Settype(Float)]
        (StringMap.add "printsetb" [Settype(Bool)]
        (StringMap.add "printseti" [Settype(Int)]
        (StringMap.add "printsets" [Settype(String)]
        (StringMap.add "sliceb" [Settype(Bool); Datatype(Int); Datatype(Int)]
        (StringMap.add "slices" [Settype(String); Datatype(Int); Datatype(Int)]
        (StringMap.add "slicef" [Settype(Float); Datatype(Int); Datatype(Int)]
        (StringMap.add "slicei" [Settype(Int); Datatype(Int); Datatype(Int)]
        (StringMap.add "strcomp" [Datatype(String); Datatype(String)]
        (StringMap.add "seti" [Settype(Int)]
        (StringMap.add "setf" [Settype(Float)]
        (StringMap.add "sets" [Settype(String)]
        (StringMap.add "setb" [Settype(Bool)]
        (StringMap.add "diffi" [Settype(Int); Settype(Int)]
        (StringMap.add "difff" [Settype(Float); Settype(Float)]
        (StringMap.add "diffs" [Settype(String); Settype(String)] 
        (StringMap.add "diffb" [Settype(Bool); Settype(Bool)]
        (StringMap.add "unioni" [Settype(Int); Settype(Int)] 
        (StringMap.add "unionf" [Settype(Float); Settype(Float)] 
        (StringMap.add "unions" [Settype(String); Settype(String)] 
        (StringMap.add "unionb" [Settype(Bool); Settype(Bool)]
        (StringMap.add "intersecti" [Settype(Int); Settype(Int)]
        (StringMap.add "intersectf" [Settype(Float); Settype(Float)]
        (StringMap.add "intersects" [Settype(String); Settype(String)]
        (StringMap.add "intersectb" [Settype(Bool); Settype(Bool)]
        (StringMap.add "founds" [Datatype(String); Settype(String)]
        (StringMap.add "foundi" [Datatype(Int); Settype(Int)]
        (StringMap.add "foundf" [Datatype(Float); Settype(Float)]
        (StringMap.add "foundb" [Datatype(Bool); Settype(Bool)]
        (StringMap.add "appends" [Settype(String); Settype(String); Settype(String); Datatype(Int); Datatype(Int)]
        (StringMap.add "appendf" [Settype(Float); Settype(Float); Settype(Float); Datatype(Int); Datatype(Int)]
        (StringMap.add "appendb" [Settype(Bool); Settype(Bool); Settype(Bool); Datatype(Int); Datatype(Int)]
        (StringMap.singleton "appendi" [Settype(Int); Settype(Int); Settype(Int); Datatype(Int); Datatype(Int)]
        ))))))))))))))))))))))))))))))))))
    in
    
    (* Update env variable through each function *)
    let env_ref = ref(env) in 
    let _ = List.iter (fun (s, l) -> 
        let env = convert_fdecl_to_sfdecl s l !env_ref in env_ref := env) 
    (StringMap.bindings built_in_decls)
    in !env_ref
   

(* Convert an ast tree to an sast tree *)
let convert_ast_to_sast globals functions fmap  =

    (* Verify main function exists *)
    let _ = try StringMap.find "main" fmap 
    with Not_found -> raise (E.MissingMainFunction) in
    
    (* temp env var to check globals *)
    let env =  { 
        env_globals = StringMap.empty;
        env_fmap = fmap;
        env_fname = "main";
        env_return_type = Ast.Datatype(Int);
        env_flocals = StringMap.empty;
        env_in_while = false;
        env_set_return = false; 
        env_sfmap = StringMap.empty;
    } 
    in
    (* Semantically check globals *)
    (* TODO: check if globals are reassigned to an incompatible type *)
    let env_ref = ref(env) in 
    let sglobals = List.rev (List.fold_left 
        (fun l (n, e) -> 
            let se, env = expr_to_sexpr e !env_ref 
            in let typ = sexpr_to_type se 
            in env_ref := env; 
            (n, se, typ) :: l) [] globals)
    in

    (* Create a map of the globals and add to env *)
    let globals_map = List.fold_left (fun m (n, _, t) -> 
        StringMap.add n t m) StringMap.empty sglobals
    in
    
    (* Create env variable to start the semant check *) 
    let env =  { 
        env_globals = globals_map;
        env_fmap = fmap;
        env_fname = "main";
        env_return_type = Ast.Datatype(Void);
        env_flocals = StringMap.empty;
        env_in_while = false;
        env_set_return = false; 
        env_sfmap = StringMap.empty;
    } 
    in

    (* Check that they are no duplicate functions *)
    report_duplicate (List.map (fun fd -> fd.fname) functions); 

     (* Convert all library functions *) 
    let env = convert_lib_fdecl_to_sfdecl env in

    (* Convert fdecls to sfdecls through main *)
    let env = convert_fdecl_to_sfdecl "main" [] env in
    
    (* Convert all library functions *) 
    (*let env = convert_lib_fdecl_to_sfdecl env in*)

    (* Return all semantically checked functions and globals *)
    let sfdecls = List.rev(List.fold_left (fun l (_, sfd) -> sfd :: l) 
    [] (StringMap.bindings env.env_sfmap))
    in (sglobals, sfdecls) 


(* Add all functions declared (besides lib functions) to a function map *)
let build_fdecl_map functions =
  
   (* Reserved functions *)
   let built_in_decls =  StringMap.add "print"
   { fname = "print"; formals = [("x")];
   body =  []   } (StringMap.add "append"
   { fname = "append"; formals = ["x"; "y"];
   body =  []; }( StringMap.add "strcmp"
   { fname = "strcmp"; formals = ["x"; "y"];
   body = [] } (StringMap.add "set"
   { fname = "set"; formals = ["x"];
   body = [] }(StringMap.add "open"
   { fname = "open"; formals = ["x"; "y"];
   body = []; }  (StringMap.add "str"
   { fname = "str"; formals = ["x"];
   body = []; } (StringMap.add "pop"
   { fname = "pop"; formals = ["x"];
   body = []; } (StringMap.add "read"
   { fname = "read"; formals = ["x"];
   body = []; } (StringMap.add "write"
   { fname = "write"; formals = ["x"; "y"];
   body = []; } (StringMap.add "split"
   { fname = "split"; formals = ["x"; "y"]; 
   body = []; } (StringMap.add "str_to_int" 
   { fname = "str_to_int"; formals = ["x"];
   body = []; } (StringMap.add "close" 
   { fname = "close"; formals = ["x"]; 
   body = []; } (StringMap.add "write" 
   { fname = "write"; formals = ["x"; "y"];
   body = []; }  


     (StringMap.singleton "prints"
     { fname = "print"; formals = [("x")];
       body =  []  }) ))))))))))))
  in 
  
  (* Make sure none of the functions have a reserved name 
   * Note: lib functions are appended at end of file; 
   * error will come up as a duplicate function name *)
  let check_functions m fdecl = 
      if StringMap.mem fdecl.fname m then 
          raise (E.DuplicateFunctionName fdecl.fname)
      else if StringMap.mem fdecl.fname built_in_decls then 
          raise (E.FunctionNameReserved fdecl.fname)
      else StringMap.add fdecl.fname fdecl m
  in 
   List.fold_left (fun m fdecl -> check_functions m fdecl) 
  built_in_decls functions 


let check (globals, functions) =
    let fmap  = build_fdecl_map functions in
    let sast = convert_ast_to_sast globals functions fmap in sast 

