{ open Parser }

let whitespace = [' ' '\t' '\r' '\n']
let digit = ['0'-'9']
let alpha = ['a'-'z' 'A'-'Z']
let ascii = [' '-'!' '#'-'[' ']'-'~']
let float = (digit+) ['.'] digit+
let int = digit+
let string = '"' ((ascii)* as s) '"' 
let id = alpha (alpha | digit | '_')*


rule token = parse
    whitespace { token lexbuf } (* Whitespace *)
| "/*"     { comment lexbuf }   (* Comments *)
| '('      { LPAREN } 
| ')'      { RPAREN } 
| '{'      { LBRACE }
| '}'      { RBRACE }
| '['      { LBRACK }
| ']'      { RBRACK }
| ';'      { SEMI }   
| ','      { COMMA }
| ':'      { COLON }
| '|'      { STHAT }

(* Operators *)
| '+'      { PLUS }   
| '-'      { MINUS }  
| '*'      { TIMES }  
| '/'      { DIVIDE } 
| '%'      { MOD }

| '#'      { HASH } 
| '?'      { QMARK }
| '&'      { UNION }

(* Equality Operators *)
| '='      { ASSIGN }  
| "=="     { EQ }      
| "!="     { NEQ }
| '<'      { LT }      
| "<="     { LEQ }     
| ">"      { GT }      
| ">="     { GEQ }     
| "&&"     { AND }     
| "||"     { OR }      
| "!"      { NOT }

(* Control *)
| "if"     { IF }
| "else"   { ELSE }
| "while"  { WHILE }
| "return" { RETURN }
| "break"  { BREAK }

(* Keywords *)
| "def"    { DEF }
| "true"   { TRUE }
| "false"  { FALSE }

| int as lxm { INTLIT(int_of_string lxm) }
| float as lxm { FLOATLIT(float_of_string lxm) }
| string { STRLIT(s) }
| id as lxm { ID(lxm) }
| eof { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and comment = parse
  "*/" { token lexbuf }
| _ { comment lexbuf }
