open Ast

type sexpr = 
        SIntLit of int * datatype
        | SFloatLit of float * datatype
        | SStrLit of string * datatype
        | SBoolLit of bool * datatype
        | SSet of sexpr list *  datatype
        | SSetAccess of string * sexpr * datatype
        | SId of string * datatype
        | SUnop of unop * sexpr * datatype
        | SBinop of sexpr * op * sexpr * datatype 
        | SCall of string * sexpr list * datatype 
        | SNoexpr

   
and sstmt = 
        SBlock of sstmt list
        | SExpr of sexpr * datatype     
        | SIf of sexpr * sstmt * sstmt
        | SWhile of sexpr * sstmt
        | SAssign of string * sexpr * datatype
        | SSetElementAssign of string * sexpr * sexpr * datatype
        | SIter of sconstraints *  sstmt 
        | SReturn of sexpr * datatype
        | SBreak

and sconstraints = sstmt * sexpr * sstmt 


type sfdecl = {
    styp : datatype;
    sfname : string;
    slocals : (string * datatype) list;
    sformals : (string * datatype) list;
    sbody : sstmt list; 
}

type sglobal = string * sexpr * datatype 

type sprogram = sglobal list * fdecl list

let sstring_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Mod -> "%"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"
  | Carat -> "^"
  | Qmark -> "?"
  | Intersect -> "&"
  | Union -> "|"

let sstring_of_uop = function
    Neg -> "-"
  | Not -> "!"
  | Card -> "#" 

let rec sstring_of_typ = function
    Datatype(Int)       -> "int"
  | Datatype(String)    -> "string"
  | Datatype(Bool)      -> "bool" 
  | Datatype(Void)      -> "void"
  | Datatype(Float)     -> "float"
  | Settype(s)          -> "set(" ^ sstring_of_typ (Datatype(s)) ^ ")"


let rec sstring_of_expr = function
    SIntLit(l,_t)           -> string_of_int l
  | SFloatLit(l, _)         -> string_of_float l
  | SStrLit(s, _)           -> s
  | SBoolLit(true, _)       -> "true"
  | SBoolLit(false, _)      -> "false"
  | SId(s, _)               -> s
  | SBinop(e1, o, e2, _)    ->
      sstring_of_expr e1 ^ " " ^ sstring_of_op o ^ " " ^ sstring_of_expr e2
  | SUnop(o, e, _)          -> sstring_of_uop o ^ sstring_of_expr e
  | SCall(f, el, _)         -> f ^ "(" ^ String.concat ", " (List.map sstring_of_expr el) ^ ")"
  | SNoexpr                 -> "noexpr"
  | SSet(l, t)              -> "[" ^ String.concat " " (List.map sstring_of_expr l) ^ "]" ^ sstring_of_typ t
  | SSetAccess(s,e,_)       -> s ^ "[" ^ sstring_of_expr e ^ "]" 



let rec sstring_of_stmt = function
    SBlock(stmts)                   -> "{\n" ^ String.concat "" (List.map sstring_of_stmt stmts) ^ "}\n"
  | SExpr(expr, _)                  ->  sstring_of_expr expr ^ ";\n";
  | SReturn(expr, _)                -> "return " ^ sstring_of_expr expr ^ ";\n";
  | SIf(e, s, SBlock([]))           -> "if (" ^ sstring_of_expr e ^ ")\n" ^ sstring_of_stmt s
  | SIf(e, s1, s2)                  ->  "if (" ^ sstring_of_expr e ^ ")\n" ^
                                         sstring_of_stmt s1 ^ "else\n" ^ sstring_of_stmt s2
  | SWhile(e, s)                    -> "while (" ^ sstring_of_expr e ^ ") " ^ sstring_of_stmt s
  | SAssign(v, e, _)                -> v ^ " = " ^ sstring_of_expr e ^  ";\n" 
  | SIter(cl, body)                 -> "(" ^ sstring_of_constraints cl ^ ") " ^ sstring_of_stmt body
  | SBreak                          -> "break;\n"
  | SSetElementAssign(s, e1, e2, _) -> s ^ "[" ^ sstring_of_expr e1 ^ "] = " ^ sstring_of_expr e2 ^ ";\n"


and sstring_of_constraints (s1, e2, s3) = 
    sstring_of_stmt s1 ^ ", " ^ sstring_of_expr e2 ^ ", " ^ sstring_of_stmt s3 


let sstring_of_vinit (s, e, _) = s ^ " = " ^ sstring_of_expr e ^ ";\n"


let sstring_of_fdecl fdecl =
  sstring_of_typ fdecl.styp ^ " " ^
  fdecl.sfname ^ "(" ^ String.concat ", " (List.map fst fdecl.sformals) ^
  ") {\n" ^ 
  String.concat "" (List.map sstring_of_stmt fdecl.sbody) ^ "}" (*^
  "}\n Locals:\n" ^ String.concat " " (List.map (fun (f, s) -> "name: " ^ f ^ " type: " ^ sstring_of_typ s ^"\n")  fdecl.slocals) ^    "\nParams:\n" ^ String.concat " " (List.map (fun (f, s) -> "name: " ^ f ^ " type: " ^ sstring_of_typ s ^"\n")  fdecl.sformals) *)


let sstring_of_program (vars, funcs) =
  String.concat "" (List.map sstring_of_vinit vars) ^ "\n" ^
  String.concat "\n" (List.map sstring_of_fdecl funcs)
