type op = Add | Sub | Mult | Div | Mod |  Equal | Neq 
        | Less | Leq | Greater | Geq | And | Or | Carat | Qmark | Intersect | Union 
type unop = Neg | Not | Card 
type typ = Int | Void | String | Float | Bool 
type datatype = Settype of typ | Datatype of typ 



type expr = 
        IntLit of int 
        | FloatLit of float
        | StrLit of string
        | BoolLit of bool
        | Set of expr list
        | SetAccess of string * expr 
        | Id of string 
        | Unop of unop * expr 
        | Binop of expr * op * expr 
        | Call of string * expr list
        | Slice of string * expr * expr 
        | Noexpr 
       

and constraints = expr * op * string * op * expr 

and stmt = 
        Block of stmt list
        | Expr of expr         
        | If of expr * stmt * stmt
        | While of expr * stmt
        | Assign of string * expr
        | SetElementAssign of string * expr * expr 
        | Iter of constraints list * expr * stmt
        | Return of expr
        | Break


type fdecl =  {
    fname : string;
    formals : string list; 
    body : stmt list; 
}


type global =  string * expr 

type program = global list * fdecl list


(* Pretty-printing functions (from MicroC) *)

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Mod -> "%"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"
  | Carat -> "^"
  | Qmark -> "?"
  | Intersect -> "&"
  | Union -> "|"

let string_of_uop = function
    Neg -> "-"
  | Not -> "!"
  | Card -> "#" 

  
let rec string_of_expr = function
    IntLit(l)               -> string_of_int l
  | FloatLit(l)             -> string_of_float l
  | StrLit(s)               -> s
  | BoolLit(true)           -> "true"
  | BoolLit(false)          -> "false"
  | Id(s)                   -> s
  | Binop(e1, o, e2)        -> string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  | Unop(o, e)              -> string_of_uop o ^ string_of_expr e
  | Call(f, el)             -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr                  -> "noexpr"
  | Set(l)                  -> "[" ^ String.concat " " (List.map string_of_expr l) ^ "]"
  | SetAccess(s,e)          -> s ^ "[" ^ string_of_expr e ^ "]"
  | Slice(s, e1, e2)        -> s ^ "[" ^ string_of_expr e1 ^ ":" ^ string_of_expr e2 ^ "]"


let string_of_constraints (e1, op1, s, op2, e2) = 
     string_of_expr e1 ^ string_of_op op1 ^ 
     s ^ string_of_op op2 ^ string_of_expr e2

let rec string_of_stmt = function
    Block(stmts)                -> "Block{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr)                  -> string_of_expr expr ^ ";\n";
  | Return(expr)                -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([]))         -> "if (" ^ string_of_expr e ^ ")\n" ^ 
        string_of_stmt s
  | If(e, s1, s2)               ->  "if (" ^ string_of_expr e ^ ")\n" ^
                                    string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | While(e, s)                 -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s
  | Assign(st, e)               -> st ^ " = " ^ string_of_expr e ^ ";\n" 
  | Iter(cl, e, s)              -> "(" ^ String.concat "" (List.map string_of_constraints cl) ^ "| " 
                                    ^ string_of_expr e  ^ ")" ^ string_of_stmt s
  | SetElementAssign(s, e1, e2) -> s ^ "[" ^ string_of_expr e1 ^ "] = " ^ string_of_expr e2 ^ ";\n"
  | Break                       -> "break\n;" 
 
 

let rec string_of_typ = function
    Datatype(Int) -> "int"
  | Datatype(String) -> "string"
  | Datatype(Bool) -> "bool" 
  | Datatype(Void) -> "void"
  | Datatype(Float) -> "float"
  | Settype(s) -> string_of_typ (Datatype(s)) 


let string_of_vinit (s, e) = s ^ "=" ^ string_of_expr e ^ ";\n"


let string_of_fdecl fdecl =
  "def " ^ fdecl.fname ^ "(" ^ String.concat "," (List.map (fun x -> x) fdecl.formals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"


let string_of_program (vars, funcs) = 
  String.concat "" (List.map string_of_vinit vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl funcs)
