#include "job_list.h"
#include <stdio.h>

static struct job *job_lookup(struct job *beg, struct job *end, int jid)
{
    //printf("search with beg=%p, end=%p, jid=%d\n", beg, end, jid);
    struct job *mid = beg + (end-beg)/2;
    if ((mid == end) || (mid->jid > jid && mid == beg))
        return NULL;
    if (mid->jid > jid)
        return job_lookup(beg, mid, jid);
    if (mid->jid < jid)
        return job_lookup(mid +1, end, jid);
    return mid;
}

static void job_list_resize(struct job_list *jl, int dif) /* dif = +/-1 */
{
    int oldlen = jl->len;
    jl->len += dif;
    if ((!(jl->len & (jl->len - 1)) && (dif < 0)) ||
            (!((jl->len - 1) & (jl->len - 2)) && (dif > 0))) {
        struct job *old_start = jl->jobs;
        int newsize;
        if ((jl->len < 2) || (dif < 0))
            newsize = jl->len * sizeof (struct job);
        else
            newsize = (jl->len - 1) * 2 * sizeof(struct job);
        jl->jobs = realloc(jl->jobs, newsize);
        for (int i = 0; i < ((dif > 0) ? oldlen : jl->len); i++) {
            if (jl->jobs[i].prev)
                jl->jobs[i].prev = (struct job *) (((uint8_t *) jl->jobs[i].prev) +
                    (((uint8_t *) jl->jobs) - ((uint8_t *) old_start)));
            if (jl->jobs[i].next)
                jl->jobs[i].next = (struct job *) (((uint8_t *) jl->jobs[i].next) +
                    (((uint8_t *) jl->jobs) - ((uint8_t *) old_start)));
        }
        if (jl->first)
            jl->first = (struct job *) (((uint8_t *) jl->first) +
                    (((uint8_t *) jl->jobs) - ((uint8_t *) old_start)));
        if (jl->last)
            jl->last = (struct job *) (((uint8_t *) jl->last) +
                    (((uint8_t *) jl->jobs) - ((uint8_t *) old_start)));
        if (dif > 0)
            memset(jl->jobs + oldlen, 0, dif * sizeof(struct job));
    }
}

int job_add(struct job_list *jl, int ord, uint8_t *data, uint32_t len,
        int sock, int restart)
{
    pthread_mutex_lock(&jl->lock);
    job_list_resize(jl, +1);
    jl->jobs[jl->len - 1] = (struct job){ .jid = jl->next_jid++, .ord = ord,
        .len = len, .data = data, .sock = sock, .attempts = restart };
    int jid = jl->jobs[jl->len - 1].jid;
    //printf("data at %p\n", jl->jobs[jl->len - 1].data);
    pthread_mutex_unlock(&jl->lock);
    return jid;
}

void job_enqueue(struct job_list *jl, int jid)
{
    pthread_mutex_lock(&jl->lock);
    struct job *j = job_lookup(jl->jobs, jl->jobs + jl->len, jid);
    //printf("enqueue job %d at %p\n", jid, j);
    //printf("current first is %p\n", jl->first);
    if (j != NULL) {
        j->prev = jl->last;
        if (j->prev)
            j->prev->next = j;
        if (jl->first == NULL)
            jl->first = j;
        jl->last = j;
        j->next = NULL;
    }
    //printf("new first is %p\n", jl->first);
    //printf("job's prev is %p\n", j->prev);
    //printf("job's next is %p\n", j->next);
    pthread_mutex_unlock(&jl->lock);
}

static void job_enqueue_front_nolock(struct job_list *jl, int jid)
{
    struct job *j = job_lookup(jl->jobs, jl->jobs + jl->len, jid);
    if (j != NULL) {
        j->next = jl->first;
        if (j->next)
            j->next->prev = j;
        if (jl->last == NULL)
            jl->last = j;
        jl->first = j;
        j->prev = NULL;
    }
}

void job_enqueue_front(struct job_list *jl, int jid)
{
    pthread_mutex_lock(&jl->lock);
    job_enqueue_front_nolock(jl, jid);
    pthread_mutex_unlock(&jl->lock);
}

int job_dequeue(struct job_list *jl, struct pak_header *hdr, uint8_t **data,
        int sock)
{
    pthread_mutex_lock(&jl->lock);
    struct job *j = jl->first;
    //printf("dequeueing job at %p\n", j);
    if (j != NULL && ((j->sock == 0) || (j->sock == sock))) {
        jl->first = j->next;
        if (jl->first)
            jl->first->prev = NULL;
        j->prev = NULL;
        j->next = NULL;
        hdr->ord = j->ord;
        hdr->jid = j->jid;
        *data = j->data;
        hdr->len = j->len;
    } else {
        hdr->ord = 0;
        *data = NULL;
        hdr->len = 0;
        hdr->jid = 0;
    }
    pthread_mutex_unlock(&jl->lock);
    return -1 * (hdr->ord == 0);
}

void job_set_socket(struct job_list *jl, int jid, int sock)
{
    pthread_mutex_lock(&jl->lock);
    struct job *j = job_lookup(jl->jobs, jl->jobs + jl->len, jid);
    if (j != NULL)
        j->sock = sock;
    pthread_mutex_unlock(&jl->lock);
}

void job_socket_failed(struct job_list *jl, int sock)
{
    pthread_mutex_lock(&jl->lock);
    for (struct job *j = jl->jobs + jl->len - 1; j != jl->jobs - 1; j--)
        if (j->sock == sock) {
            if (j->attempts-- > 0) {
                j->sock = JOB_UNASSIGNED;
                job_enqueue_front_nolock(jl, j->jid);
            } else
                j->sock = JOB_FAILED;
        }
    pthread_mutex_unlock(&jl->lock);
}

int job_status(struct job_list *jl, int jid)
{
    pthread_mutex_lock(&jl->lock);
    struct job *j = job_lookup(jl->jobs, jl->jobs + jl->len, jid);
    int stat = (j == NULL) ? JOB_LOST : j->sock;
    pthread_mutex_unlock(&jl->lock);
    return stat;
}

void job_add_data(struct job_list *jl, int jid, uint8_t *data, uint32_t len)
{
    job_status(jl, jid);
    pthread_mutex_lock(&jl->lock);
    struct job *j = job_lookup(jl->jobs, jl->jobs + jl->len, jid);
    if (j != NULL) {
        free(j->data);
        j->len = len;
        j->data = data;
        j->sock = JOB_FINISHED;
    } else if (data)
        free(data);
    pthread_mutex_unlock(&jl->lock);
}

int job_remove(struct job_list *jl, int jid, uint8_t **data, uint32_t *len)
{
    pthread_mutex_lock(&jl->lock);
    struct job *j = job_lookup(jl->jobs, jl->jobs + jl->len, jid);
    //printf("found job %d at %p\n", jid, j);
    //printf("data at %p\n", j->data);
    int stat = (j == NULL) ? JOB_LOST : j->sock;
    if ((data != NULL) && (len != NULL)) {
        if (j != NULL && j->sock == JOB_FINISHED) {
            *data = j->data;
            *len = j->len;
        } else {
            if (j->data)
                free(j->data);
            *data = NULL;
            *len = 0;
        }
    }
    if (j != NULL) {
        int off = j - jl->jobs;
        if (jl->first == j)
            jl->first = j->next;
        if (jl->last == j)
            jl->last = j->prev;
        if (j->prev)
            j->prev->next = j->next;
        if (j->next)
            j->next->prev = j->prev;
        memmove(j, j + 1, (jl->len - off - 1) * sizeof(struct job));
        job_list_resize(jl, -1);
        for (int i = 0; i < jl->len; i++) {
            if ((jl->jobs[i].next) && (jl->jobs[i].next > jl->jobs + off))
                jl->jobs[i].next--;
            if (jl->jobs[i].prev && (jl->jobs[i].prev > jl->jobs + off))
                jl->jobs[i].prev--;
        }
        if (jl->first > jl->jobs + off)
            jl->first--;
        if (jl->last > jl->jobs + off)
            jl->last--;

    }
    pthread_mutex_unlock(&jl->lock);
    return stat;
}

void job_list_destroy(struct job_list *jl)
{
    for (struct job *j = jl->jobs; j != jl->jobs + jl->len; j++)
        if (j->data)
            free(j->data);
    free(jl->jobs);
    pthread_mutex_destroy(&jl->lock);
    *jl = (struct job_list) {0};
}
