#!/bin/sh

# Regression testing script for M/s
# Step through a list of files
#  Compile, run, and check the output of each expected-to-work test
#  Compile and check the error of each expected-to-fail test

# Path to the LLVM interpreter
LLI="lli"
#LLI="/usr/local/opt/llvm/bin/lli"

# Path to the ms compiler.  Usually "./mscompile.native"
# Try "_build/mscompile.native" if ocamlbuild was unable to create a symbolic link.
MSCOMPILE="./mscompile.native"
#MSCOMPILE="_build/mscompile.native"

# Set a port number to run
PORT="8888"

# Set time limit for all operations
ulimit -t 30

globallog=testall.log
rm -f $globallog
error=0
globalerror=0

keep=0

Usage() {
    echo "Usage: testall.sh [options] [.ms files]"
    echo "-k    Keep intermediate files"
    echo "-h    Print this help"
    exit 1
}

SignalError() {
    if [ $error -eq 0 ] ; then
	echo "FAILED"
	error=1
    fi
    echo "  $1"
}

# Compare <outfile> <reffile> <difffile>
# Compares the outfile with reffile.  Differences, if any, written to difffile
Compare() {
    generatedfiles="$generatedfiles $3"
    echo diff -b $1 $2 ">" $3 1>&2
    diff -b "$1" "$2" > "$3" 2>&1 || {
	SignalError "$1 differs"
	echo "FAILED $1 differs from $2" 1>&2
    cat "$3"
    }
}

# Run <args>
# Report the command, run it, and report any errors
Run() {
    echo $* 1>&2
    eval $* || {
	SignalError "$1 failed on $*"
	return 1
    }
}

# RunFail <args>
# Report the command, run it, and expect an error
RunFail() {
    echo $* 1>&2
    eval $* && {
	SignalError "failed: $* did not report an error"
	return 1
    }
    return 0
}

Check() {
    error=0
    basename=`echo $1 | sed 's/.*\\///
                             s/.ms//'`
    reffile=`echo $1 | sed 's/.ms$//'`
    basedir="`echo $1 | sed 's/\/[^\/]*$//'`/."

    echo -n "$basename..."

    echo 1>&2
    echo "###### Testing $basename" 1>&2

    generatedfiles=""

    # Note: libmaster.a and libslave.a must be present in the compilers folder already
    generatedfiles="$generatedfiles ${basename}.ll ${basename}.out" &&
    Run "$MSCOMPILE" "<" $1 ">" "${basename}.ll" &&
    Run "llc ${basename}.ll" && 
    Run "gcc -L. ${basename}.s -lmsmaster -pthread -lm -o ${basename}-master" &&
    Run "gcc -L. ${basename}.s -lmsslave -pthread -lm -o ${basename}-slave" &&
    # Change port number if tests are freezing
    Run "./${basename}-master $PORT > ${basename}.out" &&
    Compare ${basename}.out ${reffile}.out ${basename}.diff

    # Report the status and clean up the generated files

    if [ $error -eq 0 ] ; then
    if [ $keep -eq 0 ] ; then
        rm -f $generatedfiles
    fi
    echo "OK"
    echo "###### SUCCESS" 1>&2
    else
    echo "###### FAILED" 1>&2
    globalerror=$error
    fi
}

CheckRemote() {
    error=0
    basename=`echo $1 | sed 's/.*\\///
                             s/.ms//'`
    reffile=`echo $1 | sed 's/.ms$//'`
    basedir="`echo $1 | sed 's/\/[^\/]*$//'`/."

    echo -n "$basename..."

    echo 1>&2
    echo "###### Testing $basename" 1>&2

    generatedfiles=""

    # Note: libmaster.a and libslave.a must be present in the compilers folder already
    generatedfiles="$generatedfiles ${basename}.ll ${basename}.out" &&
    Run "$MSCOMPILE" "<" $1 ">" "${basename}.ll" &&
    Run "llc ${basename}.ll" && 
    Run "gcc -L. ${basename}.s -lmsmaster -pthread -lm -o ${basename}-master" &&
    Run "gcc -L. ${basename}.s -lmsslave -pthread -lm -o ${basename}-slave" &&
    # Change port number if tests are freezing
    Run './${basename}-master $PORT > ${basename}.out & PID=$! ; while [ -z "`netstat -an | grep $PORT`" ] ; do : ; done ; ./${basename}-slave $PORT ; wait $PID' &&
    Compare ${basename}.out ${reffile}.out ${basename}.diff

    # Report the status and clean up the generated files

    if [ $error -eq 0 ] ; then
    if [ $keep -eq 0 ] ; then
        rm -f $generatedfiles
    fi
    echo "OK"
    echo "###### SUCCESS" 1>&2
    else
    echo "###### FAILED" 1>&2
    globalerror=$error
    fi
}

CheckFail() {
    error=0
    basename=`echo $1 | sed 's/.*\\///
                             s/.ms//'`
    reffile=`echo $1 | sed 's/.ms$//'`
    basedir="`echo $1 | sed 's/\/[^\/]*$//'`/."

    echo -n "$basename..."

    echo 1>&2
    echo "###### Testing $basename" 1>&2

    generatedfiles=""

    generatedfiles="$generatedfiles ${basename}.err ${basename}.diff" &&
    RunFail "$MSCOMPILE" "<" $1 "2>" "${basename}.err" ">>" $globallog &&
    Compare ${basename}.err ${reffile}.err ${basename}.diff

    # Report the status and clean up the generated files

    if [ $error -eq 0 ] ; then
	if [ $keep -eq 0 ] ; then
	    rm -f $generatedfiles
	fi
	echo "OK"
	echo "###### SUCCESS" 1>&2
    else
	echo "###### FAILED" 1>&2
	globalerror=$error
    fi
}

while getopts kdpsh c; do
    case $c in
	k) # Keep intermediate files
	    keep=1
	    ;;
	h) # Help
	    Usage
	    ;;
    esac
done

shift `expr $OPTIND - 1`

LLIFail() {
  echo "Could not find the LLVM interpreter \"$LLI\"."
  echo "Check your LLVM installation and/or modify the LLI variable in testall.sh"
  exit 1
}

which "$LLI" >> $globallog || LLIFail


if [ $# -ge 1 ]
then
    files=$@
else
    files="tests/test-*.ms tests/fail-*.ms"
fi

for file in $files
do
    case $file in
	*test-remote-*)
	    CheckRemote $file 2>> $globallog
	    ;;
    *test-*)
        Check $file 2>> $globallog
        ;;
	*fail-*)
	    CheckFail $file 2>> $globallog
	    ;;
	*)
	    echo "unknown file type $file"
	    globalerror=1
	    ;;
    esac
done

exit $globalerror
