(* Ocamllex scanner for M/s *)

{ open Parser }

let whitespace = [' ' '\t' '\r' '\n']
let alpha = ['a'-'z' 'A'-'Z']
let digit = ['0'-'9']
let ascii = ([' '-'!' '#'-'[' ']'-'~'])
let escape = '\\' ['\\' ''' '"' 'n' 'r' 't']
let escape_char = ''' (escape) '''
let id = (alpha | '_')(alpha | digit | '_')*

rule token = parse
  whitespace { token lexbuf }  (* Whitespace *)
| "/*"     	 { comment lexbuf }  (* Comments *)
| "//"	   	 { line_comment lexbuf }

(* Operators and separators *)
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LCURL }
| '}'      { RCURL }
| '['      { LBRACE }
| ']'      { RBRACE }
| '-''>'   { FIELD }
| ';'      { SEMI }
| ','      { COMMA }
| '.'	   { DOT }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '='      { ASSIGN }
| ':'      { COLON }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&&"     { AND }
| "||"     { OR }
| '!'      { NOT }

(* Branching control *)
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "while"  { WHILE }
| "return" { RETURN }

(* Types *)
| "int"    { INT }
| "double" { DOUBLE }
| "string" { STRING }
| "bool"   { BOOL }
| "void"   { VOID }
| "true"   { TRUE }
| "false"  { FALSE }
| "vector" { VECTOR }
| "struct" { STRUCT }

(* Job related *)
| "job"    { JOB }
| "master" { MASTER }
| "remote" { REMOTE }
| "get"    { GET }
| "cancel" { CANCEL }
| "running" { RUNNING }
| "finished" { FINISHED }
| "failed" { FAILED }

(* Vector related *)
| "size"   {SIZE}
| "::" 	   {PUSHBACK}

| "<<" {CONCAT}
(* Alternative float literal? *)
(*| ['-''+']?digit*('.')digit+(['e''E']['-''+']?digit+)? as lxm { DOUBLE_LITERAL(float_of_string lxm) } *)
| (digit+)'.'(digit+) as lxm { DOUBLE_LITERAL(float_of_string lxm)}
| digit+ as lxm 	{ INT_LITERAL(int_of_string lxm) }
| '"'((ascii|escape|alpha|digit)* as lxm)'"'  { STRING_LITERAL(lxm) }
| id as lxm 		{ ID(lxm) }
| eof 				{ EOF }
| _ as char 		{ raise (Failure("illegal character " ^ Char.escaped char)) }

and comment = parse
  "*/" { token lexbuf }
| _    { comment lexbuf }

and line_comment = parse
  '\n' { token lexbuf }
| _    { line_comment lexbuf }
