/* Ocamlyacc parser for M/s */

%{
open Ast
%}

%token SEMI LPAREN RPAREN LCURL RCURL LBRACE RBRACE COMMA COLON DOT
%token PLUS MINUS TIMES DIVIDE ASSIGN 
%token EQ NEQ LT LEQ GT GEQ TRUE FALSE AND OR NOT
%token RETURN IF ELSE FOR WHILE INT DOUBLE BOOL STRING VOID CONCAT
%token VECTOR
%token STRUCT FIELD
%token JOB MASTER REMOTE GET CANCEL RUNNING FINISHED FAILED

%token <int> INT_LITERAL
%token <float> DOUBLE_LITERAL
%token <string> STRING_LITERAL
%token <string> ID
%token EOF

%token SIZE PUSHBACK

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%right SIZE
%left OR
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE
%left CONCAT
%right NOT NEG
%right RBRACE
%left LBRACE
%right FIELD

%start program
%type <Ast.program> program

%%

program:
  struct_type_decl program { let (f,s) = $2 in f, $1 :: s }
| fdecl program { let (f,s) = $2 in $1 :: f, s }
| master program_terminable { let (f,s) = $2 in $1::f, s }

program_terminable:
  struct_type_decl program_terminable { let (f,s) = $2 in f, $1 :: s }
| fdecl program_terminable { let (f,s) = $2 in $1 :: f, s }
| EOF { [], [] }

master:
  MASTER LCURL stmt_list RCURL {
    {
      fname = "master";
      typ = Int;
      formals = [];
      body = List.rev  $3;
    }
  }

fdecl:
   JOB typ ID LPAREN formals_opt RPAREN LCURL stmt_list RCURL
     { { typ = $2;
	 fname = $3;
	 formals = $5;
	 body = List.rev $8 } }

struct_type_decl:
  STRUCT ID LCURL bind_list RCURL SEMI { {sname = $2; blist = List.rev $4} }

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    typ ID                   { [($1,$2)] }
  | formal_list COMMA typ ID { ($3,$4) :: $1 }

bind:
    typ ID SEMI { ($1, $2) }

bind_list:
    /* nothing */ { [] }
  | bind_list bind { $2 :: $1 }

typ:
    INT { Int }
  | BOOL { Bool }
  | DOUBLE { Double }
  | STRING { Vector(Char) }
  | VOID { Void }
  | VECTOR LT typ GT { Vector($3) }
  | JOB LT typ GT {Job($3)}
  | STRUCT ID { Struct($2) }

vdecl:
  typ ID SEMI {VarDecl($1,$2)}
  | typ ID ASSIGN expr SEMI { VarDeclAssign($1, $2, $4) }

/* first list is vdecl list, second list is statement list */
stmt_list:
    /* nothing */  { [] }
  | stmt_list stmt { $2 :: $1 }

stmt:
    expr SEMI { Expr $1 }
  | vdecl { $1 }
  | RETURN SEMI { Return Noexpr }
  | RETURN expr SEMI { Return $2 }
  | LCURL stmt_list RCURL { Block(List.rev $2) }
  | IF LPAREN expr RPAREN stmt %prec NOELSE { If($3, $5, Block([])) }
  | IF LPAREN expr RPAREN stmt ELSE stmt { If($3, $5, $7) }
  | FOR LPAREN expr_opt SEMI expr SEMI expr_opt RPAREN stmt { For($3, $5, $7, $9) }
  | WHILE LPAREN expr RPAREN stmt { While($3, $5) }
  | expr PUSHBACK expr SEMI {VectorPushBack($1, $3)}

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

expr:
    INT_LITERAL      { Literal($1) }
  | DOUBLE_LITERAL   { DoubleLit($1) }
  | STRING_LITERAL   { StringLit($1) }
  | TRUE             { BoolLit(true) }
  | FALSE            { BoolLit(false) }
  | ID               { Id($1) }
  | LBRACE actuals_opt RBRACE { ListLiteral($2) }
  | expr PLUS   expr { Binop($1, Add,   $3) }
  | expr MINUS  expr { Binop($1, Sub,   $3) }
  | expr TIMES  expr { Binop($1, Mult,  $3) }
  | expr DIVIDE expr { Binop($1, Div,   $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq,   $3) }
  | expr LT     expr { Binop($1, Less,  $3) }
  | expr LEQ    expr { Binop($1, Leq,   $3) }
  | expr GT     expr { Binop($1, Greater, $3) }
  | expr GEQ    expr { Binop($1, Geq,   $3) }
  | expr AND    expr { Binop($1, And,   $3) }
  | expr OR     expr { Binop($1, Or,    $3) }
  | MINUS expr %prec NEG { Unop(Neg, $2) }
  | NOT expr         { Unop(Not, $2) }
  | ID ASSIGN expr   { Assign($1, $3) }
  | ID LPAREN actuals_opt RPAREN { Call($1, $3) }
  | REMOTE ID LPAREN actuals_opt RPAREN { RemoteCall($2, $4) }
  | GET ID { Get($2) }
  | CANCEL ID { Cancel($2) }
  | ID DOT RUNNING { Running($1) }
  | ID DOT FINISHED { Finished($1) }
  | ID DOT FAILED { Failed($1) }
  | LPAREN expr RPAREN { $2 }
  | ID vector_indices {VectorAccess($1, $2)}
  | ID vector_indices ASSIGN expr {VectorAssign($1, $2, $4)}
  | ID LBRACE expr COLON expr RBRACE {VectorRangeAccess($1, $3, $5)}
  | expr FIELD ID               { StructFieldAccess($1, $3) }
  | expr FIELD ID ASSIGN expr   { StructFieldAssign($1, $3, $5)}
  | SIZE expr {VectorSize($2)}
  | expr CONCAT expr {Concat($1, $3)}

vector_indices:
    LBRACE expr RBRACE {[$2]}
  | vector_indices LBRACE expr RBRACE {$3 :: $1}

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
    expr                    { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }

