(* Code generation: translate takes a semantically checked AST and
produces LLVM IR

LLVM tutorial: Make sure to read the OCaml version of the tutorial

http://llvm.org/docs/tutorial/index.html

Detailed documentation on the OCaml LLVM library:

http://llvm.moe/
http://llvm.moe/ocaml/

*)

module S = String
module L = Llvm
module A = Ast

module StringMap = Map.Make(String)

let translate (functions, struct_decls) =
	let context = L.global_context () in
	let the_module = L.create_module context "M/s"
	and double_type = L.double_type context
  and char_t = L.i8_type context
	and i32_t = L.i32_type context
	and i8_t = L.i8_type context
	and i1_t = L.i1_type context
	and void_t = L.void_type context in
  let vec_t = L.named_struct_type context "vec_t" in
  ignore (L.struct_set_body vec_t [|L.pointer_type vec_t ; i32_t ; i32_t|] false) ;



  let struct_defs = Hashtbl.create 5000 in
  let rec create_struct_map map cur_idx = function
      [] -> StringMap.empty
    | [(typ, name)] -> StringMap.add name (typ, cur_idx) map
    | h::t -> let m = create_struct_map map cur_idx [h] in create_struct_map m (cur_idx+1) t
  in
  let rec create_global_map global_map = function
      [] -> StringMap.empty
    |  [x] -> StringMap.add x.A.sname (create_struct_map StringMap.empty 0 x.A.blist) global_map
    | h::t -> let m = create_global_map global_map [h] in create_global_map m t
  in

  let global_map = create_global_map StringMap.empty struct_decls in

  let (*rec*) ltype_of_typ = function
      A.Int -> i32_t
    | A.Char -> char_t
    | A.Double -> double_type
    | A.Bool -> i1_t
    | A.Void -> void_t
    | A.Job(_) -> i32_t
    | A.Vector(_) -> vec_t (*ltype_of_typ t*)
        | A.Struct(name) -> Hashtbl.find struct_defs name
        | _ -> raise(Failure("unrecognized type")) in

  let struct_def struct_decl =
      let struct_body = List.fold_left (fun a (typ, _(*name*)) -> Array.append a [|ltype_of_typ typ|]) [||] struct_decl.A.blist
      in let struct_name = L.named_struct_type context struct_decl.A.sname
      in ignore(L.struct_set_body struct_name struct_body false); struct_name;
    in ignore (List.map (fun struct_decl ->
        let llvm_def = struct_def struct_decl in Hashtbl.add struct_defs struct_decl.A.sname llvm_def)
        struct_decls) ;

	 (* Declare printf(), which the print built-in function will call *)
	let printf_t = L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
	let printf_func = L.declare_function "printf" printf_t the_module in

	(* declare master library functions *)
    let start_job_t = L.function_type i32_t [| i32_t ; L.pointer_type i8_t ;
        i32_t |] in
    let start_job_func = L.declare_function "start_job" start_job_t the_module in

    let reap_job_t = L.function_type void_t [| i32_t ;
        L.pointer_type (L.pointer_type i8_t) ; L.pointer_type i32_t |] in
    let reap_job_func = L.declare_function "reap_job" reap_job_t the_module in

    let job_status_t = L.function_type i32_t [| i32_t |] in
    let get_job_status_func = L.declare_function "get_job_status" job_status_t the_module in

    let cancel_job_t = L.function_type i32_t [| i32_t |] in
    let cancel_job_func = L.declare_function "cancel_job" cancel_job_t the_module in

  let memcpy_t = L.function_type void_t [|L.pointer_type i8_t; L.pointer_type i8_t; i32_t; i32_t; i1_t|] in
  let memcpy_func = L.declare_function "llvm.memcpy.p0i8.p0i8.i32" memcpy_t the_module in

	(* Define each function (arguments and return type) so we can call it *)
	(*master_decls is a map that stores key=main, value=(llvm's def of main, AST of master to map *)

  let get_ordinal f = let rec get_index e n = function
    | [] -> -1
    | x::_ when (x.A.fname = e) -> n
    | x::rest when (x.A.fname = "master") -> get_index e n rest
    | _::rest -> get_index e (n+1) rest in
  L.const_int i32_t (get_index f 0 functions + 1) in

        let zeroth = L.const_int i32_t 0 in
        let first = L.const_int i32_t 1 in
        let second = L.const_int i32_t 2 in

        (* serialization code *)
        let serialize builder evals the_function typs =
                let rec compute_size (b, size) e typ = (match typ with
                    A.Vector(t) ->
                        let e_store = L.build_alloca vec_t "e_store" b in
                        ignore (L.build_store e e_store b) ;
                        let iter = L.build_alloca i32_t "vec_sz_var" b in
                        ignore (L.build_store (L.const_null i32_t) iter b);
                        let vec_ptr = L.build_gep e_store [|zeroth; zeroth|] "vec_ptr" b in
                        let ptr = L.build_load vec_ptr "ptr" b in
                        let vec_sz_p = L.build_gep e_store [|zeroth; first|] "vec_sz_p" b in
                        let vec_sz = L.build_load vec_sz_p "vec_sz" b in
                        let size_val = L.build_load size "size_val" b in
                        let size_add = L.build_add size_val (L.const_int i32_t 4) "size_plus4" b in
                        ignore (L.build_store size_add size b);
                        let pred_bb = L.append_block context "check_size" the_function in
		                let pred_builder = L.builder_at_end context pred_bb in
                        let iter_val = L.build_load iter "iter_val" pred_builder in
                        let keeplooping = L.build_icmp L.Icmp.Slt iter_val vec_sz "keeplooping" pred_builder in
		                let sz_vec_elem_bb = L.append_block context "del_vec_elem" the_function in
                        let sz_vec_elem_builder = L.builder_at_end context sz_vec_elem_bb in
                        let iter_val = L.build_load iter "iter_val" sz_vec_elem_builder in
                        let origin_ptr = L.build_gep ptr [|iter_val|] "dest_ptr" sz_vec_elem_builder in
                        let origin_ptr_cast = L.build_bitcast origin_ptr (L.pointer_type (ltype_of_typ t)) "origin_ptr_cast" sz_vec_elem_builder in
                        let inner_vec = L.build_load origin_ptr_cast "inner_vec" sz_vec_elem_builder in
                        let (newbuilder, _) = compute_size (sz_vec_elem_builder, size) inner_vec t in
                        let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                        ignore (L.build_store new_iter_val iter newbuilder);
		                let merge_bb = L.append_block context "merge" the_function in
                        let merge_builder = L.builder_at_end context merge_bb in
		                ignore (L.build_cond_br keeplooping sz_vec_elem_bb merge_bb pred_builder);
			            ignore (L.build_br pred_bb newbuilder);
                        ignore (L.build_br pred_bb b);
                        (merge_builder, size)
                  | A.Struct(n) ->
                        let e_store = L.build_alloca (ltype_of_typ (A.Struct(n))) "store_struct" b in
                        ignore (L.build_store e e_store b) ;
                        let struct_def_map = StringMap.find n global_map in
                        let bindings = StringMap.bindings struct_def_map in
                        let size_elem (b, size) binding =
                            let (_, (fieldtyp, idx)) = binding in
                            let field_ptr = L.build_struct_gep e_store idx "field_ptr" b in
                            let field = L.build_load field_ptr "field" b in
                            let (newbuilder, _) = compute_size (b, size) field fieldtyp in
                            (newbuilder, size) in
                        List.fold_left size_elem (b, size) bindings
                  | t -> let size_val = L.build_load size "size_val" b in
                                  let size_add = L.build_add size_val (L.build_trunc (L.size_of (ltype_of_typ t)) i32_t "trunc" b) "size_add" b in
                                  ignore (L.build_store size_add size b) ;
                                  (b, size)) in
                let size = L.build_alloca i32_t "size" builder in
                ignore (L.build_store (L.const_null i32_t) size builder) ;
                let (nnbuilder, serializelen_p) = List.fold_left2 compute_size (builder, size) evals typs in
                let serializelen = L.build_load serializelen_p "serializelen" nnbuilder in
                let serializebuf = L.build_alloca (L.pointer_type i8_t) "serializebuf" nnbuilder in
                let serializebuf_val = L.build_array_malloc i8_t serializelen "serializearg" nnbuilder in
                ignore (L.build_store serializebuf_val serializebuf nnbuilder);

                let rec serialize_next (ptr, b) e typ =
                  let target = L.build_load ptr "target" b in
                  (match typ with
                    A.Vector(typ) ->
                  let target_size_p = L.build_bitcast target (L.pointer_type i32_t) "size_32" b in
                  let e_store = L.build_alloca vec_t "e_store" b in
                  ignore (L.build_store e e_store b) ;
                  let vec_ptr = L.build_gep e_store [|zeroth; zeroth|] "vec_ptr" b in
                  let ptr_v = L.build_load vec_ptr "ptr" b in
                  let vec_sz_p = L.build_gep e_store [|zeroth; first|] "vec_sz_p" b in
                  let vec_sz = L.build_load vec_sz_p "vec_sz" b in
                  ignore (L.build_store vec_sz target_size_p b) ;
                  ignore (L.build_store (L.build_gep target [|L.const_int i32_t 4|] "next_target" b) ptr b);
                  let iter = L.build_alloca i32_t "vec_sz_var" b in
                  ignore (L.build_store (L.const_null i32_t) iter b);
                  let pred_bb = L.append_block context "check_size" the_function in
		          let pred_builder = L.builder_at_end context pred_bb in
                  let iter_val = L.build_load iter "iter_val" pred_builder in
                  let keeplooping = L.build_icmp L.Icmp.Slt iter_val vec_sz "keeplooping" pred_builder in
		          let sz_vec_elem_bb = L.append_block context "del_vec_elem" the_function in
                  let sz_vec_elem_builder = L.builder_at_end context sz_vec_elem_bb in
                  let iter_val = L.build_load iter "iter_val" sz_vec_elem_builder in
                  let ptr_v_cast = L.build_bitcast ptr_v (L.pointer_type (ltype_of_typ typ)) "ptr_v_cast" sz_vec_elem_builder in
                  let origin_ptr = L.build_gep ptr_v_cast [|iter_val|] "dest_ptr" sz_vec_elem_builder in
                  let inner_elem = L.build_load origin_ptr "inner_elem" sz_vec_elem_builder in
                  let (_, newbuilder) = serialize_next (ptr, sz_vec_elem_builder) inner_elem typ  in
                  let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                  ignore (L.build_store new_iter_val iter newbuilder);
		          let merge_bb = L.append_block context "merge" the_function in
                  let merge_builder = L.builder_at_end context merge_bb in
		          ignore (L.build_cond_br keeplooping sz_vec_elem_bb merge_bb pred_builder);
			      ignore (L.build_br pred_bb newbuilder);
                  ignore (L.build_br pred_bb b);
                  (ptr, merge_builder)
                  | A.Struct(n) ->
                        let e_store = L.build_alloca (ltype_of_typ (A.Struct(n))) "store_struct" b in
                        ignore (L.build_store e e_store b);
                        let struct_def_map = StringMap.find n global_map in
                        let bindings = StringMap.bindings struct_def_map in
                        let serialize_elem b binding =
                            let (_, (fieldtyp, idx)) = binding in
                            let field_ptr = L.build_struct_gep e_store idx "field_ptr" b in
                            let field = L.build_load field_ptr "field" b in
                            let (_, newbuilder) = serialize_next (ptr, b) field fieldtyp in
                            newbuilder in
                        (ptr, List.fold_left serialize_elem b bindings)
                    | typ -> let target_typ_p = L.build_bitcast target (L.pointer_type (ltype_of_typ typ)) "elem_nn" b in
                             ignore (L.build_store e target_typ_p b) ;
                             ignore (L.build_store (L.build_gep target [|L.build_trunc (L.size_of (ltype_of_typ typ)) i32_t "trunc" b|] "newtarget" b) ptr b) ;
                             (ptr, b)) in
                (* serialize each argument into serializebuf using list fold magic *)
                let (_, finalbuilder) = List.fold_left2 serialize_next (serializebuf, nnbuilder) evals typs in
                (*returns serialize buf*)
                (serializebuf_val, serializelen, finalbuilder) in

        let rec deserialize (lst, ptr, the_function, builder) typ =
            let ptr_val = L.build_load ptr "ptr_val" builder in
            (match typ with
                A.Vector(t) -> let size = L.build_bitcast ptr_val (L.pointer_type i32_t) "size" builder in
                               let size_val = L.build_load size "size_val" builder in
                               let iter = L.build_alloca i32_t "iter" builder in
                               ignore (L.build_store (L.const_null i32_t) iter builder);
                               let new_ptr = L.build_gep ptr_val [|L.const_int i32_t 4|] "new_ptr" builder in
                               ignore (L.build_store new_ptr ptr builder);
                               let vec_ptr = L.build_alloca vec_t "vec_ptr" builder in
                               let vec_ptr_p = L.build_gep vec_ptr [|zeroth; zeroth|] "vec_ptr_p" builder in
                               let vec_ptr_sz = L.build_gep vec_ptr [|zeroth; first|] "vec_ptr_sz" builder in
                               let vec_ptr_ln = L.build_gep vec_ptr [|zeroth; second|] "vec_ptr_ln" builder in
                               let buf = L.build_array_malloc (ltype_of_typ t) size_val "buf" builder in
                               let buf_cast = L.build_bitcast buf (L.pointer_type vec_t) "buf_cast" builder in
                               ignore (L.build_store buf_cast vec_ptr_p builder);
                               ignore (L.build_store size_val vec_ptr_sz builder);
                               ignore (L.build_store size_val vec_ptr_ln builder);
                               let pred_bb = L.append_block context "check_size" the_function in
		                       let pred_builder = L.builder_at_end context pred_bb in
                               let iter_val = L.build_load iter "iter_val" pred_builder in
                               let keeplooping = L.build_icmp L.Icmp.Slt iter_val size_val "keeplooping" pred_builder in
		                       let sz_vec_elem_bb = L.append_block context "del_vec_elem" the_function in
                               let sz_vec_elem_builder = L.builder_at_end context sz_vec_elem_bb in
                               let iter_val = L.build_load iter "iter_val" sz_vec_elem_builder in
                               let ptr_v_cast = L.build_bitcast buf (L.pointer_type (ltype_of_typ t)) "ptr_v_cast" sz_vec_elem_builder in
                               let origin_ptr = L.build_gep ptr_v_cast [|iter_val|] "dest_ptr" sz_vec_elem_builder in
                               let (res_l, _, _, newbuilder) = deserialize ([], ptr, the_function, sz_vec_elem_builder) t in
                               let res = (match res_l with [r] -> r | _ -> raise(Failure("expected to only deserialize one thing")) ) in
                               ignore (L.build_store res origin_ptr newbuilder);
                               let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                               ignore (L.build_store new_iter_val iter newbuilder);
		                       let merge_bb = L.append_block context "merge" the_function in
                               let merge_builder = L.builder_at_end context merge_bb in
		                       ignore (L.build_cond_br keeplooping sz_vec_elem_bb merge_bb pred_builder);
			                   ignore (L.build_br pred_bb newbuilder);
                               ignore (L.build_br pred_bb builder);
                               ((L.build_load vec_ptr "vec" merge_builder)::lst, ptr, the_function, merge_builder)
                  | A.Struct(n) ->
                        let e_store = L.build_alloca (ltype_of_typ (A.Struct(n))) "store_struct" builder in
                        let struct_def_map = StringMap.find n global_map in
                        let bindings = StringMap.bindings struct_def_map in
                        let serialize_field b binding =
                            let (_, (fieldtyp, idx)) = binding in
                            let field_ptr = L.build_struct_gep e_store idx "field_ptr" b in
                            let (field, _, _, newbuilder) = deserialize ([], ptr, the_function, b) fieldtyp in
                            let res = (match field with [r] -> r | _ -> raise(Failure("expected to only deserialize one thing"))) in
                            ignore (L.build_store res field_ptr newbuilder) ;
                            newbuilder in
                        let nbuilder = List.fold_left serialize_field builder bindings in
                        ((L.build_load e_store "struct" nbuilder)::lst, ptr, the_function, nbuilder)
              | typ -> let src = L.build_bitcast ptr_val (L.pointer_type (ltype_of_typ typ)) "src" builder in
                       let new_ptr_val = L.build_gep ptr_val [|L.build_trunc (L.size_of (ltype_of_typ typ)) i32_t "trunc" builder|] "new_ptr_val" builder in
                       ignore (L.build_store new_ptr_val ptr builder) ;
                       ((L.build_load src "deserialized_typ" builder)::lst, ptr, the_function, builder) ) in

              let rec struct_delete ptr struct_def_map builder the_function  =
                    let bindings = StringMap.bindings struct_def_map in
                    let struct_field_delete_helper builder binding  =
                      let (_, (fieldtyp, idx)) = binding in
                          (match fieldtyp with
                              A.Vector(in_typ) -> let vec_ptr = L.build_struct_gep ptr idx "vec_ptr_delete" builder in
                              (*raise(Failure (L.string_of_llvalue((vec_ptr)) ^ "; "))*)

                                                vector_delete vec_ptr builder the_function in_typ
                            | A.Struct(in_typ) ->
                                let field_val = L.build_struct_gep ptr idx "inner_struct_field_delete" builder in
                                (*raise(Failure (L.string_of_llvalue((field_val)) ^ "; "))*)

                                let field_struct_def_map = StringMap.find in_typ global_map  in
                                struct_delete field_val field_struct_def_map builder the_function

                            | _ -> builder) in
                      List.fold_left struct_field_delete_helper builder bindings;
    and vector_delete vec builder the_function = function
        A.Vector(typ) -> let ptr_p = L.build_gep vec [|zeroth; zeroth|] "ptr_p" builder in
            let size_p = L.build_gep vec [|zeroth; first|] "size_p" builder in
            let ptr = L.build_load ptr_p "ptr" builder in
            let size = L.build_load size_p "size" builder in
            let iter = L.build_alloca i32_t "iter" builder in
            ignore (L.build_store (L.const_int i32_t 0) iter builder) ;
            let zerosize = L.build_icmp L.Icmp.Slt size (L.const_int i32_t 1) "iszerosize" builder in
            let pred_bb = L.append_block context "check_size" the_function in
		      let pred_builder = L.builder_at_end context pred_bb in
              let iter_val = L.build_load iter "iter_val" pred_builder in
              let keeplooping = L.build_icmp L.Icmp.Slt iter_val size "keeplooping" pred_builder in
		    let del_vec_elem_bb = L.append_block context "del_vec_elem" the_function in
              let del_vec_elem_builder = L.builder_at_end context del_vec_elem_bb in
              let iter_val = L.build_load iter "iter_val" del_vec_elem_builder in
              let origin_ptr = L.build_gep ptr [|iter_val|] "dest_ptr" del_vec_elem_builder in
              let newbuilder = vector_delete origin_ptr del_vec_elem_builder the_function typ in
                  let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                  ignore (L.build_store new_iter_val iter newbuilder);
                  let merge_bb = L.append_block context "merge" the_function in
              let merge_builder = L.builder_at_end context merge_bb in
              ignore(L.build_free ptr merge_builder) ;
            let mergeagain_bb = L.append_block context "mergeagain" the_function in
              let mergeagain_builder = L.builder_at_end context mergeagain_bb in
        ignore (L.build_cond_br zerosize mergeagain_bb pred_bb builder);
        ignore (L.build_cond_br keeplooping del_vec_elem_bb merge_bb pred_builder);
      ignore (L.build_br pred_bb newbuilder);
            ignore (L.build_br mergeagain_bb merge_builder);
            mergeagain_builder

      | A.Struct(inner_t) -> let ptr_p = L.build_gep vec [|zeroth; zeroth|] "ptr_p" builder in
            let size_p = L.build_gep vec [|zeroth; first|] "size_p" builder in
            let ptr = L.build_load ptr_p "ptr" builder in
            let size = L.build_load size_p "size" builder in
            let iter = L.build_alloca i32_t "iter" builder in
            ignore (L.build_store (L.const_int i32_t 0) iter builder) ;
            let zerosize = L.build_icmp L.Icmp.Slt size (L.const_int i32_t 1) "iszerosize" builder in
            let pred_bb = L.append_block context "check_size" the_function in
          let pred_builder = L.builder_at_end context pred_bb in
              let iter_val = L.build_load iter "iter_val" pred_builder in
              let keeplooping = L.build_icmp L.Icmp.Slt iter_val size "keeplooping" pred_builder in
        let del_vec_elem_bb = L.append_block context "del_vec_elem" the_function in
              let del_vec_elem_builder = L.builder_at_end context del_vec_elem_bb in
              let iter_val = L.build_load iter "iter_val" del_vec_elem_builder in
              let origin_ptr = L.build_gep ptr [|iter_val|] "dest_ptr" del_vec_elem_builder in
              let struct_def_map = StringMap.find inner_t global_map in
                  let llvm_struct_type = ltype_of_typ(A.Struct(inner_t)) in
                  let struct_p = L.build_bitcast origin_ptr (L.pointer_type llvm_struct_type) "struct_p" del_vec_elem_builder in
                  let newbuilder = struct_delete struct_p struct_def_map del_vec_elem_builder the_function in
                  let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                  ignore (L.build_store new_iter_val iter newbuilder);
                  let merge_bb = L.append_block context "merge" the_function in
              let merge_builder = L.builder_at_end context merge_bb in
              ignore(L.build_free ptr merge_builder) ;
            let mergeagain_bb = L.append_block context "mergeagain" the_function in
              let mergeagain_builder = L.builder_at_end context mergeagain_bb in
        ignore (L.build_cond_br zerosize mergeagain_bb pred_bb builder);
        ignore (L.build_cond_br keeplooping del_vec_elem_bb merge_bb pred_builder);
      ignore (L.build_br pred_bb newbuilder);
            ignore (L.build_br mergeagain_bb merge_builder);
            mergeagain_builder
        | _ -> let ptr_p = L.build_gep vec [|zeroth; zeroth|] "ptr_p" builder in
            let size_p = L.build_gep vec [|zeroth; first|] "size_p" builder in
            let ptr = L.build_load ptr_p "ptr" builder in
            let size = L.build_load size_p "size" builder in
            let iszerosize = L.build_icmp L.Icmp.Slt size (L.const_int i32_t 1) "iszerosize" builder in
            let nonzerosize_bb = L.append_block context "del_nonzerosize" the_function in
              let nonzerosize_builder = L.builder_at_end context nonzerosize_bb in
              ignore (L.build_free ptr nonzerosize_builder);
            let merge_bb = L.append_block context "merge" the_function in
              let merge_builder = L.builder_at_end context merge_bb in
            ignore(L.build_br merge_bb nonzerosize_builder);
            ignore(L.build_cond_br iszerosize merge_bb nonzerosize_bb builder);
            merge_builder
        in
	let function_decls =
		let function_decl m fdecl =
			let name = fdecl.A.fname
			(* formal_type should be an empty array list*)
			and formal_type = Array.of_list (List.map (fun (t,_) -> ltype_of_typ t) fdecl.A.formals)
			(*ftype is int*)
		in let ftype = L.function_type (ltype_of_typ fdecl.A.typ) formal_type in
		(* stores key=main, value=(llvm's def of main, AST of master to map *)
			StringMap.add name (L.define_function name ftype the_module, fdecl) m in
		List.fold_left function_decl StringMap.empty functions in

	(* Fill in the body of the given function *)
	let build_function_body fdecl =
		let (the_function, _) = StringMap.find fdecl.A.fname function_decls in
		(* builder used always points to end of block*)
		let builder = L.builder_at_end context (L.entry_block the_function) in

		let int_format_str = L.build_global_stringptr "%d\n" "fmt" builder in
		let double_format_str = L.build_global_stringptr "%f\n" "fmt" builder in

		let true_str = L.build_global_stringptr "true\n" "true_str" builder in
		let false_str = L.build_global_stringptr "false\n" "false_str" builder in
    let string_str = L.build_global_stringptr "%s\n" "false_str" builder in

        let vector_type_map = Hashtbl.create 5000 in
        let add_vector_type s typ = Hashtbl.add vector_type_map s typ in
        let is_vector n = Hashtbl.mem vector_type_map n in
        let get_vector_type n = Hashtbl.find vector_type_map n in
        let string_lit_map = Hashtbl.create 5000 in
        let add_to_string_lit_cleanup ptr =
          Hashtbl.add string_lit_map ptr ""
        in

    let struct_type_map = Hashtbl.create 5000 in
    let add_struct_type s typ = Hashtbl.add struct_type_map s typ in
    let get_struct_type n = try (Hashtbl.find struct_type_map n) with Not_found -> raise(Failure("fooooo")) in
    let is_struct id = Hashtbl.mem struct_type_map id in
		(* adds formals and local vars to local_var_map
			 set user var name to llvm's param pointer
		*)
		let local_var_map = Hashtbl.create 5000 in
    let llval_to_id_map = Hashtbl.create 5000 in
		let add_formal (t, n) p = (match t with
                A.Vector(typ) -> ignore (add_vector_type n typ)
              | A.Struct(typ) -> ignore (add_struct_type n typ)
              | _ -> ()) ; L.set_value_name n p;
		let local = L.build_alloca (ltype_of_typ t) n builder in
			ignore (L.build_store p local builder); Hashtbl.replace local_var_map n local in
		ignore (List.iter2 add_formal fdecl.A.formals (Array.to_list (L.params the_function))) ;

		let add_var_init id llvalue =
			let _ = Hashtbl.add llval_to_id_map llvalue id in Hashtbl.add local_var_map id llvalue
		in
		let lookup s =
      if Hashtbl.mem local_var_map s then Hashtbl.find local_var_map s
    else raise (Failure("cannot find symbol in local_var_map"))
		in
        let job_type_map = Hashtbl.create 5000 in
        let add_job_type s typ = Hashtbl.add job_type_map s typ in

    let vector_decl(id, typ, builder) =
          let vec_alloc = L.build_alloca (vec_t) "vec_alloc" builder in
          let ptr = L.build_gep vec_alloc [|zeroth; zeroth|] "ptr" builder in
          ignore (L.build_store (L.const_pointer_null (L.pointer_type vec_t))  ptr builder) ;
          let size = L.build_gep vec_alloc [|zeroth; first|] "size" builder in
          ignore (L.build_store (L.const_null i32_t) size builder) ;
          let len = L.build_gep vec_alloc [|zeroth; second|] "len" builder in
          ignore (L.build_store (L.const_null i32_t) len builder) ;
          add_vector_type id typ ; vec_alloc
    in

    let rec string_store s i n vec builder =
          (match i with
            | i when (i == n) ->
              let ith = L.const_int i32_t i in
              let index = L.build_gep vec [|ith|] "index" builder in
              ignore (L.build_store (L.const_null i8_t) index builder) ;
            | _ ->
              let c = S.get s i in
              let c1 = L.const_int i8_t (int_of_char c) in
              let ith = L.const_int i32_t i in
              let index = L.build_gep vec [|ith|] "index" builder in
              ignore (L.build_store c1 index builder) ;
              string_store s (i+1) n vec builder

          )
    in

    let string_decl s builder =
          let s_length = S.length s in
          let vec_alloc = L.build_alloca vec_t "vec_alloc" builder in
          let ptr = L.build_gep vec_alloc [|zeroth; zeroth|] "ptr" builder in
          let s_length_null = L.const_int i32_t (s_length + 1) in
          let oldvec = L.build_array_malloc char_t s_length_null "oldvec" builder in
          let newvec = L.build_bitcast oldvec (L.pointer_type vec_t) "newvec" builder in
          ignore (L.build_store newvec ptr builder) ;
          let size = L.build_gep vec_alloc [|zeroth; first|] "size" builder in
          ignore (L.build_store s_length_null size builder) ;
          let len = L.build_gep vec_alloc [|zeroth; second|] "len" builder in
          ignore (L.build_store s_length_null len builder) ;
          ignore (string_store s 0 s_length oldvec builder) ;
          L.build_load vec_alloc "vec" builder

    in

    let string_concat vect1_orig vect2_orig builder =
        let vect1 = L.build_alloca vec_t "tmp_vect1" builder in
        let vect2 = L.build_alloca vec_t "tmp_vect1" builder in
        let _ = L.build_store vect1_orig vect1 builder in
        let _ = L.build_store vect2_orig vect2 builder in
        let size1 = L.build_gep vect1 [|zeroth; first|] "size_1" builder in
        let size1 = L.build_load size1 "loaded_size_1" builder in
        let ptr1 = L.build_gep vect1 [|zeroth; zeroth|] "ptr_1" builder in
        (*let len1 = L.build_gep vect1 [|zeroth; second|] "len_1" builder in*)
        let size2 = L.build_gep vect2 [|zeroth; first|] "size_2" builder in
        let size2 = L.build_load size2 "loaded_size_2" builder in
        let ptr2 = L.build_gep vect2 [|zeroth; zeroth|] "ptr_2" builder in
        (*let len2 = L.build_gep vect2 [|zeroth; second|] "len_2" builder in*)
        let newsize = L.build_add size1 size2 "new_size" builder in
        let newsize = L.build_sub newsize (L.const_int i32_t 1) "new_size_sub" builder in
        let newvec_alloc = L.build_alloca vec_t "new_vec_alloc" builder in
        let new_ptr = L.build_gep newvec_alloc [|zeroth; zeroth|] "new_ptr" builder in
        let new_size_var = L.build_gep newvec_alloc [|zeroth; first|] "new_size" builder in
        let new_leng_var = L.build_gep newvec_alloc [|zeroth; second|] "new_leng" builder in
        let malloc_inner = L.build_array_malloc char_t newsize "malloc_string" builder in
        let malloc_inner_cast = L.build_bitcast malloc_inner (L.pointer_type vec_t) "malloc_inner_cast" builder in
        ignore (L.build_store malloc_inner_cast new_ptr builder);
        let cast_vect1 = L.build_bitcast ptr1 (L.pointer_type (L.pointer_type i8_t)) "cast_vect1" builder in
        let cast_vect2 = L.build_bitcast ptr2 (L.pointer_type (L.pointer_type i8_t)) "cast_vect2" builder in
        let cast_newvec = L.build_bitcast new_ptr (L.pointer_type (L.pointer_type i8_t)) "cast_newvec" builder in
        let cast_vect1 = L.build_load cast_vect1 "cast_vect1" builder in
        let cast_vect2 = L.build_load cast_vect2 "cast_vect1" builder in
        let cast_newvec = L.build_load cast_newvec "cast_vect1" builder in
        let leng_to_cp = L.build_sub size1 (L.const_int i32_t 1) "copy_leng_first" builder in
        let _ = L.build_call memcpy_func [|cast_newvec; cast_vect1; leng_to_cp; L.const_int i32_t 1; L.const_int i1_t 1|] "" builder in
        let move_cp_ptr = L.build_gep cast_newvec [|leng_to_cp|] "move_cp_ptr" builder in

        let _ = L.build_call memcpy_func [|move_cp_ptr; cast_vect2; size2; L.const_int i32_t 1; L.const_int i1_t 1|] "" builder in
       (*
        raise(Failure (L.string_of_llvalue((move_cp_ptr)) ^ "; " ^ L.string_of_llvalue((vect1))))
        *)
        ignore(L.build_store newsize new_leng_var builder);
        ignore(L.build_store newsize new_size_var builder);
        L.build_load newvec_alloc "newvec" builder

      in








              let rec struct_copy lptr rptr struct_def_map builder =
                let bindings = StringMap.bindings struct_def_map in
                let struct_field_assign_helper builder binding  =
                  let (_, (fieldtyp, idx)) = binding in
                    (match fieldtyp with
                        A.Vector(in_typ) ->
                  let lvec_ptr = L.build_struct_gep lptr idx "l_field_vec_cp_assign" builder in
                  let rvec_ptr = L.build_struct_gep rptr idx "r_field_vec_cp_assign" builder in
                  let load_rptr = L.build_load rvec_ptr "loaded_rptr" builder in
                  let builder3 = builder in
                  let temp = L.build_alloca vec_t "temp" builder3 in
                  let _ = L.build_store load_rptr temp builder3 in
                    let (vec_cpy, builder4) = vector_copy temp builder3 in_typ in
                    let sizeof = L.build_trunc (L.size_of (ltype_of_typ fieldtyp)) i32_t "sizeof" builder4 in
                    let temp = L.build_alloca vec_t "temp" builder4 in
                    let _ = L.build_store vec_cpy temp builder4 in
                    let src_cast = L.build_bitcast temp (L.pointer_type i8_t) "src_cast" builder4 in
                    let dst_cast = L.build_bitcast lvec_ptr (L.pointer_type i8_t) "dst_cast" builder4 in
                    let _ = L.build_call memcpy_func [|dst_cast; src_cast; sizeof; L.const_int i32_t 1; L.const_int i1_t 1|] "" builder4
                  in builder4
                | A.Struct(in_typ) ->
                    let l_field_val = L.build_struct_gep lptr idx "l_inner_struct_field" builder in
                    let r_field_val = L.build_struct_gep rptr idx "r_inner_struct_field" builder in
                    let field_struct_def_map = StringMap.find in_typ global_map  in
                    let builder = struct_copy l_field_val r_field_val field_struct_def_map builder in builder;
                | _ ->
                    let l_field_ptr = L.build_struct_gep lptr idx "l_field_cp_assign" builder in
                    let r_field_ptr = L.build_struct_gep rptr idx "r_field_cp_assign" builder in
                    let load_rptr = L.build_load r_field_ptr "loaded_rptr" builder in
                    ignore(L.build_store load_rptr l_field_ptr builder); builder)
                  in
            let builder = List.fold_left struct_field_assign_helper builder bindings in builder;

    and vector_copy vec builder outer_type = match outer_type with
        A.Vector(typ) -> let ptr_p = L.build_gep vec [|zeroth; zeroth|] "ptr_p" builder in
            let size_p = L.build_gep vec [|zeroth; first|] "size_p" builder in
            let len_p = L.build_gep vec [|zeroth; second|] "len_p" builder in
            let ptr = L.build_load ptr_p "ptr" builder in
            let size = L.build_load size_p "size" builder in
            let len = L.build_load len_p "len" builder in
            let iter = L.build_alloca i32_t "iter" builder in
            ignore (L.build_store (L.const_int i32_t 0) iter builder) ;
            let out_vec = L.build_alloca vec_t "out_vec" builder in
            let out_ptr = L.build_gep out_vec [|zeroth; zeroth|] "out_ptr" builder in
            let out_size = L.build_gep out_vec [|zeroth; first|] "out_size" builder in
            let out_len = L.build_gep out_vec [|zeroth; second|] "out_len" builder in
            ignore (L.build_store size out_size builder) ;
            ignore (L.build_store len out_len builder) ;
            let zerosize = L.build_icmp L.Icmp.Slt size (L.const_int i32_t 1) "iszerosize" builder in
            let zerosize_bb = L.append_block context "zerosize" the_function in
              let zerosize_builder = L.builder_at_end context zerosize_bb in
              ignore (L.build_store (L.const_pointer_null (L.pointer_type vec_t)) out_ptr zerosize_builder) ;
            let nonzerosize_bb = L.append_block context "nonzerosize" the_function in
              let nonzerosize_builder = L.builder_at_end context nonzerosize_bb in
              let newvec = L.build_array_malloc vec_t len "newvec" nonzerosize_builder in
              ignore (L.build_store newvec out_ptr nonzerosize_builder) ;
            let pred_bb = L.append_block context "check_size" the_function in
		      let pred_builder = L.builder_at_end context pred_bb in
              let iter_val = L.build_load iter "iter_val" pred_builder in
              let keeplooping = L.build_icmp L.Icmp.Slt iter_val size "keeplooping" pred_builder in
		    let copy_vec_elem_bb = L.append_block context "copy_vec_elem" the_function in
              let copy_vec_elem_builder = L.builder_at_end context copy_vec_elem_bb in
              let iter_val = L.build_load iter "iter_val" copy_vec_elem_builder in
              let vec_ptr = L.build_load out_ptr "vec_ptr" copy_vec_elem_builder in
              let dest_ptr = L.build_gep vec_ptr [|iter_val|] "dest_ptr" copy_vec_elem_builder in
              let origin_ptr = L.build_gep ptr [|iter_val|] "origin_ptr" copy_vec_elem_builder in
              let (newvec, newbuilder) = vector_copy origin_ptr copy_vec_elem_builder typ in
                    ignore (L.build_store newvec dest_ptr newbuilder) ;
                    let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                    ignore (L.build_store new_iter_val iter newbuilder);
                    let merge_bb = L.append_block context "merge" the_function in
                      let merge_builder = L.builder_at_end context merge_bb in
                      ignore (L.build_cond_br zerosize zerosize_bb nonzerosize_bb builder) ;
                      ignore (L.build_br pred_bb zerosize_builder);
                      ignore (L.build_br pred_bb nonzerosize_builder);
                      ignore (L.build_cond_br keeplooping copy_vec_elem_bb merge_bb pred_builder);
                      ignore (L.build_br pred_bb newbuilder);
                      (L.build_load out_vec "out_vec_val" merge_builder, merge_builder)
      | A.Struct(typ) -> let ptr_p = L.build_gep vec [|zeroth; zeroth|] "ptr_p" builder in
            let size_p = L.build_gep vec [|zeroth; first|] "size_p" builder in
            let len_p = L.build_gep vec [|zeroth; second|] "len_p" builder in
            let ptr = L.build_load ptr_p "ptr" builder in
            let size = L.build_load size_p "size" builder in
            let len = L.build_load len_p "len" builder in
            let iter = L.build_alloca i32_t "iter" builder in
            ignore (L.build_store (L.const_int i32_t 0) iter builder) ;
            let out_vec = L.build_alloca vec_t "out_vec" builder in
            let out_ptr = L.build_gep out_vec [|zeroth; zeroth|] "out_ptr" builder in
            let out_size = L.build_gep out_vec [|zeroth; first|] "out_size" builder in
            let out_len = L.build_gep out_vec [|zeroth; second|] "out_len" builder in
            ignore (L.build_store size out_size builder) ;
            ignore (L.build_store len out_len builder) ;
            let zerosize = L.build_icmp L.Icmp.Slt size (L.const_int i32_t 1) "iszerosize" builder in
            let zerosize_bb = L.append_block context "zerosize" the_function in
              let zerosize_builder = L.builder_at_end context zerosize_bb in
              ignore (L.build_store (L.const_pointer_null (L.pointer_type vec_t)) out_ptr zerosize_builder) ;
            let nonzerosize_bb = L.append_block context "nonzerosize" the_function in
              let nonzerosize_builder = L.builder_at_end context nonzerosize_bb in


              let newvec = L.build_array_malloc (ltype_of_typ (A.Struct(typ))) len "newvec" nonzerosize_builder in
              (*vec_t **)
              let newvec_cast = L.build_bitcast newvec (L.pointer_type vec_t) "newvec_cast" nonzerosize_builder in
              (*output_ptr -> vect_t** *)
              ignore (L.build_store newvec_cast out_ptr nonzerosize_builder) ;
            let pred_bb = L.append_block context "check_size" the_function in
            let pred_builder = L.builder_at_end context pred_bb in
              let iter_val = L.build_load iter "iter_val" pred_builder in
              let keeplooping = L.build_icmp L.Icmp.Slt iter_val size "keeplooping" pred_builder in
               let copy_vec_elem_bb = L.append_block context "copy_vec_elem" the_function in
              let copy_vec_elem_builder = L.builder_at_end context copy_vec_elem_bb in
              let iter_val = L.build_load iter "iter_val" copy_vec_elem_builder in

              (* vec_t* *)
              let vec_ptr = L.build_load out_ptr "vec_ptr" copy_vec_elem_builder in
              let struct_vec_ptr = L.build_bitcast vec_ptr (L.pointer_type (ltype_of_typ (A.Struct(typ)))) "struct_vec_ptr" copy_vec_elem_builder in
              let struct_ptr = L.build_bitcast ptr (L.pointer_type (ltype_of_typ (A.Struct(typ)))) "struct_vec_ptr" copy_vec_elem_builder in

              (* struct xyz* *)
              let dest_ptr = L.build_gep struct_vec_ptr [|iter_val|] "dest_ptr" copy_vec_elem_builder in
              let origin_ptr = L.build_gep struct_ptr [|iter_val|] "origin_ptr" copy_vec_elem_builder in
        let struct_def_map = StringMap.find typ global_map in
        let newbuilder = struct_copy dest_ptr origin_ptr struct_def_map copy_vec_elem_builder in
        let new_iter_val = L.build_add iter_val (L.const_int i32_t 1) "new_iter_val" newbuilder in
                    ignore (L.build_store new_iter_val iter newbuilder);
                    let merge_bb = L.append_block context "merge" the_function in
                      let merge_builder = L.builder_at_end context merge_bb in
                      ignore (L.build_cond_br zerosize zerosize_bb nonzerosize_bb builder) ;
                      ignore (L.build_br pred_bb zerosize_builder);
                      ignore (L.build_br pred_bb nonzerosize_builder);
                      ignore (L.build_cond_br keeplooping copy_vec_elem_bb merge_bb pred_builder);
                      ignore (L.build_br pred_bb newbuilder);
                      (L.build_load out_vec "out_vec_val" merge_builder, merge_builder)



      | typ -> let ptr_p = L.build_gep vec [|zeroth; zeroth|] "ptr_p" builder in
            let size_p = L.build_gep vec [|zeroth; first|] "size_p" builder in
            let len_p = L.build_gep vec [|zeroth; second|] "len_p" builder in
            let ptr = L.build_load ptr_p "ptr" builder in
            let size = L.build_load size_p "size" builder in
            let len = L.build_load len_p "len" builder in
            let out_vec = L.build_alloca vec_t "out_vec" builder in
            let out_ptr = L.build_gep out_vec [|zeroth; zeroth|] "out_ptr" builder in
            let out_size = L.build_gep out_vec [|zeroth; first|] "out_size" builder in
            let out_len = L.build_gep out_vec [|zeroth; second|] "out_len" builder in
            ignore (L.build_store size out_size builder) ;
            ignore (L.build_store len out_len builder) ;
            let iszerosize = L.build_icmp L.Icmp.Slt size (L.const_int i32_t 1) "iszerosize" builder in
            let zerosize_bb = L.append_block context "zerosize" the_function in
              let zerosize_builder = L.builder_at_end context zerosize_bb in
              ignore (L.build_store (L.const_pointer_null (L.pointer_type vec_t)) out_ptr zerosize_builder) ;
            let nonzerosize_bb = L.append_block context "nonzerosize" the_function in
              let nonzerosize_builder = L.builder_at_end context nonzerosize_bb in
              let newvec = L.build_array_malloc (ltype_of_typ typ) len "newvec" nonzerosize_builder in
              let newvec_cast = L.build_bitcast newvec (L.pointer_type vec_t) "newvec_cast" nonzerosize_builder in
              ignore (L.build_store newvec_cast out_ptr nonzerosize_builder) ;
              let src_cast = L.build_bitcast ptr (L.pointer_type i8_t) "src_cast" nonzerosize_builder in
              let dst_cast = L.build_bitcast newvec (L.pointer_type i8_t) "dst_cast" nonzerosize_builder in
              let sizeof = L.build_trunc (L.size_of (ltype_of_typ typ)) i32_t "sizeof" nonzerosize_builder in
              let sz_to_copy = L.build_mul size sizeof "sz_to_copy" nonzerosize_builder in
              ignore (L.build_call memcpy_func [|dst_cast; src_cast; sz_to_copy; L.const_int i32_t 1; L.const_int i1_t 1|] "" nonzerosize_builder) ;
            let merge_bb = L.append_block context "merge" the_function in
              let merge_builder = L.builder_at_end context merge_bb in
            ignore(L.build_br merge_bb zerosize_builder);
            ignore(L.build_br merge_bb nonzerosize_builder);
            ignore(L.build_cond_br iszerosize zerosize_bb nonzerosize_bb builder);
            (L.build_load out_vec "out_vec_val" merge_builder, merge_builder) in

        (* takes a builder and a function:
        if current block is terminator, do not go to merge block
        else go to merge block
        if we have a "return" in a if block, we have to know if we want to
         add a terminator block.
       *)
   let add_terminal builder f =
      match L.block_terminator (L.insertion_block builder) with
        Some _ -> ()
      | None -> ignore (f builder) in
    let vector_access need_to_load ptr typ indices builder =
      let rec vector_access_helper ptr typ indices = (match (typ, indices) with
            (_, []) -> ptr
          | (A.Vector(in_typ), i::tl) -> let new_vec_p = L.build_gep ptr [|zeroth;zeroth|] "vect_gep" builder in
                                         let new_vec = L.build_load new_vec_p "vec_load" builder in
                                         let new_vec_elem = L.build_gep new_vec [|i|] "elem" builder in
                                         vector_access_helper new_vec_elem in_typ tl
          | (final_type, i::_) -> let elem_arr_p = L.build_gep ptr [|zeroth;zeroth|] "elem_arr_p" builder in
                                   let elem_arr = L.build_load elem_arr_p "elem_arr_load" builder in
                                   let elem_arr_cast = L.build_bitcast elem_arr (L.pointer_type (ltype_of_typ final_type)) "elem_arr_cast" builder in
                                   L.build_gep elem_arr_cast [|i|] "elem" builder) in
        let target_ptr = vector_access_helper ptr typ indices in
      if need_to_load then L.build_load target_ptr "element" builder
      else target_ptr in

     (* Construct code for an expression; return its value
      Always attach the expression to the latest builder.
      Do not create a new block. Block creation happens at
      statement gen for Block type.
   *)
    let rec get_struct_access_field_info = function
        A.Id s ->  (A.Struct(get_struct_type s), -1)
      | A.StructFieldAccess(e, field_name) ->
        let struct_type_name = (match (get_struct_access_field_info e) with
                | A.Struct(name), _ -> name
                | _ -> raise(Failure("accessing field of something not a struct"))) in
        let struct_def_map = StringMap.find struct_type_name global_map  in
        StringMap.find field_name struct_def_map (* returns (typ, idx) of field_name*)
      | A.VectorAccess(id, _) -> let rec walk_back_type = function
                A.Vector(typ) -> walk_back_type typ
              | typ -> typ in
            (walk_back_type (get_vector_type id), -1)
      | _ -> raise(Failure("accessing field of something not a struct")) in


      let rec struct_copy_assign lptr rptr struct_def_map builder delete_vec =
        let bindings = StringMap.bindings struct_def_map in
        let struct_field_assign_helper builder binding  =
          let (_, (fieldtyp, idx)) = binding in
            (match fieldtyp with
              A.Vector(in_typ) ->
                  let lvec_ptr = L.build_struct_gep lptr idx "l_field_vec_cp_assign" builder in
                  let rvec_ptr = L.build_struct_gep rptr idx "r_field_vec_cp_assign" builder in
                  let load_rptr = L.build_load rvec_ptr "loaded_rptr" builder in
                  let builder3 = vector_delete lvec_ptr builder the_function in_typ in
                  let temp = L.build_alloca vec_t "temp" builder3 in
                  let _ = L.build_store load_rptr temp builder3 in
                  if (delete_vec)
                  then

                    let (vec_cpy, builder4) = vector_copy temp builder3 in_typ in
                    let builder5 = vector_delete rvec_ptr builder4 the_function in_typ in
                      ignore(L.build_store vec_cpy lvec_ptr builder5); builder5
                  else
                  (*
                    raise(Failure (L.string_of_llvalue((lvec_ptr)) ^ "; " ^ L.string_of_llvalue((rvec_ptr))))
                      *)
                    let (vec_cpy, builder4) = vector_copy temp builder3 in_typ in
                      ignore(L.build_store vec_cpy lvec_ptr builder4); builder4

                | A.Struct(in_typ) ->
                  (* %struct_field = getelementptr inbounds %example* %b, i32 0, i32 2;
                     %struct_field1 = getelementptr inbounds %example* %a, i32 0, i32 2
                     raise(Failure (L.string_of_lltype((ltype_of_typ fieldtyp)) ^ " " ^ field_name))
                   *)


                    let l_field_val = L.build_struct_gep lptr idx "l_inner_struct_field" builder in

                    let r_field_val = L.build_struct_gep rptr idx "r_inner_struct_field" builder in

                    let field_struct_def_map = StringMap.find in_typ global_map  in
                    let (_, builder) = struct_copy_assign l_field_val r_field_val field_struct_def_map builder delete_vec in builder;

                | _ ->
                    let l_field_ptr = L.build_struct_gep lptr idx "l_field_cp_assign" builder in
                    let r_field_ptr = L.build_struct_gep rptr idx "r_field_cp_assign" builder in
                    let load_rptr = L.build_load r_field_ptr "loaded_rptr" builder in
                    ignore(L.build_store load_rptr l_field_ptr builder); builder) in
        let builder = List.fold_left struct_field_assign_helper builder bindings in (lptr, builder);
    in

		let rec expr builder = function
			  A.Literal i -> (L.const_int i32_t i, builder)
			| A.DoubleLit f -> (L.const_float double_type f, builder)
      | A.StringLit s -> let result = string_decl s builder in
                let vect1 = L.build_alloca vec_t "gc_tmp_vect1" builder in
                let _ = L.build_store result vect1 builder in
                let _ = add_to_string_lit_cleanup vect1 in (result, builder)
			| A.BoolLit b -> (L.const_int i1_t (if b then 1 else 0), builder)
			| A.Noexpr -> (L.const_int i32_t 0, builder)
            | A.Id s -> ((if (Hashtbl.mem struct_type_map s) then lookup s else L.build_load (lookup s) s builder), builder)
			| A.Binop (e1, op, e2) ->
				let (e1', builder') = expr builder e1 in
				let (e2', builder'') = expr builder' e2 in
				let exp_type1 = L.classify_type(L.type_of e1') in
				(*let exp_type2 =*) ignore (L.classify_type(L.type_of e2'));
				((match exp_type1 with
					| L.TypeKind.Double ->
		               (match op with
		                  A.Add     -> L.build_fadd
		                | A.Sub     -> L.build_fsub
		                | A.Mult    -> L.build_fmul
		                | A.Div     -> L.build_fdiv
		                | A.And     -> L.build_and
		                | A.Or      -> L.build_or
		                | A.Equal   -> L.build_fcmp L.Fcmp.Oeq
		                | A.Neq     -> L.build_fcmp L.Fcmp.One
		                | A.Less    -> L.build_fcmp L.Fcmp.Ult
		                | A.Leq     -> L.build_fcmp L.Fcmp.Ole
		                | A.Greater -> L.build_fcmp L.Fcmp.Ogt
		                | A.Geq     -> L.build_fcmp L.Fcmp.Oge
		               ) e1' e2' "tmp" builder''
					| _ ->
						(match op with
						  A.Add			-> L.build_add
						| A.Sub			-> L.build_sub
						| A.Mult		-> L.build_mul
						| A.Div			-> L.build_sdiv
						| A.And			-> L.build_and
						| A.Or			-> L.build_or
						| A.Equal		-> L.build_icmp L.Icmp.Eq
						| A.Neq			-> L.build_icmp L.Icmp.Ne
						| A.Less		-> L.build_icmp L.Icmp.Slt
						| A.Leq			-> L.build_icmp L.Icmp.Sle
						| A.Greater 	-> L.build_icmp L.Icmp.Sgt
						| A.Geq			-> L.build_icmp L.Icmp.Sge
						) e1' e2' "tmp" builder''
				), builder'')

			| A.Unop(op, e) ->
				let (e', builder') = expr builder e in
				((match op with
					A.Neg			-> L.build_neg
				  | A.Not			-> L.build_not) e' "tmp" builder', builder')
			| A.Assign (s, e) -> if ((is_vector s) && (match e with A.Call(_,_) -> false | A.Get(_) -> false | _ -> true))
                                 then (let (vec_ptr, vec_typ, builder'') = (match e with
                                         A.Id(id) -> (lookup id, get_vector_type id, builder)
                                       | e -> let temp = L.build_alloca vec_t "temp" builder in
                                              let (e', b') = expr builder e in
                                              ignore (L.build_store e' temp b');
                                              (temp, get_vector_type s, b')) in
                                 let builder3 = vector_delete (lookup s) builder'' the_function (get_vector_type s) in
                                 let (vec_cpy, builder4) = vector_copy vec_ptr builder3 vec_typ in
                                 ignore (L.build_store vec_cpy (lookup s) builder4);
                                 expr builder4 e)
                          else (if (is_vector s)
                              then (let (e', builder') = expr builder e in
                              let builder'' = vector_delete (lookup s) builder' the_function (get_vector_type s) in
									            ignore (L.build_store e' (lookup s) builder''); (e', builder''))
                          else (if ((is_struct s)  && (match e with A.StringLit(_) -> false | A.Call(_,_) -> false | A.Get(_) -> false | _ -> true))
                              then let (e', builder') = expr builder e in
                              let struct_type = get_struct_type s in
                              let struct_def_map = StringMap.find struct_type global_map  in
                              let lptr = lookup s in
                              (*raise(Failure (L.string_of_llvalue(lptr) ^ ";" ^ L.string_of_llvalue(e')))*)
                              struct_copy_assign lptr e' struct_def_map builder' false
                          else (if (is_struct s) then (let (e', builder') = expr builder e in
                                  let struct_type = get_struct_type s in
                                  let field_struct_def_map = StringMap.find struct_type global_map  in
                                  let builder'' = struct_delete (lookup s) field_struct_def_map builder' the_function in
                                  ignore (L.build_store e' (lookup s) builder''); (e', builder''))

                            else (let (e', builder') = expr builder e in
                            ignore (L.build_store e' (lookup s) builder'); (e', builder')))))
      | A.VectorAccess(id, list_e) -> let (idx_list, builder') =
          List.fold_left (fun (accumulate, b) arg -> let (e', b') = (expr b arg) in (e'::accumulate, b')) ([], builder) list_e in
          let ptr = lookup id in
          let typ = get_vector_type id in
          (match typ with
            A.Struct(_) -> (vector_access false ptr typ idx_list builder', builder')
          | _ -> (vector_access true ptr typ idx_list builder', builder'))


      | A.VectorAssign(id, e1_list, e2) -> let (idx_list_rev, builder') =
          (List.fold_left (fun (accumulate, b) e -> let (e', b') = (expr b e) in (e'::accumulate, b')) ([], builder) e1_list) in
          let idx_list = List.rev idx_list_rev in
          let typ = get_vector_type id in
		  let rec walk_back_type = function
                (0, typ) -> typ
              | (n, A.Vector(typ)) -> walk_back_type (n - 1, typ)
              | _ -> raise(Failure("Walking back type of something not a vector")) in
          let innertyp = walk_back_type ((List.length idx_list) - 1, typ) in
          (match innertyp with
              A.Vector(_) when (match e2 with A.Call(_,_) -> false | A.Get(_) -> false | _ -> true) ->
                    let (vec_ptr, vec_typ, builder'') = (match e2 with
                         A.Id(id2) -> (lookup id2, get_vector_type id2, builder)
                       | e -> let temp = L.build_alloca vec_t "temp" builder in
                              let (e', b') = expr builder e in
                              ignore (L.build_store e' temp b');
                              (temp, get_vector_type id, b')) in
                    let element = vector_access false (lookup id) typ idx_list builder'' in
                    let builder3 = vector_delete element builder'' the_function vec_typ in
                    let (vec_cpy, builder4) = vector_copy vec_ptr builder3 vec_typ in
                    ignore (L.build_store vec_cpy element builder4);
                    expr builder4 e2
            | A.Vector(_)  -> let element = vector_access false (lookup id) typ idx_list builder' in
                   let builder'' = vector_delete element builder' the_function innertyp in
                   let (right_hand_val, builder3) = expr builder'' e2 in
                   ignore (L.build_store right_hand_val element builder3);
                   (right_hand_val, builder3)
            | A.Struct(in_typ) when (match e2 with A.Call(_,_) -> false | A.Get(_) -> false | _ -> true) ->
                   let element = vector_access false (lookup id) typ idx_list builder' in
                   let (right_hand_val, builder'') = expr builder' e2 in
                   let field_struct_def_map = StringMap.find in_typ global_map  in
                   let builder3 = struct_delete element field_struct_def_map builder'' the_function in
                   let builder4 = struct_copy element right_hand_val field_struct_def_map builder3 in
                   (right_hand_val, builder4)
            | A.Struct(in_typ) -> 
                   let element = vector_access false (lookup id) typ idx_list builder' in
                   let (right_hand_val, builder'') = expr builder' e2 in
                   let field_struct_def_map = StringMap.find in_typ global_map  in
                   let builder3 = struct_delete element field_struct_def_map builder'' the_function in
                   ignore (L.build_store right_hand_val element builder3);
                   (right_hand_val, builder3)
            | _ -> let element = vector_access false (lookup id) typ idx_list builder' in
                   let (right_hand_val, builder'') = expr builder' e2 in
                   ignore (L.build_store right_hand_val element builder'');
                   (right_hand_val, builder''))
      | A.VectorSize(e) ->
          let (vector_val, builder'') = (match e with
                  A.Id(id) -> (lookup id, builder)
                | A.VectorAccess(id, list_e) -> let (idx_list, builder') =
                        List.fold_left (fun (accumulate, b) arg -> let (e', b') = (expr b arg) in (e'::accumulate, b')) ([], builder) list_e in
                        let ptr = lookup id in
                        let typ = get_vector_type id in
                        (vector_access false ptr typ idx_list builder', builder')
                | _ -> raise(Failure("trying to take the size of something not a vector"))) in
          let size_p = L.build_gep vector_val [|zeroth;first|] "size_p" builder'' in
          (L.build_load size_p "size" builder'', builder'')
      | A.Concat(e1, e2) -> let (result2, builder) = expr builder e2 in let (result1, builder) = expr builder e1 in

            let result = string_concat result1 result2 builder in
            let vect1 = L.build_alloca vec_t "gc_tmp_vect1" builder in
            let _ = L.build_store result vect1 builder in
            let _ = add_to_string_lit_cleanup vect1 in (result, builder)
      | A.Get(job) -> let typ = Hashtbl.find job_type_map job in
        let jid        = L.build_load (lookup job) job builder in
        let out_data   = L.build_alloca (L.pointer_type i8_t) "out_data" builder in
        let out_len    = L.build_alloca (i32_t) "out_len" builder in
        ignore(L.build_call reap_job_func [| jid; out_data; out_len |] "" builder) ;
        let out_data_val = L.build_load out_data "out_data_val" builder in
        let (lst, _, _, newbuilder) = deserialize ([], out_data, the_function, builder) typ in
        let retval = (match lst with [r] -> r | _ -> raise(Failure("multiple returns from job")) ) in
        ignore(L.build_free out_data_val  newbuilder) ;
        (retval, newbuilder)

      | A.Cancel(job) -> let jid = L.build_load (lookup job) job builder in
        (L.build_call cancel_job_func [| jid |] "cancel_job" builder, builder)

      | A.Running(job) -> let jid = L.build_load (lookup job) job builder in
        let status = L.build_call get_job_status_func [| jid |] "get_job_status" builder in
        (L.build_icmp L.Icmp.Sle (L.const_int i32_t 0) status "running" builder, builder)
      | A.Finished(job) -> let jid = L.build_load (lookup job) job builder in
        let status = L.build_call get_job_status_func [| jid |] "get_job_status" builder in
        (L.build_icmp L.Icmp.Eq (L.const_int i32_t (-1)) status "finished" builder, builder)
      | A.Failed(job) -> let jid = L.build_load (lookup job) job builder in
        let status = L.build_call get_job_status_func [| jid |] "get_job_status" builder in
        (L.build_icmp L.Icmp.Eq (L.const_int i32_t (-2)) status "failed" builder, builder)

      | A.RemoteCall (f, e) -> let (_, fdecl) = StringMap.find f function_decls in
        let (evals, nbuilder) = List.fold_left (fun (acc, b) e ->
                (match e with
                 A.Id(s) when (is_struct(s)) -> ((L.build_load (lookup s) "struct" b)::acc, b)
                | _ -> let (e', b') = expr b e in (e'::acc, b')))
             ([], builder) (List.rev e) in
        let typs = List.fold_left (fun acc formal -> let (typ, _) = formal in typ::acc) [] (List.rev fdecl.A.formals) in
        let (arg, len, builder') = serialize nbuilder evals the_function typs in
        let ord = get_ordinal f in
        (L.build_call start_job_func [| ord ; arg ; len |] "start_job" builder', builder')

      | A.StructFieldAccess(e, field_name) ->
        let struct_type_name = (match (get_struct_access_field_info e) with
                | A.Struct(name), _ -> name
                | _ -> raise(Failure("assigning to field of something not a struct"))) in

        let (struct_val, builder'') = (match e with
                    A.VectorAccess(id, list_e) -> let (idx_list, builder') =
                        List.fold_left (fun (accumulate, b) arg -> let (e', b') = (expr b arg) in (e'::accumulate, b')) ([], builder) list_e in
                        let ptr = lookup id in
                        let typ = get_vector_type id in
                        (vector_access false ptr typ idx_list builder', builder')
                  | e -> expr builder e) in

        let struct_def_map = StringMap.find struct_type_name global_map  in
        let (typ, field_idx) = StringMap.find field_name struct_def_map  in
        (*raise(Failure (L.string_of_llvalue (struct_val)))*)

          let ptr =  L.build_struct_gep struct_val field_idx "struct_field" builder'' in
          (( match typ with
              A.Struct(_) -> ptr
            | _ -> L.build_load ptr "struct_field_load" builder''
          ), builder'')
      | A.StructFieldAssign(e1, field_name, e) ->
        let (result, builder') = expr builder e  in
        let (struct_val, builder'') = expr builder' e1 in
        let struct_type_name = (match (get_struct_access_field_info e1) with
                | A.Struct(name), _ -> name
                | _ -> raise(Failure("assigning to field of something not a struct"))) in
        let struct_def_map = StringMap.find struct_type_name global_map  in
        let (typ, field_idx) = StringMap.find field_name struct_def_map  in
         let ptr = L.build_struct_gep struct_val field_idx "struct_field" builder'' in (*
          raise(Failure (L.string_of_llvalue (struct_val) ^ L.string_of_llvalue ptr))) *)
          (match typ with
          A.Struct(inner_type) ->

            (*
            let struct_val_cast = L.build_bitcast result (L.pointer_type i8_t) "struct_src_cast" builder'' in
            let ptr_cast = L.build_bitcast ptr (L.pointer_type i8_t) "struct_dest_cast" builder'' in
            let struct_size = L.size_of (ltype_of_typ typ) in
            (L.build_call memcpy64_func [|ptr_cast; struct_val_cast; struct_size; L.const_int i32_t 1; L.const_int i1_t 1|] "" builder'', builder'')
          *)
          let struct_def_map = StringMap.find inner_type global_map in
            struct_copy_assign ptr result struct_def_map builder'' false
          | A.Vector(inner_t) -> let builder3 = vector_delete ptr builder'' the_function inner_t in
              let temp = L.build_alloca vec_t "temp" builder3 in
              ignore (L.build_store result temp builder3) ;
              let (vec_cpy, builder4) = vector_copy temp builder3 inner_t in
              (L.build_store vec_cpy ptr builder4, builder4)

          | _ -> (L.build_store result ptr builder'', builder''))


      | A.Call ("print", [e])  -> let (e', b') = expr builder e in
				(L.build_call printf_func [| int_format_str ; e' |] "printf" b', b')

			| A.Call ("printb", [e]) -> let (e', b') = expr builder e in
				let bstring = L.build_select e' true_str false_str "select" b' in
				(L.build_call printf_func [| bstring |] "printf" b', b')

			| A.Call ("printd", [e]) -> let (e', b') = expr builder e in
				(L.build_call printf_func [| double_format_str ; e' |] "printf" b', b')
      | A.Call ("prints", [e]) -> let (e', b') = expr builder e in
        let vect1 = L.build_alloca vec_t "tmp_vect1" builder in
        let _ = L.build_store e' vect1 builder in
        let ptr1 = L.build_gep vect1 [|zeroth; zeroth|] "ptr_1" builder in
        let cast_vect1 = L.build_bitcast ptr1 (L.pointer_type (L.pointer_type i8_t)) "cast_vect1" builder in
        let load_cast_vect1 = L.build_load cast_vect1 "load_cast_vect1" builder in
        (L.build_call printf_func [| string_str ; load_cast_vect1 |] "printf" b', b')

			| A.Call (f, act) ->
				 let (fdef, fdecl) = StringMap.find f function_decls in
				 	(* needs to evaluate expr from right to left. int a = b = (a+1) *)

					let (actuals, _, builder') = List.fold_left (fun (acc, fmls, b) e -> let (e', b') = expr b e in
                                       (match fmls with
                                        (A.Vector(typ), _)::tl -> let tmp = L.build_alloca vec_t "tmp" b' in
                                                                  ignore (L.build_store e' tmp b');
                                                                  let (v_cpy, b'') = (match e with 
                                                                      A.Call(_,_) | A.RemoteCall(_,_) -> (L.build_load tmp "vec" b', b')
                                                                    | _ -> vector_copy tmp b' typ) in
                                                                  (v_cpy::acc, tl, b'')
                                        | (A.Struct(name), _)::tl -> let dst = L.build_alloca (ltype_of_typ (A.Struct(name))) "dst" b' in
                                                                     let field_struct_def_map = StringMap.find name global_map  in
                                                                     let nbuilder = (match e with
                                                                           A.Call(_,_) | A.RemoteCall(_,_) -> 
                                                                                ignore (L.build_store e' dst b'); b'
                                                                         | _ -> struct_copy dst e' field_struct_def_map b') in
                                                                     ((L.build_load dst "struct_copy" nbuilder)::acc, tl, nbuilder)
                                        | _::tl -> (e'::acc, tl, b')
                                        | [] -> raise(Failure("too many arguments!"))))
                             ([], (List.rev fdecl.A.formals), builder) (List.rev act)  in
					let result = (match fdecl.A.typ with A.Void -> "" | _ -> f ^ "_result") in
				 (L.build_call fdef (Array.of_list actuals) result builder', builder')
            | _ -> raise(Failure("unrecognized expr")) in

     (* make space for the return value *)
     let retval = (match fdecl.A.typ with
             A.Void -> L.const_null i32_t
           | typ -> let retval = L.build_alloca (ltype_of_typ typ) "retval" builder in
                    ignore (L.build_store (L.const_null (ltype_of_typ fdecl.A.typ)) retval builder) ;
                    retval ) in

     (* where to go right before returning *)
     let return_bb = L.append_block context "return_loc" the_function in
       let return_builder = L.builder_at_end context return_bb in

	 (* Build the code for the given statement; return the builder for
			 the statement's successor *)
		let rec stmt builder = function
	      A.Block sl -> List.fold_left stmt builder sl
			| A.Expr e -> let (_, b') = (expr builder e) in b'
      | A.VarDecl(typ,id) ->
          (match typ with
            A.Vector(inner_t) -> ignore(let vector_result = vector_decl(id, inner_t, builder) in
                                        (add_var_init id vector_result)); builder
            | A.Job(innertyp) -> ignore(add_var_init id (L.build_alloca (ltype_of_typ typ) id builder));
                                        ignore(add_job_type id innertyp); builder
            | A.Struct(struct_name) -> let _ = add_struct_type id struct_name in
                  let buf = L.build_alloca (ltype_of_typ typ) id builder in
                  ignore(L.build_store (L.const_null (ltype_of_typ typ)) buf builder);
                  ignore(add_var_init id buf); builder
            | _ -> ignore(add_var_init id (L.build_alloca (ltype_of_typ typ) id builder)); builder)
			| A.Return e -> let rec walk_back_type = function
                                 (0, typ) -> typ
                               | (n, A.Vector(typ)) -> walk_back_type (n - 1, typ)
                               | _ -> raise(Failure("Walking back type of something not a vector")) in
                (match fdecl.A.typ with
							A.Void -> ignore(L.build_br return_bb builder); builder
                          | A.Struct(in_typ) ->
                                  let (r_field_val, nbuilder) = expr builder e in
                                  let field_struct_def_map = StringMap.find in_typ global_map  in
                                  struct_copy retval r_field_val field_struct_def_map nbuilder
                          | A.Vector(typ) -> ( match e with
                                A.VectorAccess(id, lst) -> let (idx_list, builder') =
                                    List.fold_left (fun (accumulate, b) arg -> let (e', b') = (expr b arg) in (e'::accumulate, b')) ([], builder) lst in
                                    let ptr = lookup id in
                                    let innertyp = walk_back_type ((List.length lst), (get_vector_type id)) in
                                    let vec = vector_access false ptr (get_vector_type id) idx_list builder' in
                                    let (cpy, newbuilder) = vector_copy vec builder' innertyp in
                                    ignore (L.build_store cpy retval newbuilder) ; newbuilder
                              | A.Id(id) -> let (cpy, newbuilder) = vector_copy (lookup id) builder typ in
                                    ignore (L.build_store cpy retval newbuilder); newbuilder
                              | _ -> raise(Failure("trying to return a vector but expr doesn't seem to be a vector")))
					      | _ -> let (e', b') = expr builder e in
                                 ignore (L.build_store e' retval b');
                                 ignore (L.build_br return_bb b'); b');

      | A.VectorPushBack(e1, e) ->
          let (right_hand_val, builder'') = (match e with
              A.VectorAccess(id, lst) -> let (idx_list, builder') =
                  List.fold_left (fun (accumulate, b) arg -> let (e', b') = (expr b arg) in (e'::accumulate, b')) ([], builder) lst in
                  let ptr = lookup id in
                  let typ = get_vector_type id in
                  (vector_access true ptr typ idx_list builder', builder')
            | A.Id(id) when (is_vector id) -> (lookup id, builder)
            | A.Id(id) when (Hashtbl.mem struct_type_map id) -> let in_typ = get_struct_type id in
                    let l_field_val = L.build_alloca (ltype_of_typ (A.Struct(in_typ))) "new_struct" builder in
                    let r_field_val = lookup id in
                    let field_struct_def_map = StringMap.find in_typ global_map  in
                    let builder' = struct_copy l_field_val r_field_val field_struct_def_map builder in
                    (L.build_load l_field_val "new_struct_val" builder', builder')
            | _ -> expr builder e ) in
          (* we need to get typ of vector somehow -- how to do this with expr? *)
          let rec walk_back_type = function
              (0, typ) -> typ
            | (n, A.Vector(typ)) -> walk_back_type (n - 1, typ)
            | _ -> raise(Failure("Walking back type of something not a vector")) in
          let (vector_val, vector_typ, builder4) = (match e1 with
              A.VectorAccess(id, lst) -> let (idx_list, builder3) =
                  List.fold_left (fun (accumulate, b) arg -> let (e', b') = (expr b arg) in (e'::accumulate, b')) ([], builder'') lst in
                  let ptr = lookup id in
                  let typ = get_vector_type id in
                  ((vector_access false ptr typ idx_list builder3), (walk_back_type ((List.length lst), (get_vector_type id))), builder3)
            | A.Id(id) -> (lookup id, get_vector_type id, builder'')
            | _ -> raise(Failure("VectorPushBack on something not a vector")) ) in
          let size_p = L.build_gep vector_val [|zeroth; first|] "size_p" builder4 in
          let size_var = L.build_load size_p "size_var" builder4 in
          let len_p = L.build_gep vector_val [|zeroth; second|] "len_p" builder4 in
          let len_var = L.build_load len_p "leng_var" builder4 in
          let should_resize = L.build_icmp L.Icmp.Slt size_var len_var "size_var_cmp_leng" builder4 in

          let resize_bb = L.append_block context "resize" the_function in
            let resize_builder = L.builder_at_end context resize_bb in
            let resize_zero = L.build_icmp L.Icmp.Slt len_var (L.const_int i32_t 1) "size_var_nonzero" resize_builder in
          let resize_zero_bb = L.append_block context "resize_zero" the_function in
            let resize_zero_builder = L.builder_at_end context resize_zero_bb in
            ignore(L.build_store (L.const_int i32_t 1) len_p resize_zero_builder) ;
          let resize_nonzero_bb = L.append_block context "resize_nonzero" the_function in
            let resize_nonzero_builder = L.builder_at_end context resize_nonzero_bb in
            let newlen = L.build_shl len_var (L.const_int i32_t 1) "shl" resize_nonzero_builder in
            ignore(L.build_store newlen len_p resize_nonzero_builder) ;
          let resize_merge_bb = L.append_block context "resize_merge" the_function in
            let resize_merge_builder = L.builder_at_end context resize_merge_bb in
            let len_val = L.build_load len_p "new_len_val" resize_merge_builder in
            let ptr_p = L.build_gep vector_val [|zeroth;zeroth|] "ptr_p" resize_merge_builder in
            let ptr_val = L.build_load ptr_p "ptr_val" resize_merge_builder in
            let vec_mem = L.build_array_malloc (ltype_of_typ vector_typ) len_val "vec_mem" resize_merge_builder in
            let vec_mem_byte_cast = L.build_bitcast vec_mem (L.pointer_type i8_t) "vec_mem_byte_cast" resize_merge_builder in
            let ptr_val_byte_cast = L.build_bitcast ptr_val (L.pointer_type i8_t) "ptr_val_byte_cast" resize_merge_builder in
            let sizeof = L.build_trunc (L.size_of (ltype_of_typ vector_typ)) i32_t "sizeof" resize_merge_builder in
            let sz_to_copy = L.build_mul sizeof size_var "sz_to_copy" resize_merge_builder in
            ignore (L.build_call memcpy_func [|vec_mem_byte_cast; ptr_val_byte_cast; sz_to_copy; L.const_int i32_t 1; L.const_int i1_t 1|] "" resize_merge_builder );
            ignore (L.build_free ptr_val resize_merge_builder) ;
            let vec_mem_cast = L.build_bitcast vec_mem (L.pointer_type vec_t) "vec_mem_cast" resize_merge_builder in
            ignore (L.build_store vec_mem_cast ptr_p resize_merge_builder) ;
          let merge_bb = L.append_block context "pushback" the_function in
            let merge_builder = L.builder_at_end context merge_bb in
            let element = vector_access false vector_val vector_typ [size_var] merge_builder in
            let new_merge_builder = (match vector_typ with
                    A.Vector(intyp) -> let (cpy, new_merge_builder) = vector_copy right_hand_val merge_builder intyp in
                                       ignore (L.build_store cpy element new_merge_builder) ;
                                       new_merge_builder
                  | _ -> ignore (L.build_store right_hand_val element merge_builder) ;
                           merge_builder ) in
          let new_size = L.build_add size_var (L.const_int i32_t 1) "new_size" new_merge_builder in
          ignore (L.build_store new_size size_p new_merge_builder) ;
          ignore(L.build_cond_br should_resize merge_bb resize_bb builder4);
          ignore(L.build_cond_br resize_zero resize_zero_bb resize_nonzero_bb resize_builder);
          add_terminal resize_merge_builder (L.build_br merge_bb);
          add_terminal resize_zero_builder (L.build_br resize_merge_bb);
          add_terminal resize_nonzero_builder (L.build_br resize_merge_bb);
          new_merge_builder
      | A.If (predicate, then_stmt, else_stmt) ->
				 let (bool_val, builder') = expr builder predicate in
	 let merge_bb = L.append_block context "merge" the_function in

	 let then_bb = L.append_block context "then" the_function in
	 add_terminal (stmt (L.builder_at_end context then_bb) then_stmt)
		 (L.build_br merge_bb);

	 let else_bb = L.append_block context "else" the_function in
	 add_terminal (stmt (L.builder_at_end context else_bb) else_stmt)
		 (L.build_br merge_bb);

	 ignore (L.build_cond_br bool_val then_bb else_bb builder');
	 L.builder_at_end context merge_bb

			| A.While (predicate, body) ->
		let pred_bb = L.append_block context "while" the_function in
		ignore (L.build_br pred_bb builder);

		let body_bb = L.append_block context "while_body" the_function in
		add_terminal (stmt (L.builder_at_end context body_bb) body)
			(L.build_br pred_bb);

		let pred_builder = L.builder_at_end context pred_bb in
		let (bool_val, pred_builder') = expr pred_builder predicate in

		let merge_bb = L.append_block context "merge" the_function in
		ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder');
		L.builder_at_end context merge_bb

			| A.For (e1, e2, e3, body) -> stmt builder
			( A.Block [A.Expr e1 ; A.While (e2, A.Block [body ; A.Expr e3]) ] )
            | A.VarDeclAssign(typ, id, e) -> let b' = stmt builder (A.VarDecl(typ, id)) in
                let (_, b'') = expr b' (A.Assign(id, e)) in b''
		in

		(* Build the code for each statement in the function *)
		let builder = stmt builder (A.Block fdecl.A.body) in

		(* Add a "return" (that is, branch) if the last block falls off the end *)
		add_terminal builder (L.build_br return_bb) ;

        (* destroy all vectors *)
       let vects = Hashtbl.fold (fun s t l -> (s,t)::l) vector_type_map [] in
       let finalbuilder = List.fold_left (fun b (s,t) ->
        (*
        raise(Failure (L.string_of_llvalue((lookup s)) ^ "; " ^ L.string_of_lltype(ltype_of_typ t) ^ "; "))
          *)
          vector_delete (lookup s) b the_function t

        ) return_builder vects in

       let literals = Hashtbl.fold (fun s _ l -> s::l) string_lit_map [] in
       let reallyfinalbuilder = List.fold_left (fun b s ->
          (*
          raise(Failure (L.string_of_llvalue(s) ^ "; "))
          *)

          vector_delete s b the_function A.Char

        ) finalbuilder literals in


       let structs = Hashtbl.fold (fun s t l -> (s,t)::l) struct_type_map [] in

       let rec delete_vecs_of_fields ptr struct_def_map builder the_function =
              let bindings = StringMap.bindings struct_def_map in
              let delete_vecs_of_fields_helper builder binding =
                  let (_, (fieldtyp, idx)) = binding in
                  (match fieldtyp with
                    A.Vector(in_typ) -> let target_ptr = L.build_struct_gep ptr idx "vec" builder in
                        vector_delete target_ptr builder the_function in_typ
                    | A.Struct(in_typ) -> let target_ptr = L.build_struct_gep ptr idx "inner_struct_field" builder in
                        let field_struct_def_map = StringMap.find in_typ global_map  in
                        delete_vecs_of_fields target_ptr field_struct_def_map builder the_function
                    | _ -> builder)
              in
              List.fold_left delete_vecs_of_fields_helper builder bindings

        in
        let delete_vecs_of_struct builder strct =
          let (id, typ) = strct in
           let ptr = lookup id in
           let struct_def_map = StringMap.find typ global_map in
           delete_vecs_of_fields ptr struct_def_map builder the_function
        in
        let reallyfinalbuilder = List.fold_left delete_vecs_of_struct reallyfinalbuilder structs in

       (* execute the actual return *)
       add_terminal reallyfinalbuilder (if (fdecl.A.typ = A.Void) then L.build_ret_void
                                    else (L.build_ret (L.build_load retval "final_retval" reallyfinalbuilder))) ;

	in
	let _ = List.iter build_function_body functions in

	let job_decls =
		let job_decl m jobdecl =
		match jobdecl.A.fname with
		"master" -> m
		| _ ->
			let job_name = jobdecl.A.fname ^ "_job"
			in let void_pt = L.pointer_type i8_t
			in let void_ppt = L.pointer_type void_pt
			in let job_formal_type = [|void_pt; i32_t; void_ppt; L.pointer_type i32_t|]
		in let jobtype = L.function_type void_t job_formal_type in
			StringMap.add job_name (L.define_function job_name jobtype the_module, jobdecl) m in
		List.fold_left job_decl StringMap.empty functions in

		let build_job_body jobdecl =
			match jobdecl.A.fname with
			"master" -> ()
			| _ ->
			let (the_job_function, _) = StringMap.find (jobdecl.A.fname ^ "_job") job_decls in
			let (the_function, _) = StringMap.find jobdecl.A.fname function_decls in
		let builder = L.builder_at_end context (L.entry_block the_job_function) in

		(*set_value_name n p -> sets name in AST to equal to the name in llvm builder*)

		let add_formal n p = L.set_value_name n p in
		ignore (List.iter2 add_formal ["input_data"; "input_leng"; "output_data"; "output_leng"]
					(Array.to_list (L.params the_job_function))) ;
                let input_buf = L.param the_job_function 0 in
                let input_store = L.build_alloca (L.pointer_type i8_t) "input_store" builder in
                ignore (L.build_store input_buf input_store builder);
                let typs = List.map (fun (typ, _) -> typ) jobdecl.A.formals in
                let (arg_ints, _, _, newbuilder) = List.fold_left deserialize ([], input_store, the_job_function, builder) typs in
				let return_result = (match jobdecl.A.typ with A.Void -> "" | _ -> jobdecl.A.fname ^ "_result") in
				let tmp_call = L.build_call the_function (Array.of_list (List.rev arg_ints)) return_result newbuilder in
				let last_builder = (match jobdecl.A.typ with
                        A.Void -> ignore (L.build_store (L.const_null (L.pointer_type i8_t)) (L.param the_job_function 2) newbuilder) ;
                                  ignore (L.build_store (L.const_null i32_t) (L.param the_job_function 3) newbuilder);
                                  newbuilder
                      | typ -> let (buf, len, nnbuilder) = serialize newbuilder [tmp_call] the_job_function [typ] in
				               ignore (L.build_store buf (L.param the_job_function 2) nnbuilder) ;
				               ignore (L.build_store len (L.param the_job_function 3) nnbuilder);
                               (match typ with A.Vector(t) -> let vec_store = L.build_alloca vec_t "vec_store" nnbuilder in
                                                              ignore (L.build_store tmp_call vec_store nnbuilder) ;
                                                              vector_delete vec_store nnbuilder the_job_function t
                                               | A.Struct(n) -> let e_store = L.build_alloca (ltype_of_typ (A.Struct(n))) "e_store" nnbuilder in
                                                              ignore (L.build_store tmp_call e_store nnbuilder);
                                                              let field_struct_def_map = StringMap.find n global_map  in
                                                              struct_delete e_store field_struct_def_map nnbuilder the_job_function
                                                | _ -> nnbuilder)) in

				ignore(L.build_ret_void last_builder) in

             List.iter build_job_body functions ;

  let jobdecl_of_fdecl out_list = function
      x when (x.A.fname = "master") -> out_list
    | x -> let (a,_) = (StringMap.find (x.A.fname ^ "_job") job_decls) in
           a::out_list in
  let job_ptrs = List.fold_left jobdecl_of_fdecl [] (List.rev functions) in
  let job_func_type = L.function_type void_t [| (L.pointer_type i8_t) ; i32_t ;
    (L.pointer_type (L.pointer_type i8_t)); (L.pointer_type i32_t) |] in
  let job_ptr_arr = L.const_array (L.pointer_type job_func_type) (Array.of_list job_ptrs) in
  ignore (L.define_global "job_funcs" job_ptr_arr the_module) ;

  ignore (L.define_global "num_jobs" (L.const_int i32_t (List.length (StringMap.bindings job_decls))) the_module) ;

the_module
