#!/bin/sh

# Regression testing script for MatchaScript frontend (AST, scanner, parser)
# Step through a list of files

# Set time limit for all operations
ulimit -t 30

globallog=testall.log
rm -f $globallog
error=0
globalerror=0

keep=0

Usage() {
    echo "Usage: testfront.sh [options] [.ms files]"
    echo "-k    Keep intermediate files"
    echo "-h    Print this help"
    exit 1
}

SignalError() {
    if [ $error -eq 0 ] ; then
	echo "FAILED"
	error=1
    fi
    echo "  $1"
}

# Compare <outfile> <reffile> <difffile>
# Compares the outfile with reffile.  Differences, if any, written to difffile
Compare() {
    generatedfiles="$generatedfiles $3"
    echo diff -b $1 $2 ">" $3 1>&2
    diff -b "$1" "$2" > "$3" 2>&1 || {
	SignalError "$1 differs"
	echo "FAILED $1 differs from $2" 1>&2
    }
}

# Run <args>
# Report the command, run it, and report any errors
Run() {
    echo $* 1>&2
    eval $* || {
	SignalError "$1 failed on $*"
	return 1
    }
}

Check() {
    error=0
    basename=`echo $1 | sed 's/.*\\///
                             s/.ms//'`
    basedir="`echo $1 | sed 's/\/[^\/]*$//'`/."

    echo -n "$basename..."

    echo 1>&2
    echo "###### Parsing $basename" 1>&2

    generatedfiles=""

    # Make sure that the parsed tokens from the code are the same as the parsed tokens from the pretty printer
    generatedfiles="$generatedfiles ${basename}.pretty ${basename}.prettytokens" &&
    Run "ocaml frontend_tests/scannerprint.ml" "<" $1 ">" "frontend_tests/${basename}.tokens" &&
    sed -i '$s/\w*$//' frontend_tests/${basename}.tokens &&
    Run "./MatchaScript.native -a < tests/${basename}.ms > ${basename}.pretty" &&
    Run "ocaml frontend_tests/scannerprint.ml < ${basename}.pretty > ${basename}.prettytokens" &&
    sed -i '$s/\w*$//' ${basename}.prettytokens &&
    # Compare tokens generated by original program and tokens generated by pretty-printed AST
    Compare ${basename}.prettytokens frontend_tests/${basename}.tokens ${basename}.diff

    # Report the status and clean up the generated files

    if [ $error -eq 0 ] ; then
	if [ $keep -eq 0 ] ; then
	    rm -f $generatedfiles
	fi
	echo "OK"
	echo "###### SUCCESS" 1>&2
    else
	echo "###### FAILED" 1>&2
	globalerror=$error
    fi
}

while getopts kdpsh c; do
    case $c in
	k) # Keep intermediate files
	    keep=1
	    ;;
	h) # Help
	    Usage
	    ;;
    esac
done

shift `expr $OPTIND - 1`

ocamllex frontend_tests/scannerprint.mll

if [ $# -ge 1 ]
then
    files=$@
else
    files="tests/test-*.ms"
fi

for file in $files
do
    case $file in
	*test-*)
	    Check $file 2>> $globallog
	    ;;
	*)
	    echo "unknown file type $file"
	    globalerror=1
	    ;;
    esac
done

exit $globalerror
