(* MatCV scanner *)

{ open Parser }

rule token = parse
|  [' ' '\t' '\r' '\n'] { token lexbuf } (* Whitespace *)
| "/*"     { comment 0 lexbuf }           (* Comments *)
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| '['      { LSQBRACKET }
| ']'      { RSQBRACKET }
| ':'      { COLON }
| ';'      { SEMI }
| ','      { COMMA }
| '.'      { DOT }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| "+."     { MATPLUS }
| "-."     { MATMINUS }
| "*."     { MATTIMES }
| "/."     { MATDIVIDE }
| '%'      { MOD }
| '^'      { EXP }
| '='      { ASSIGN }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&&"     { AND }
| "||"     { OR }
| "!"      { NOT }
| "row"     { ROW }
| "col"     { COL }
| "ele"     { ELE }
| "pixel"    { PIXEL }
| "var"     { VARKEYWORD }
| "const"     { CONSTANT }
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "break"    { BREAK }
| "continue"    { CONTINUE }
| "exit"    { EXIT }
| "while"  { WHILE }
| "return" { RETURN }
| "function" { FUNCTION }
| "true"   { TRUE }
| "false"  { FALSE }
| ['0'-'9']+ as lexeme { LITERAL(int_of_string lexeme) }
| ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']* as lexeme { ID(lexeme) }
| eof { EOF }
| _ as char { raise (Failure("Illegal Character: " ^ Char.escaped char)) }

(* For nested comments *)
and comment level = parse
| "/*" {comment (level + 1) lexbuf}
| "*/" {  if level = 0 then token lexbuf else comment (level - 1) lexbuf}
| _    { comment level lexbuf }
