(*
 * Code generation for MatCV
*)

module L = Llvm
module A = Ast

let printError message =
    print_string("\nError: " ^ message); exit 1


let printWarning message =
    print_string ("\nWarning: " ^ message)



let mergeSymbolTables globalSymbolTable localSymbolTable = let mergedSymbolTable = Hashtbl.create 100 in
        let _ = Hashtbl.iter (fun key value -> Hashtbl.add mergedSymbolTable key value) localSymbolTable
        in
        let _ = Hashtbl.iter (fun key value -> if not (Hashtbl.mem mergedSymbolTable key) then Hashtbl.add mergedSymbolTable key value) globalSymbolTable
        in mergedSymbolTable



let translate (gstatements, functions) =
  let context = L.global_context () in
  let the_module = L.create_module context "MatCV"
  and i32_t  = L.i32_type  context
  and i8_t   = L.i8_type   context
  and i1_t   = L.i1_type   context
  and void_t = L.void_type context
  and mat_t = L.pointer_type (L.i32_type context) in

  let ltypeOfType = function
      A.Int -> i32_t
    | A.Bool -> i1_t
    | A.Mat(_) -> mat_t
    | A.Void -> void_t
    | A.Annotation _ -> printError "Unable to resolve symbols"
    | A.Func | A.FuncSignature (_, _)| A.Empty| A.Keyword -> printError "Error during compilation. Should have caught this in semantic check."
  in 

  let initOfType = function
    | A.Mat(_) -> (L.const_pointer_null mat_t)
    | t -> L.const_int (ltypeOfType t) 0


  in


  let globalSymbolTable = Hashtbl.create 100 
  in
  let mainaexpr = A.ALiteral(0, A.Int)
  in let mainreturnstatement = A.AReturn(A.Int, mainaexpr, A.Void)
  in let gstatements = gstatements @ [(mainreturnstatement)]
  in
  (* Generate code for global statements *)
  let functions = ({A.afname = ("main", A.Func); A.aformals = [] ; A.abody = gstatements; A.retType = A.Int}) :: functions
  in

  let createGlobalVar id idType symbolTable =  let _ = if not (Hashtbl.mem symbolTable id) then
                let init = initOfType idType 
                in
                Hashtbl.add symbolTable id ((L.define_global id init the_module), idType)  in ()


  in

  let declareGlobalVariableUsingStatement symbolTable = function 
      | A.AVarDecl(x, _) -> (match x with

              | A.AMatrix(id, idType, _, _, _, _) -> createGlobalVar id idType symbolTable 
              | A.AExprAssign(id, idType, _, _) -> createGlobalVar id idType symbolTable 
              | A.ADimAssign(id, idType, _, _, _) -> createGlobalVar id idType symbolTable 

              | _ -> ()
      )
              
      | _ -> ()

  in

  (* Generate code for declaration of global variables *)
  let _ = List.iter (declareGlobalVariableUsingStatement globalSymbolTable) gstatements 

  in

  let functionTable = Hashtbl.create 100
  in

  let functionDecl afunc =
      let name = (fst afunc.A.afname)
      and formals = Array.of_list (List.map (fun (_, typ) -> ltypeOfType typ) afunc.A.aformals)
      in let ftype = L.function_type (ltypeOfType afunc.A.retType) formals in
      Hashtbl.add functionTable name (L.define_function name ftype the_module, afunc)
      in
      let _ = List.iter functionDecl functions 
  in


  (* Generate code for forward declaration of functions *)

  (* TODO: Uncomment the following lines *)

  let printf_t = L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
  let printf_func = L.declare_function "printf" printf_t the_module

  in
  let copyMat_t = L.function_type i32_t [| L.pointer_type i32_t; L.pointer_type i32_t|] in
  let copyMat_func = L.declare_function "copyMat" copyMat_t the_module
  in

  let minusMat_t = L.function_type i32_t [| L.pointer_type i32_t; L.pointer_type i32_t|] in
  let minusMat_func = L.declare_function "minusMat" minusMat_t the_module
  in

  let addMat_t = L.function_type i32_t [| L.pointer_type i32_t; L.pointer_type i32_t|] in
  let addMat_func = L.declare_function "addMat" addMat_t the_module
  in

  let generateFunctionBody afunc = 
    let (theFunction, _) = Hashtbl.find functionTable (fst afunc.A.afname)
    in
    let builder = L.builder_at_end context (L.entry_block theFunction)
    in
    let int_format_str = L.build_global_stringptr "%d\n" "fmt" builder

    in
    let memoryMap = Hashtbl.create 100
    in
    let localSymbolTable = Hashtbl.create 100
    in

    let declareFormals (name, typ) value = 
        let _ = (L.set_value_name name value;
        let local = L.build_alloca (ltypeOfType typ) name builder
        in
        ignore (L.build_store value local builder);
        Hashtbl.add localSymbolTable name (local, typ)
        )
        in
        ()
    in
    let _ = List.iter2 declareFormals afunc.A.aformals (Array.to_list (L.params theFunction))
    in
    let lookup globalSymbolTable localSymbolTable name = if Hashtbl.mem localSymbolTable name then (fst (Hashtbl.find localSymbolTable name))
                      else (fst (Hashtbl.find globalSymbolTable name))
    in



    let rec genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder = function
      | A.ALiteral(value, _) -> L.const_int i32_t value
      | A.ABoolLit(value, _) -> L.const_int i1_t (if value then 1 else 0)
      | A.AId(id, _) -> L.build_load (lookup globalSymbolTable localSymbolTable id) id builder
      | A.AUnboundedAccessRead(id, idType, aexpr, _) -> let loadedLMatrix = L.build_load (lookup globalSymbolTable localSymbolTable id) id builder
      in let e = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr
      in
      let iPtr = (L.build_in_bounds_gep loadedLMatrix [|e|] (id ^ "_iPtr") builder)
      in
      L.build_load iPtr id builder

      | A.AUnboundedAccessWrite(id, idType, aexpr1, aexpr2, _) -> let loadedLMatrix = L.build_load (lookup globalSymbolTable localSymbolTable id) id builder
      in let e1 = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr1
      in let e2 = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr2
      in
      let iPtr = (L.build_in_bounds_gep loadedLMatrix [|e1|] (id ^ "_iPtr") builder)
      in
      let _ = L.build_store e2 iPtr builder
      in
      L.build_load iPtr id builder

     | A.AMatPlus(id1, idType1, id2, idType2, _) -> let loadedLMatrix1 = L.build_load (lookup globalSymbolTable localSymbolTable id1) id1 builder
     in let loadedLMatrix2 = L.build_load (lookup globalSymbolTable localSymbolTable id2) id2 builder
     in
     let _ = L.build_call addMat_func [|loadedLMatrix1; loadedLMatrix2|]
     "addMat" builder
     in
     loadedLMatrix1

     | A.AMatMinus(id1, idType1, id2, idType2, _) -> let loadedLMatrix1 = L.build_load (lookup globalSymbolTable localSymbolTable id1) id1 builder
     in let loadedLMatrix2 = L.build_load (lookup globalSymbolTable localSymbolTable id2) id2 builder
     in
     let _ = L.build_call minusMat_func [|loadedLMatrix1; loadedLMatrix2|]
     "minusMat" builder
     in
     loadedLMatrix1



      (*| A.Size(id, _, _) ->*)
      | A.ANoexpr(_) -> L.const_int i32_t 0
      | A.ABinaryOp(aexpr1, op, aexpr2, _) ->
              let e1 = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr1 in let e2 = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr2
              in
              (match op with
                      | A.Add     -> L.build_add
                      | A.Sub     -> L.build_sub
                      | A.Mul     -> L.build_mul
                      | A.Div     -> L.build_sdiv
                      | A.And     -> L.build_and
                      | A.Or      -> L.build_or
                      | A.Equal   -> L.build_icmp L.Icmp.Eq
                      | A.Neq     -> L.build_icmp L.Icmp.Ne
                      | A.Less    -> L.build_icmp L.Icmp.Slt
                      | A.Leq     -> L.build_icmp L.Icmp.Sle
                      | A.Greater -> L.build_icmp L.Icmp.Sgt
                      | A.Geq     -> L.build_icmp L.Icmp.Sge
                      | A.Mod     -> L.build_srem
                      (* TODO: Change the code for Exp
                       * Following code is just a placeholder
                       *)
                      | A.Exp     -> L.build_mul
               ) e1 e2 "tmp" builder
      | A.AUnop(uop, aexpr, exprType) ->
              let e = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr in
              (match uop with
              | A.Neg -> L.build_neg
              | A.Not -> L.build_not) e "tmp" builder
      | A.ACall ("print", _, [aexpr], _) | A.ACall ("printb", _, [aexpr], _) ->
      L.build_call printf_func [| int_format_str ; (genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr)|]
        "printf" builder
      | A.ACall(id, _, aExprList, _) -> let actuals = List.map (fun aexpr -> genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr) aExprList 
                                        in
                                        let (lfunc, afunc) = Hashtbl.find functionTable id
                                        in
                                        let retType = afunc.A.retType
                                        in
                                        let result = (match retType with 
                                                          | A.Void -> ""
                                                          | _ -> id ^ "_result") in
                                            L.build_call lfunc (Array.of_list actuals) result builder
    | A.AMatAccess(id, idType, aExprList, nDims, exprType) ->
                let dimIndices = List.map (fun aexpr -> genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr) aExprList 
                in
                let lMatrix = (lookup globalSymbolTable localSymbolTable id)
                in
                let loadedLMatrix = L.build_load lMatrix (id ^ "_load") builder
                in
                let rec findDimSizes dimSizeList = function
                        | 0 -> dimSizeList
                        | i -> findDimSizes (
                            (
                                let dimPtr = (L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t i|] (id ^ "_dim_" ^ string_of_int i) builder)
                                in
                                L.build_load dimPtr (id ^ "_dim_" ^ string_of_int i ^ "value_") builder

                            )
                            ::dimSizeList) (i - 1)

                in let dimSizes = findDimSizes [] nDims
                in
                let (index, _) = List.fold_left2 (fun (result, productDim) size_i index_i ->
                        let productDimMulIndex = (L.build_mul productDim index_i ("tmp_") builder)
                        in let result = (L.build_add result productDimMulIndex ("tmp2_") builder)
                        in
                        let productDim = (L.build_mul productDim size_i ("tmp3_") builder)
                        in
                        (result, productDim) )
               ((L.const_int i32_t (1 + nDims)), (L.const_int i32_t 1))
               (List.rev dimSizes) (List.rev dimIndices)
                in  
                let elementPtr = L.build_in_bounds_gep loadedLMatrix [|index|] (id ^ "_element") builder
                in
                L.build_load elementPtr (id ^ "_element") builder 

               
    in

  
    let freeMatrixFun lMatrix id builder =
          (* FREE: *)
          let freeMatrix = L.build_load lMatrix (id ^ "_free") builder
          in
          let _ = L.build_free freeMatrix builder
          in
          ()

    in


    let genCodeForVarDecl globalSymbolTable localSymbolTable memoryMap builder = function 
      | A.ANodecl(_) -> let _ = L.const_int i32_t 0 in ()
      | A.AMatrix(id, idType, aExprListList, nRows, nCols, _) ->
              let _ = (if Hashtbl.mem localSymbolTable id then
                          if Hashtbl.mem memoryMap id then
                              (* free, malloc *)
                              (* FREE: *)
                              let _ = freeMatrixFun  (lookup globalSymbolTable localSymbolTable id) id builder
                              in
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t (L.const_int i32_t (nRows * nCols + 3)) (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in ()
                          else
                              (* malloc, add to memory map *)
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t (L.const_int i32_t (nRows * nCols + 3)) (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in
                              (* Add to memory map *)
                              Hashtbl.add memoryMap id (lookup globalSymbolTable localSymbolTable id)

                      else if Hashtbl.mem globalSymbolTable id then
                              (* free, malloc *)
                              (* FREE: *)
                              let _ = freeMatrixFun  (lookup globalSymbolTable localSymbolTable id) id builder
                              in
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t (L.const_int i32_t (nRows * nCols + 3)) (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in
                              ()
                      else
                          (* alloc ptr, malloc, add to local map, add to memory map *)
                          (* Alloc *)
                          let lMatrix = L.build_alloca mat_t id builder
                          in
                          let _ = Hashtbl.add localSymbolTable id (lMatrix, idType)
                          in 
                          let _ = Hashtbl.add memoryMap id lMatrix
                          in
                          (* MALLOC *)
                          let mallocMatrix = L.build_array_malloc i32_t (L.const_int i32_t (nRows * nCols + 3)) (id ^ "_malloc") builder
                          in
                          let _ = L.build_store mallocMatrix lMatrix builder
                          in
                          ()
              ) in
              (* Assign dimensions and values *)
                let lMatrix = (lookup globalSymbolTable localSymbolTable id)
                in
                let loadedLMatrix = L.build_load lMatrix (id ^ "_load") builder
                in
                let zeroIndexPtr = L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t 0|] (id ^ "_zero_index") builder
                in
                let _ = L.build_store (L.const_int i32_t 2) zeroIndexPtr builder
                in
                let oneIndexPtr = L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t 1|] (id ^ "_one_index") builder
                in
                let _ = L.build_store (L.const_int i32_t nRows) oneIndexPtr builder
                in
                let secondIndexPtr = L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t 2|] (id ^ "_two_index") builder
                in
                let _ = L.build_store (L.const_int i32_t nCols) secondIndexPtr builder
                in
                let _ = List.fold_left (fun acc aExprList ->
          List.fold_left (fun acc aexpr -> let lvalue = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr in
          let ptrIndex = L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t acc|] (id ^ "_ptr_index") builder in
          let _ = L.build_store lvalue ptrIndex builder in
                          (acc + 1)) acc aExprList
            ) 3 aExprListList
              in
              ()

    | A.ADimAssign(id, idType, aExprList, nDims,  _) -> 
                            let (productSizes, dimSizeList) = List.fold_left (fun (acc, dimSizeLst)  aexpr -> let laexpr = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr in let dimLst = 
                                dimSizeLst @ [laexpr]
                            in
                            (L.build_mul acc laexpr "expr_prod_size" builder, dimLst)) (L.const_int i32_t 1, []) aExprList
    in
    let matrixSize = (L.build_add productSizes  (L.const_int i32_t (nDims + 1)) "mat_size" builder)
    in
              let _ = (if Hashtbl.mem localSymbolTable id then
                          if Hashtbl.mem memoryMap id then
                              (* free, malloc *)
                              (* FREE: *)
                              let _ = freeMatrixFun  (lookup globalSymbolTable localSymbolTable id) id builder
                              in
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in ()
                          else
                              (* malloc, add to memory map *)
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in
                              (* Add to memory map *)
                              Hashtbl.add memoryMap id (lookup globalSymbolTable localSymbolTable id)

                      else if Hashtbl.mem globalSymbolTable id then
                              (* free, malloc *)
                              (* FREE: *)
                              let _ = freeMatrixFun  (lookup globalSymbolTable localSymbolTable id) id builder
                              in
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in
                              ()
                      else
                          (* alloc ptr, malloc, add to local map, add to memory map *)
                          (* Alloc *)
                          let lMatrix = L.build_alloca mat_t id builder
                          in
                          let _ = Hashtbl.add localSymbolTable id (lMatrix, idType)
                          in 
                          let _ = Hashtbl.add memoryMap id lMatrix
                          in
                          (* MALLOC *)
                          let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                          in
                          let _ = L.build_store mallocMatrix lMatrix builder
                          in
                          ()
              ) in
                let lMatrix = (lookup globalSymbolTable localSymbolTable id)
                in
                let loadedLMatrix = L.build_load lMatrix (id ^ "_load") builder
                in
                let zeroIndexPtr = L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t 0|] (id ^ "_zero_index") builder
                in
                let _ = L.build_store (matrixSize) zeroIndexPtr builder
                in
                let _ = List.fold_left (fun acc lvalue -> 
          let ptrIndex = L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t acc|] (id ^ "_ptr_index") builder in
          let _ = L.build_store lvalue ptrIndex builder in
                          (acc + 1)) 1 dimSizeList
                in
              ()

    | A.AMatElementAssign(id, idType, aExprList, aExpr, nDims, _) ->
                let dimIndices = List.map (fun aexpr -> genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aexpr) aExprList 
                in
                let lMatrix = (lookup globalSymbolTable localSymbolTable id)
                in
                let loadedLMatrix = L.build_load lMatrix (id ^ "_load") builder
                in
                let rec findDimSizes dimSizeList = function
                        | 0 -> dimSizeList
                        | i -> findDimSizes (
                            (
                                let dimPtr = (L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t i|] (id ^ "_dim_" ^ string_of_int i) builder)
                                in
                                L.build_load dimPtr (id ^ "_dim_" ^ string_of_int i ^ "value_") builder

                            )
                            ::dimSizeList) (i - 1)

                in let dimSizes = findDimSizes [] nDims
                in
                let (index, _) = List.fold_left2 (fun (result, productDim) size_i index_i ->
                        let productDimMulIndex = (L.build_mul productDim index_i ("tmp_") builder)
                        in let result = (L.build_add result productDimMulIndex ("tmp2_") builder)
                        in
                        let productDim = (L.build_mul productDim size_i ("tmp3_") builder)
                        in
                        (result, productDim) )
               ((L.const_int i32_t (1 + nDims)), (L.const_int i32_t 1))
               (List.rev dimSizes) (List.rev dimIndices)
                in  
                let elementPtr = L.build_in_bounds_gep loadedLMatrix [|index|] (id ^ "_element") builder
                in
                let eValue = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aExpr in
                let _ = L.build_store eValue elementPtr builder in ()

      
    | A.AExprAssign(id, idType, aExpr, _) ->
            (
            match idType with
            | A.Mat(nDims) ->
                let loadedLMatrix = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aExpr 
                in
                let rec findDimSizes dimSizeList = function
                        | 0 -> dimSizeList
                        | i -> findDimSizes (
                            (
                                let dimPtr = (L.build_in_bounds_gep loadedLMatrix [|L.const_int i32_t i|] (id ^ "_dim_" ^ string_of_int i) builder)
                                in
                                L.build_load dimPtr (id ^ "_dim_" ^ string_of_int i ^ "value_") builder

                            )
                            ::dimSizeList) (i - 1)

                in let dimSizes = findDimSizes [] nDims
                in
                      let productSizes  = List.fold_left (fun acc laexpr ->  
                      (L.build_mul acc laexpr "expr_prod_size" builder)) (L.const_int i32_t 1) dimSizes 
                      in
                      let matrixSize = (L.build_add productSizes  (L.const_int i32_t (nDims + 1)) "mat_size" builder)
                in
                
                let _ = (if Hashtbl.mem localSymbolTable id then
                          if Hashtbl.mem memoryMap id then
                              (* free, malloc *)
                              (* FREE: *)
                              let _ = freeMatrixFun  (lookup globalSymbolTable localSymbolTable id) id builder
                              in
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in ()
                          else
                              (* malloc, add to memory map *)
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in
                              (* Add to memory map *)
                              Hashtbl.add memoryMap id (lookup globalSymbolTable localSymbolTable id)

                      else if Hashtbl.mem globalSymbolTable id then
                              (* free, malloc *)
                              (* FREE: *)
                              let _ = freeMatrixFun  (lookup globalSymbolTable localSymbolTable id) id builder
                              in
                              (* MALLOC *)
                              let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                              in
                              let _ = L.build_store mallocMatrix (lookup globalSymbolTable localSymbolTable id) builder
                              in
                              ()
                      else
                          (* alloc ptr, malloc, add to local map, add to memory map *)
                          (* Alloc *)
                          let lMatrix = L.build_alloca mat_t id builder
                          in
                          let _ = Hashtbl.add localSymbolTable id (lMatrix, idType)
                          in 
                          let _ = Hashtbl.add memoryMap id lMatrix
                          in
                          (* MALLOC *)
                          let mallocMatrix = L.build_array_malloc i32_t matrixSize (id ^ "_malloc") builder
                          in
                          let _ = L.build_store mallocMatrix lMatrix builder
                          in
                          ()
              ) in
                let lMatrix = (lookup globalSymbolTable localSymbolTable id)
                in
                let loadedAllocMat = L.build_load lMatrix (id ^ "_load") builder
                in
                let _ = L.build_call copyMat_func [|loadedLMatrix; loadedAllocMat|]
                "copyMat" builder
                in
                ()
            | typ -> if (Hashtbl.mem localSymbolTable id) || (Hashtbl.mem globalSymbolTable id) then let lid = (lookup globalSymbolTable localSymbolTable id) in let _ = (L.build_store (genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aExpr) lid builder) in () 
                   else
                       let local = L.build_alloca (ltypeOfType typ) id builder
                       in
                       ignore (L.build_store (genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aExpr) local builder);
                       Hashtbl.add localSymbolTable id (local, typ)
            )

    in

    let addTerminal builder f =
      (match L.block_terminator (L.insertion_block builder) with
	Some _ -> ()
      | None -> ignore (f builder))
    in

    let rec genCodeForStatements ?contBB:(contBB = None) globalSymbolTable localSymbolTable memoryMap builder = function
        | A.ABlock(astatementList, _) ->  let newGlobalSymbolTable = mergeSymbolTables globalSymbolTable localSymbolTable in
                               let newLocalSymbolTable = Hashtbl.create 100 in
                               let newMemoryMap = Hashtbl.create 100 in
                               List.fold_left (fun builder astatement -> genCodeForStatements ~contBB:contBB newGlobalSymbolTable newLocalSymbolTable newMemoryMap builder astatement) builder astatementList 
        | A.AExpr(aExpr, _) -> let _ = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aExpr in builder

        | A.AVarDecl(aVarDecl, _) -> let _ = genCodeForVarDecl globalSymbolTable localSymbolTable memoryMap builder aVarDecl
                    in builder

        | A.AReturn(retType, aExpr, _) -> ignore (match afunc.A.retType with
                                                  | A.Void -> L.build_ret_void builder
                                                  | _ -> L.build_ret (genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder aExpr)
                                                                        builder); builder
      | A.AIf (predicate, thenStatement, elseStatement, _) ->
                 let boolVal = genCodeForExpression globalSymbolTable localSymbolTable memoryMap builder predicate in
                 let mergeBB = L.append_block context "merge" theFunction in

                 let thenBb = L.append_block context "then" theFunction in
                 addTerminal (genCodeForStatements ~contBB:contBB globalSymbolTable localSymbolTable memoryMap (L.builder_at_end context thenBb) thenStatement)
                   (L.build_br mergeBB);

                 let elseBb = L.append_block context "else" theFunction in
                 addTerminal (genCodeForStatements ~contBB:contBB globalSymbolTable localSymbolTable memoryMap (L.builder_at_end context elseBb) elseStatement)
                   (L.build_br mergeBB);

                 ignore (L.build_cond_br boolVal thenBb elseBb builder);
                 L.builder_at_end context mergeBB

      | A.AWhile (predicate, body, _) ->
                  let predBB = L.append_block context "while" theFunction in
                  ignore (L.build_br predBB builder);

                  let bodyBB = L.append_block context "while_body" theFunction in
                  addTerminal (genCodeForStatements globalSymbolTable localSymbolTable memoryMap ~contBB:(Some predBB) (L.builder_at_end context bodyBB) body)
                    (L.build_br predBB);

                  let predBuilder = L.builder_at_end context predBB in
                  let boolVal = genCodeForExpression globalSymbolTable localSymbolTable memoryMap predBuilder predicate in

                  let mergeBB = L.append_block context "merge" theFunction in
                  ignore (L.build_cond_br boolVal bodyBB mergeBB predBuilder);
                  L.builder_at_end context mergeBB



      | A.AFor(aVarDecl1, aExpr, aVarDecl2, aStatement, _) ->
                (let newGlobalSymbolTable = mergeSymbolTables globalSymbolTable localSymbolTable in
                let newLocalSymbolTable = Hashtbl.create 100 in
                let newMemoryMap = Hashtbl.create 100 in
                let builder = genCodeForStatements ~contBB:contBB newGlobalSymbolTable newLocalSymbolTable newMemoryMap builder (A.AVarDecl(aVarDecl1, A.Void))
                in
                let incrBB = L.append_block context "forincr" theFunction in
                let _ = (genCodeForStatements ~contBB:contBB newGlobalSymbolTable newLocalSymbolTable newMemoryMap (L.builder_at_end context incrBB) (A.AVarDecl(aVarDecl2, A.Void)))
                in 
                let forBB = L.append_block context "for" theFunction in
                let _ = L.build_br forBB builder in
                let _ = L.build_br forBB (L.builder_at_end context incrBB) in 
                let predBuilder = L.builder_at_end context forBB in
                let boolVal = genCodeForExpression newGlobalSymbolTable newLocalSymbolTable newMemoryMap predBuilder aExpr in
                let bodyBB = L.append_block context "forbody" theFunction in
                let _ = addTerminal (genCodeForStatements newGlobalSymbolTable newLocalSymbolTable newMemoryMap ~contBB:(Some incrBB) (L.builder_at_end context bodyBB) aStatement) (L.build_br incrBB)
                in
                let mergeBB = L.append_block context "merge" theFunction in
                ignore (L.build_cond_br boolVal bodyBB mergeBB predBuilder);
                L.builder_at_end context mergeBB)
                

      | A.AExit(_) -> builder
      | A.ABreak(_) -> builder
      | A.AForEachLoop (_, _, _, _, _, _, _) -> builder
      | A.AContinue(_) -> let _ = (match contBB with
                        | Some x -> L.build_br x builder
                        | _ -> printError ("Code should never reach here!")) in builder

        in
        let builder = genCodeForStatements globalSymbolTable localSymbolTable memoryMap builder (A.ABlock (afunc.A.abody, A.Void)) in
                addTerminal builder (match afunc.A.retType with
                                        | A.Void -> L.build_ret_void
                                        | t -> L.build_ret (initOfType t))

  in
  (* Generate code for function definitions *)
  let _ = List.iter generateFunctionBody functions
  in
  
  the_module
