(* MatCV AST *)

(* TODO *)

type op = Add | Sub | Mul | Div | Equal | Neq | Less | Leq | Greater | Geq | And | Or | Mod | Exp

type uop = Neg | Not

type loopType = Row | Ele | Pixel 

type expr =
    Literal of int
  | BoolLit of bool
  | Id of string
  | UnboundedAccessRead of string * expr
  | UnboundedAccessWrite of string * expr * expr
  | MatPlus of string * string
  | MatMinus of string * string
  | MatAccess of string * expr list
  | BinaryOp of expr * op * expr
  | Unop of uop * expr
  | Call of string * expr list
  | Noexpr


type varDecl =   
    | Nodecl
    | Matrix of string * expr list list
    | ExprAssign of string * expr
    | DimAssign of string * expr list
    | MatElementAssign of string * expr list * expr


type statement =
    | Block of statement list
    | Expr of expr
    | VarDecl of varDecl
    | Return of expr 
    | For of varDecl * expr * varDecl * statement
    | While of expr * statement
    | If of expr * statement * statement
    | Exit
    | Break
    | ForEachLoop of string * string * statement * loopType
    | Continue

type functionDefinition = {
    fname: string;
    formals: string list;
    body: statement list
}

type program = statement list * functionDefinition list 

(* Supported Types *)
type builtInType =
    | Void (* For things that don't return anything *)
    | Int
    | Bool
    | Mat of int
    | Annotation of string
    | Func
    (* returnType * formalType list *) 
    | FuncSignature of builtInType * builtInType list 
    | Empty
    | Keyword

(* Annotated Expression*)
type aexpr =
    ALiteral of int * builtInType
  | ABoolLit of bool * builtInType
  | AId of string * builtInType
  | AUnboundedAccessRead of string * builtInType * aexpr * builtInType
  | AUnboundedAccessWrite of string * builtInType * aexpr * aexpr * builtInType
  | AMatPlus of string * builtInType * string * builtInType * builtInType
  | AMatMinus of string * builtInType * string * builtInType * builtInType
  | AMatAccess of string * builtInType * aexpr list * int * builtInType
  | ABinaryOp of aexpr * op * aexpr * builtInType
  | AUnop of uop * aexpr * builtInType
  | ACall of string * builtInType * aexpr list * builtInType
  | ANoexpr of builtInType


(* Annotated variable declarations *)
type avarDecl =   
    | AMatrix of string * builtInType * aexpr list list * int * int * builtInType
    | AExprAssign of string * builtInType * aexpr * builtInType
    | ADimAssign of string * builtInType * aexpr list * int * builtInType
    | AMatElementAssign of string * builtInType * aexpr list * aexpr * int * builtInType
    | ANodecl of builtInType


(* Annotated statements *)
type astatement =
    | ABlock of astatement list * builtInType
    | AExpr of aexpr * builtInType
    | AVarDecl of avarDecl * builtInType
    | AReturn of builtInType * aexpr  * builtInType
    | AFor of avarDecl * aexpr * avarDecl * astatement * builtInType
    | AWhile of aexpr * astatement * builtInType
    | AIf of aexpr * astatement * astatement * builtInType
    | AExit of builtInType
    | ABreak of builtInType
    | AForEachLoop of string * builtInType * string * builtInType * astatement * loopType * builtInType
    | AContinue of builtInType

(* Annotated functions *)

type afunctionDefinition = {
    afname: string * builtInType;
    aformals: (string * builtInType) list;
    abody: astatement list;
    retType: builtInType
} 


(* Pretty Printing function *)

let rec string_of_builtInType = function
    | Void -> "Void"
    | Int -> "Int"
    | Bool -> "Bool"
    | Mat(nDims) -> "Mat(" ^ string_of_int nDims ^ ")"
    | Annotation(m) -> m
    | Func -> "Func"
    | Empty -> "Empty"
    | Keyword -> "Keyword"
    | FuncSignature(returnType, formalTypeList) -> "FuncSignature: ReturnType: " ^ string_of_builtInType returnType ^ ", Formal Types: " ^ String.concat "," (List.map (fun formalType -> string_of_builtInType formalType) formalTypeList)




let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mul -> "*"
  | Div -> "/"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"
  | Mod -> "%"
  | Exp -> "^"

let
 string_of_uop = function
  | Neg -> "-"
  | Not -> "!"

let string_of_boolLit = function
    true -> "true"
  | false -> "false"


let rec string_of_aexpr = function
  | ALiteral(l, t) ->  string_of_int l
  | ABoolLit(b, t) ->  string_of_boolLit b
  | AId(s, t) -> s
  | AUnboundedAccessRead(id, _, expr, _) -> "<" ^id ^ "," ^ string_of_aexpr expr ^ ">"
  | AUnboundedAccessWrite(id, _, expr1, expr2, _) -> "[[" ^ id ^ "," ^ string_of_aexpr expr1 ^ "," ^ string_of_aexpr expr2 ^ "]]"
  | AMatPlus(id1,_,id2,_,_) -> id1 ^ " +. " ^ id2
  | AMatMinus(id1,_,id2,_,_) -> id1 ^ " -. " ^ id2
  | AMatAccess(id, t1, exprLst,i, t2) -> id ^  string_of_dim exprLst 
  | ABinaryOp(e1, o, e2, t) ->
      string_of_aexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_aexpr e2
  | AUnop(o, e, t) ->  string_of_uop o ^ string_of_aexpr e
  | ACall(f, t1, el, t2) -> 
      f ^ "(" ^ String.concat ", " (List.map string_of_aexpr el) ^ ")"
  | ANoexpr(t) -> ""

  and string_of_dim = function
| [] -> ""
| a::lst -> "[" ^ string_of_aexpr a ^ "]" ^ string_of_dim lst

let rec string_of_row = function
  | [] -> ""
  | [a] -> string_of_aexpr a
  | a::lst-> string_of_aexpr a ^ "," ^ string_of_row lst 
  
  
let rec string_of_mat = function
  | [] -> ""
  | [a] ->  string_of_row  a
  | a::lst-> string_of_row a  ^ ";" ^ string_of_mat lst 


let rec string_of_avarDecl = function
    | AMatrix(matrixName, t1, mat, i1, i2, t2) -> "\n" ^ string_of_builtInType t1 ^ " " ^  matrixName ^ " = {" ^ string_of_mat mat ^ "}; \n" 
    | AExprAssign(id, t1,  expr, t2) ->  string_of_builtInType t1 ^ " " ^  id ^ " = "  ^ string_of_aexpr expr ^  ";"  
    | ADimAssign(id, t1, expr, i, t2) ->    string_of_builtInType t1   ^ " " ^ id ^ " = "   ^ string_of_dim expr ^ ";\n"  
    | AMatElementAssign(s, t1 , aexpr1, aexpr2, i, t2) -> string_of_builtInType t1 ^ " " ^ s ^ string_of_dim aexpr1 ^ "=" ^ string_of_aexpr aexpr2 ^ "; \n"
    | ANodecl(t) -> ""

let string_of_loopType = function
| Row -> "row" 
| Ele -> "ele"
| Pixel -> "pixel"


let rec string_of_aforDecl = function
| AExprAssign(id,t1, expr, t2) -> string_of_builtInType t1 ^ " " ^ id ^ " = " ^ string_of_aexpr expr
| AMatrix(matrixName, t1, mat, i1, i2, t2) ->  matrixName ^ " = {" ^ string_of_mat mat ^ "}"
| ADimAssign(id, t1, expr,i , t2) -> string_of_builtInType t1   ^ " " ^  id ^ " = "   ^ string_of_dim expr
| AMatElementAssign(s, t1 , aexpr1, aexpr2, i, t2) -> string_of_builtInType t1 ^ " " ^ s ^ string_of_dim aexpr1 ^ "=" ^ string_of_aexpr aexpr2
| ANodecl(t) -> ""




let rec string_of_astmt = function
    ABlock(stmts, t) ->
      "{\n" ^ String.concat "" (List.map string_of_astmt stmts)  ^ "}\n"
  | AExpr(expr, t) -> string_of_aexpr expr ^ ";\n";
  | AVarDecl(varDecl, t) ->   string_of_avarDecl varDecl ^ "\n";
  | AReturn(_, expr, t) -> "return " ^ string_of_aexpr expr ^ ";\n";
  | AIf(e, s1, ABlock([], Void), t) -> "if (" ^ string_of_aexpr e ^ ")\n" ^ string_of_astmt s1
  | AIf(e, s1, s2, t) ->  "if (" ^ string_of_aexpr e ^ ")\n" ^
      string_of_astmt s1 ^ "else\n" ^ string_of_astmt s2
  | AFor(v1, e2, v2, s, t) ->
     "for (" ^ string_of_aforDecl v1  ^ ";" ^ string_of_aexpr e2 ^ ";" ^
      string_of_aforDecl v2  ^ ") " ^ string_of_astmt s
  | AWhile(e, s, t) ->  "while (" ^ string_of_aexpr e ^ ") " ^ string_of_astmt s
  | AForEachLoop(str1,t1,str2,t2,s1,loopT,t3) ->  "for " ^ string_of_loopType loopT ^ " " ^ str1 ^ ":" ^ str2 ^ string_of_astmt s1    
  | AExit(t) -> "exit; \n"
  | ABreak(t) -> "break; \n"
  | AContinue(t) -> "continue; \n" 




let rec  string_of_aformals = function 
  | [] -> ""
  | [(s,t)] ->  string_of_builtInType t ^ " " ^ s
  | (s,t)::lst-> string_of_builtInType t ^ " " ^ s ^ "," ^ string_of_aformals lst



let string_of_afname = function
  
  | (s,t) -> string_of_builtInType t^ "_" ^ s


let string_of_afdecl fdecl =
  string_of_builtInType fdecl.retType ^ " " ^ string_of_afname fdecl.afname ^ "("  ^ (string_of_aformals fdecl.aformals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_astmt fdecl.abody) ^
  "}\n"


let string_of_program (stmnt, funcs) =
  String.concat "" (List.map string_of_astmt stmnt) ^ "\n" ^
  String.concat "\n" (List.map string_of_afdecl funcs)

