/* Ocamlyacc parser for Crayon */

%{ open Ast %}

%token SEMI COMMA LPAREN RPAREN RBRACK LBRACK LBRACE RBRACE
%token PLUS MINUS TIMES DIVIDE ASSIGN 
%token EQ NEQ LT LEQ GT GEQ AND OR NOT
%token RETURN IF ELSE FOR WHILE FUNC NEG
%token INT BOOL VOID STRING ARRAY CANVAS TRUE FALSE
%token TILDE AMPER DOLLAR AT

%token <int> NUM_LIT
%token <string> STRING_LIT
%token <string> ID
%token EOF
%token NULL

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left OR
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ 
%left PLUS MINUS
%left TIMES DIVIDE
%right NOT NEG

%token SIZE
%token ROWS
%token COLUMNS

%start program
%type <Ast.program> program

%%

program:
  decls EOF { $1 }

decls:
   /* nothing */ { [], [] }
 | decls vdecl { ($2 :: fst $1), snd $1 }
 | decls fdecl { fst $1, ($2 :: snd $1) }

fdecl:
   typ ID LPAREN formals_opt RPAREN LBRACE vdecl_list stmt_list RBRACE
     { { typ = $1;
	 fname = $2;
	 formals = $4;
	 locals = List.rev $7;
	 body = List.rev $8 } }

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    typ ID                   { [($1,$2)] }
  | formal_list COMMA typ ID { ($3,$4) :: $1 }

typ:
    INT { Int }
  | BOOL { Bool }
  | VOID { Void }
  | STRING { String }
  | ARRAY typ LBRACK NUM_LIT RBRACK { Array($2, $4) }
  | CANVAS DOLLAR { Pointer }
  | ARRAY DOLLAR { Pointer }
  | CANVAS LBRACK NUM_LIT COMMA NUM_LIT RBRACK { Canvas($3, $5) }

vdecl_list:
    /* nothing */    { [] }
  | vdecl_list vdecl { $2 :: $1 }

vdecl:
   typ ID SEMI { ($1, $2) }

stmt_list:
    /* nothing */  { [] }
  | stmt_list stmt { $2 :: $1 }

stmt:
    expr SEMI { Expr $1 }
  | RETURN SEMI { Return Noexpr }
  | RETURN expr SEMI { Return $2 }
  | LBRACE stmt_list RBRACE { Block(List.rev $2) }
  | IF LPAREN expr RPAREN stmt %prec NOELSE { If($3, $5, Block([])) }
  | IF LPAREN expr RPAREN stmt ELSE stmt    { If($3, $5, $7) }
  | FOR LPAREN expr_opt SEMI expr SEMI expr_opt RPAREN stmt
     { For($3, $5, $7, $9) }
  | WHILE LPAREN expr RPAREN stmt { While($3, $5) }

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

expr:
    NUM_LIT          { NumLit($1) }
  | STRING_LIT       { StringLit($1) }
  | TRUE             { BoolLit(true) }
  | FALSE            { BoolLit(false) }
  | ID               { Id($1) }
  | NULL             { Null }
  | expr PLUS   expr { Binop($1, Add,   $3) }
  | expr MINUS  expr { Binop($1, Sub,   $3) }
  | expr TIMES  expr { Binop($1, Mult,  $3) }
  | expr DIVIDE expr { Binop($1, Div,   $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq,   $3) }
  | expr LT     expr { Binop($1, Less,  $3) }
  | expr LEQ    expr { Binop($1, Leq,   $3) }
  | expr GT     expr { Binop($1, Greater, $3) }
  | expr GEQ    expr { Binop($1, Geq,   $3) }
  | expr AND    expr { Binop($1, And,   $3) }
  | expr OR     expr { Binop($1, Or,    $3) }
  | MINUS expr %prec NEG { Unop(Neg, $2) }
  | NOT expr         { Unop(Not, $2) }
  | expr ASSIGN expr   { Assign($1, $3) }
  | ID LPAREN actuals_opt RPAREN { Call($1, $3) }
  | LPAREN expr RPAREN { $2 }
  | ID LBRACK expr RBRACK { ArrayAccess($1, $3) }
  | LBRACK arr_lit RBRACK { ArrayLit(List.rev $2) }
  | ID LBRACK expr COMMA expr RBRACK { CanvasAccess($1, $3, $5) }
  | ID LBRACK expr COMMA expr COMMA expr RBRACK { ColorAccess($1, $3, $5, $7) }
  | SIZE LPAREN ID RPAREN { Size($3) }
  | ROWS LPAREN ID RPAREN { Rows($3) }
  | COLUMNS LPAREN ID RPAREN { Columns($3) }
  | AMPER ID { Reference($2) }
  | TILDE ID { Dereference($2) }
  | AT expr AT ID { MovePointer($2, $4) }

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
    expr  { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }
    
arr_lit:
    expr { [$1] }
  | arr_lit COMMA expr { $3 :: $1 }