/* Ocamlyacc parser for CPlus */

%{
open Ast
%}

%token SEMI LPAREN RPAREN LBRACE RBRACE LSQUARE RSQUARE COMMA
%token PLUS MINUS TIMES DIVIDE MOD ASSIGN MOD_ASSIGN NOT
%token DOT ARROW
%token AMP INC DEC
%token EQ NEQ LT LEQ GT GEQ TRUE FALSE AND OR
%token RETURN IF ELSE FOR WHILE SIZEOF
%token INT SIZE_T STRING BOOL CHAR VOID STRUCT NULL
%token <string> PRINTF PRINT PRINTB PRINTBIG MALLOC FREE ATOI STRDUP
%token <int> LITERAL
%token <string> STRINGLIT
%token <char> CHARLIT
%token <string> ID
%token <string> STRUCT_ID
%token EOF

/* Opeartor Precedence closesly modeled on C */
%nonassoc NOELSE
%nonassoc ELSE
/*%right ASSIGN
%left OR
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE
%right AMP
%right CAST
%right NOT NEG*/
/* %left DOT */


%start program
%type <Ast.program> program

%%

program:
  decls EOF { $1 }

decls:
   /* nothing */ { {globals=[]; functions=[]; structs=[]} }
 | decls declaration { {globals = ($2 @ $1.globals); functions = $1.functions; structs = $1.structs} }
 | decls func_decl { {globals = $1.globals; functions = ($2 :: $1.functions); structs = $1.structs} }
 | decls struct_decl { {globals = $1.globals; functions = $1.functions; structs = ($2 :: $1.structs)} }

func_decl:
   typ ID LPAREN formals_opt RPAREN LBRACE declaration_list stmt_list RBRACE
     { { typ = $1;
	       fname = $2;
	       formals = $4;
	       locals = List.rev $7;
	       body = List.rev $8 } }

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    typ ID                  { [($1,$2,Noexpr,false)] }
  | formal_list COMMA typ ID { ($3,$4,Noexpr,false) :: $1 }

struct_decl:
    STRUCT STRUCT_ID LBRACE declaration_list RBRACE SEMI
    { { members = $4;
        struct_name = $2; } }

typ:
    INT { Int }
  | SIZE_T { Size_t }
  | STRING { String }
  | BOOL { Bool }
  | CHAR { Char }
  | VOID { Void }
  | STRUCT_ID { Struct($1) }
  | typ TIMES { Pointer($1) }

declaration_list:
    /* nothing */    { [] }
  | declaration_list declaration { $2 @ $1 }

/* should be able to declare multiple vars of one type on the same line */
declaration: /* is a list */
   typ init_declarator_list SEMI {
     let append_typ t (a,b,c) = (match b with
          BuildArray(_,_) -> (Pointer t, a, b, c)
       |  _ -> (t, a, b, c)
       ) in
     List.map (append_typ $1) $2 }

/* id to value binding for a declarator */
init_declarator:
    ID { ($1, Noexpr, false) }
  | ID LSQUARE add_expr RSQUARE { ($1, BuildArray($1, $3), true) }
  | ID ASSIGN expr { ($1, $3, true) }

/* list of name-value tuples for declarators */
init_declarator_list:
    init_declarator { [$1] }
  | init_declarator_list COMMA init_declarator { $3 :: $1 }

stmt_list:
    /* nothing */  { [] }
  | stmt_list stmt { $2 :: $1 }

stmt:
    expr SEMI { Expr $1 }
  | selection_stmt { $1 }
  | iteration_stmt { $1 }
  | RETURN SEMI { Return Noexpr }
  | RETURN expr SEMI { Return $2 }
  | LBRACE stmt_list RBRACE { Block(List.rev $2) }

selection_stmt:
    IF LPAREN expr RPAREN stmt %prec NOELSE { If($3, $5, Block([])) }
    /*IF LPAREN expr RPAREN stmt              { If($3, $5, Block([])) }*/
  | IF LPAREN expr RPAREN stmt ELSE stmt    { If($3, $5, $7) }

iteration_stmt:
    FOR LPAREN expr_opt SEMI expr SEMI expr_opt RPAREN stmt
     { For($3, $5, $7, $9) }
  | WHILE LPAREN expr RPAREN stmt { While($3, $5) }

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

expr:
    assignment_expr  { $1 }

assignment_operator:
    ASSIGN                                { Asn }
  | MOD_ASSIGN                            { ModAsn }

assignment_expr:
    logical_or_expr                       { $1 }
  | postfix_expr assignment_operator expr { Assign($1, $2, $3) }

logical_or_expr:
    logical_and_expr                      { $1 }
  | logical_or_expr OR logical_and_expr   { Binop($1, Or, $3) }

logical_and_expr:
    equality_expr                         { $1 }
  | logical_and_expr AND equality_expr    { Binop($1, And, $3) }

equality_expr:
    relational_expr                       { $1 }
  | equality_expr EQ relational_expr      { Binop($1, Equal, $3) }
  | equality_expr NEQ relational_expr     { Binop($1, Neq, $3) }

relational_expr:
    add_expr                              { $1 }
  | relational_expr LT add_expr           { Binop($1, Less, $3) }
  | relational_expr GT add_expr           { Binop($1, Greater, $3) }
  | relational_expr LEQ add_expr          { Binop($1, Leq, $3) }
  | relational_expr GEQ add_expr          { Binop($1, Geq, $3) }

cast_expr:
    unary_expr                            { $1 }
  | LPAREN typ RPAREN cast_expr           { Cast($2, $4) }

unary_operator:
    NOT                                    { Not }
  | MINUS                                  { Neg }

unary_expr:
    postfix_expr                          { $1 }
  | unary_operator postfix_expr           { Unop($1, $2)  }
  | INC postfix_expr                      { Assign($2,Asn,Binop($2, Add, Literal(1))) }
  | DEC postfix_expr                      { Assign($2,Asn,Binop($2, Sub, Literal(1))) }

postfix_expr:
    built_in_expr                                { $1 }
  | postfix_expr INC                             { Assign($1,Asn,Binop($1, Add, Literal(1))) }
  | postfix_expr DEC                             { Assign($1,Asn,Binop($1, Sub, Literal(1))) }
  | postfix_expr LPAREN actuals_list_opt RPAREN  { Call($1, $3) }
  | postfix_expr LSQUARE postfix_expr RSQUARE    { ArrayAccess($1, $3) }
  | postfix_expr DOT ID                          { StructAccess($1, $3) }
  | postfix_expr ARROW ID                        { StructPointerAccess($1, $3) }

built_in_expr:
    primary_expr                                 { $1 }
  | PRINTF LPAREN actuals_list_opt RPAREN        { BuiltInCall($1, $3) }
  | PRINT LPAREN actuals_list_opt RPAREN         { BuiltInCall($1, $3) }
  | PRINTB  LPAREN actuals_list_opt RPAREN       { BuiltInCall($1, $3) }
  | PRINTBIG LPAREN actuals_list_opt RPAREN      { BuiltInCall($1, $3) }
  | MALLOC LPAREN actuals_list_opt RPAREN        { BuiltInCall($1, $3) }
  | FREE LPAREN actuals_list_opt RPAREN          { BuiltInCall($1, $3) }
  | ATOI LPAREN actuals_list_opt RPAREN          { BuiltInCall($1, $3) }
  | STRDUP LPAREN actuals_list_opt RPAREN        { BuiltInCall($1, $3) }

actuals_list_opt:
  /* nothing */                              { [] }
  | actuals_list                             { List.rev $1 }

actuals_list:
    expr                                     { [$1] }
  | actuals_list COMMA expr                  { $3 :: $1 }

add_expr:
    mult_expr                             { $1 }
  | add_expr PLUS mult_expr               { Binop($1, Add, $3) }
  | add_expr MINUS mult_expr              { Binop($1, Sub, $3) }

mult_expr:
    cast_expr                             { $1 }
  | mult_expr TIMES cast_expr             { Binop($1, Mult, $3) }
  | mult_expr DIVIDE cast_expr            { Binop($1, Div, $3) }
  | mult_expr MOD cast_expr               { Binop($1, Mod, $3) }

primary_expr:
    LPAREN expr RPAREN                    { $2 }
  | LITERAL                               { Literal($1) }
  | STRINGLIT                             { StringLit($1) }
  | CHARLIT                               { CharLit($1) }
  | TRUE                                  { BoolLit(true) }
  | FALSE                                 { BoolLit(false) }
  | ID                                    { Id($1) } /* Identifier */
  | AMP primary_expr                      { Address($2) }
  | TIMES primary_expr                    { Dereference($2) }
  | SIZEOF LPAREN typ RPAREN              { Sizeof($3) }
  | NULL                                  { Nullexpr }
