(* Code generation: translate takes a semantically checked AST and
produces LLVM IR

LLVM tutorial: Make sure to read the OCaml version of the tutorial

http://llvm.org/docs/tutorial/index.html

Detailed documentation on the OCaml LLVM library:

http://llvm.moe/
http://llvm.moe/ocaml/

*)

open Ast

module L = Llvm
module A = Ast

module StringMap = Map.Make(String)

let translate program =
  let context = L.global_context () in
  let the_module = L.create_module context "CPlus"
  and i64_t  = L.i64_type  context
  and i32_t  = L.i32_type  context
  and i8_t   = L.i8_type   context
  and i1_t   = L.i1_type   context
  and str_t  = L.pointer_type (L.i8_type context)
  and void_t = L.void_type context
  and ptr_t t = L.pointer_type t
  and struct_t n = L.named_struct_type context n in

  let rec ltype_of_primitive = function
      A.Int -> i32_t
    | A.Size_t -> i64_t
    | A.String -> str_t
    | A.Bool -> i1_t
    | A.Char -> i8_t
    | A.Void -> void_t
    | A.Pointer t ->
      if (t == A.Void) then ptr_t i8_t else ptr_t (ltype_of_primitive t)
    | A.Struct n -> struct_t n (* unitialized struct type *)
  in (* END FUNCTION ltype_of_primitive *)

  let structs =
    let struct_decl m sdecl =
      let name = sdecl.A.struct_name
      and member_types = Array.of_list (List.map (fun (t,_,_,_) -> ltype_of_primitive t) sdecl.A.members)
      in let stype = L.struct_type context member_types in
      StringMap.add name (stype, sdecl.A.members) m in
    List.fold_left struct_decl StringMap.empty program.structs
  in

  let struct_lookup n = try StringMap.find n structs
    with Not_found -> raise (Failure ("CODEGEN: struct " ^ n ^ " not found")) in
  (* END structs *)

  let rec ltype_of_typ = function
      A.Struct n -> fst (struct_lookup n) (* reveals "stype" *)
    | A.Pointer t ->
        if (t == A.Void) then ptr_t i8_t else ptr_t (ltype_of_typ t)
    | t -> ltype_of_primitive t
  in

  (* sometimes we need to get at the subtype *)
  let strip = function
      A.Pointer t ->  t
    | t -> t
  in

  let initialize_value typ v = match (typ,v) with
      (A.Int, A.Literal i) -> L.const_int i32_t i
    | (A.Size_t, A.Literal i) -> L.const_int i64_t i
    | (A.Size_t, A.SizeLit s) -> L.const_of_int64 i64_t s true
    | (A.String, A.StringLit s) -> L.const_string context (s^"\x00")
    | (A.Bool, A.BoolLit b) -> L.const_int i1_t (if b then 1 else 0)
    | (A.Char, A.CharLit c) -> L.const_int i8_t (Char.code c)
    | (_,A.Noexpr) -> L.const_int i32_t 0
    | (A.Pointer(t), _) -> L.const_pointer_null (ltype_of_typ t)
    | _ -> raise (Failure("CODEGEN: attempting to initialize global to non-primitive type"))
    (* | _ -> L.const_int (ltype_of_typ typ) 0 *)
  in

  (* Declare each global variable; remember its value in a map *)
  (* FIXME: currently cannot enable the initialization of global pointers *)
  let global_vars =
    let global_var map (typ, name, value, _) =
      (* let init = L.const_int (ltype_of_typ typ) 0 *)
      let init = initialize_value typ value
      in StringMap.add name (L.define_global name init the_module) map in
    List.fold_left global_var StringMap.empty program.globals
  in

  (* as with global_vars, stores their types in a separate map *)
  let global_var_types =
    let global_var map (typ, name, _, _) =
      StringMap.add name typ map in
    List.fold_left global_var StringMap.empty program.globals
  in

  (* Make also a map of names to whether we need to lazily eval their intial values *)
  (* let global_lazy_vars =
    let global_var m (_, n, v, f) =
      (* the eval field is set false once evaluated *)
      StringMap.add n { name = n; flag = f; eval = true; expr = v } m in
    List.fold_left global_var StringMap.empty program.globals
  in *)

  (* Declare printf(), which the print built-in function will call (only strings for now) *)
  let printf_t = L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
  let printf_func = L.declare_function "printf" printf_t the_module in

  (* Declare atoi(str), which returns the int of char* string *)
  let atoi_t = L.function_type (ptr_t i8_t) [| (ptr_t i8_t) |] in
  let atoi_func = L.declare_function "atoi" atoi_t the_module in

  (* Declare strdup(str), which returns a char* with space for a copy of str *)
  let strdup_t = L.function_type (ptr_t i8_t) [| (ptr_t i8_t) |] in
  let strdup_func = L.declare_function "strdup" strdup_t the_module in

  (* Declare the built-in printbig() function *)
  let printbig_t = L.function_type i32_t [| i32_t |] in
  let printbig_func = L.declare_function "printbig" printbig_t the_module in

  (* Define each function (arguments and return type) so we can call it *)
  let function_decls =
    let function_decl m fdecl =
      let name = fdecl.A.fname
      and formal_types = Array.of_list (List.map (fun (t,_,_,_) -> ltype_of_typ t) fdecl.A.formals)
      in let ftype = L.function_type (ltype_of_typ fdecl.A.typ) formal_types in
      StringMap.add name (L.define_function name ftype the_module, fdecl) m in
    List.fold_left function_decl StringMap.empty program.functions in

  (* Fill in the body of the given function *)
  let build_function_body fdecl =
    let (the_function, _) = StringMap.find fdecl.A.fname function_decls in
    let builder = L.builder_at_end context (L.entry_block the_function) in

    let int_format_str = L.build_global_stringptr "%d\n" "fmt" builder in
    let str_format_str = L.build_global_stringptr "%s\n" "fmt" builder in

    (* Construct the function's "locals": formal arguments and locally
       declared variables.  Allocate each on the stack, initialize their
       value, if appropriate, and remember their values in the "locals" map *)
    let local_vars =

      (* m = map, t = type, n = name, v = value, p = parameters *)
      let add_formal m (t, n, v, _) p =
         L.set_value_name n p;
    	   let formal = L.build_alloca (ltype_of_typ t) n builder in
         (* ignore (L.build_store p formal builder); *)
         ignore(
           (* if v = A.Noexpr then (L.build_store p formal builder) else *)
           match (t,v) with
                  ((_,A.Noexpr) | (A.Pointer(Char),_)) ->
                    ignore(L.build_store p formal builder)
                | ((_,A.Literal _) | (_,A.SizeLit _ )| (_,A.StringLit _) | (_,A.BoolLit _)) ->
                    ignore(L.build_store (initialize_value t v) formal builder)
                | _ -> ignore(None) (* IDEA: can keep these things lazy, but eval them earlier than when the ID is called!!! *)
           (* (L.build_store (initialize_value t v) formal builder) *)
         );

         StringMap.add n formal m (* meat of "add_formal" *)
      in (* end function def of "add_formal" *)

      let add_local m (t, n, v, _) =
        let local_var = L.build_alloca (ltype_of_typ t) n builder in
        (* ignore(if v != A.Noexpr then ignore(L.build_store (initialize_value t v) local_var builder)); *)
        ignore(
          match (t,v) with
            (A.Pointer(Char),_) -> ignore(None)
          | (_,A.Literal _) | (_,A.SizeLit _) | (_,A.StringLit _)| (_,A.BoolLit _) ->
              ignore(L.build_store (initialize_value t v) local_var builder)
          | _ -> ignore(None) (* Do Nothing *)
        );
        StringMap.add n local_var m (* meat of function def of "add_local" *)
      in (* end function def of "add_local" *)

      (* fold in the list of formals using "add_formal" *)
      let formals = List.fold_left2 add_formal StringMap.empty fdecl.A.formals
          (Array.to_list (L.params the_function)) in

      (* fold in the list of locals using "add_local" *)
      List.fold_left add_local formals fdecl.A.locals
    in
    (* END DEFINITION of LOCAL_VARS *)

    (* As with local_vars, make a map of types for each name *)
    let local_var_types =

      (* m = map, t = type, n = name, v = value *)
      let add_item m (t, n, _, _) =
        StringMap.add n t m
      in (* end function def of "add_item" *)

      (* fold in the list of formals using "add_formal" *)
      let formals = List.fold_left add_item StringMap.empty fdecl.A.formals in

      (* fold in the list of locals using "add_local" *)
      List.fold_left add_item formals fdecl.A.locals
    in
    (* END DEFINITION of LOCAL_VARS *)

    (* Make also a map of names to whether we need to lazily eval their intial values *)
    let local_lazy_vars =
      let add_item m (_, n, v, f) =
        (* the eval field is set false once evaluated *)
        StringMap.add n { name = n; flag = f; eval = true; expr = v } m
      in
      let formals = List.fold_left add_item StringMap.empty fdecl.A.formals in
      List.fold_left add_item formals fdecl.A.locals
    in

    (* Return the value for a variable or formal argument *)
    let lookup n = try StringMap.find n local_vars
      with Not_found -> try StringMap.find n global_vars
        with Not_found -> raise (Failure("CODEGEN: can't find var " ^ n))
    in

    (* Return the type for a variable or formal argument *)
    let lookup_type n =
      (* let is_int s =
        try ignore (int_of_string s); true
          with _ -> false
      in
      if is_int n then A.Int else *)
        try StringMap.find n local_var_types
        with Not_found ->
          (* SUPER HACK: must not fail! So default to void *)
          try StringMap.find n global_var_types
              (* with Not_found -> raise (Failure("CODEGEN: can't find type of " ^ n)) *)
            with Not_found -> A.Void
    in

    (* Did we decide to evaluate this lazily? *)
    (* let lookup_lazy n = try StringMap.find n local_lazy_vars
      with Not_found -> try StringMap.find n global_lazy_vars
        with Not_found -> raise (Failure ("CODEGEN: can't find lazy var"))
    in *)

    let failsafe_flag = 0 in

    (* Construct code for a primary expression; return its value *)
    (* let rec primary_expr builder = function *)
    let rec primary_expr builder = function
        A.Literal i -> L.const_int i32_t i
      | A.SizeLit s -> L.const_of_int64 i64_t s true
      | A.StringLit s -> L.build_global_stringptr (s^"\x00") "strptr" builder
      | A.BoolLit b -> L.const_int i1_t (if b then 1 else 0)
      | A.CharLit c -> L.const_int i8_t (Char.code c)
      | A.Noexpr -> L.const_int i32_t 0
      | A.Nullexpr -> L.const_null (void_t)
      | A.Id s -> L.build_load (lookup s) s builder
      (* Address - just use the lookup function, which in fact will give us
         the 'stack-pointer' returned by alloca when this var was created *)
      | A.Address e -> lookup (A.string_of_expr e)
      | A.Dereference e -> L.build_load (primary_expr builder e) "deref" builder
      | A.Sizeof t ->
          L.size_of (ltype_of_typ t) (* returns a 64-bit (i64_t) integer *)
      | e -> (* for the case with the parentheses *)
        if failsafe_flag = 1 then
          raise (Failure("CODEGEN: expression reduction failed!!!"))
        else expr builder e

    (* Construct code for mult_expr; return its value *)
    and mult_expr builder = function
        A.Binop (e1, ((A.Mult | A.Div | A.Mod) as op), e2) ->
              let e1' = mult_expr builder e1
              and e2' = cast_expr builder e2 in
              (match op with
                       A.Mult -> L.build_mul
                     | A.Div -> L.build_sdiv
                     | A.Mod -> L.build_srem
                     | _ -> raise (Failure("CODEGEN: improper mult expr!!!"))
              ) e1' e2' "mult_expr" builder
      | e -> cast_expr builder e

    (* Construct code for add_expr; return its value *)
    and add_expr builder = function
        A.Binop (e1, ((A.Add | A.Sub) as op), e2) ->
            (* HACK: string_of_expr will not work for much!! *)
            let t1 = lookup_type (string_of_expr e1)
            and t2 = lookup_type (string_of_expr e2) in
            (* FIXME: only pointer addition works currently, not subtraction *)
              (match (op, t1, t2) with
                  (A.Add, A.Pointer(_), _) ->
                      L.build_gep (add_expr builder e1) [| (mult_expr builder e2) |] "add.ptr" builder
                | (A.Add, _, A.Pointer(_)) ->
                      L.build_gep (mult_expr builder e2) [| (add_expr builder e1) |] "add.ptr" builder
                | (A.Add, _, _) -> L.build_add (add_expr builder e1) (mult_expr builder e2) "add_expr" builder
                | (A.Sub, _, _) -> L.build_sub (add_expr builder e1) (mult_expr builder e2) "sub_expr" builder
                | _ -> raise (Failure("CODEGEN: improper add/sub expr!!!"))
              )
      | e -> mult_expr builder e

    (* Construct code for a built_in expression; return its value  *)
    and built_in_expr builder = function
        A.BuiltInCall (n, [e]) -> (match n with
            "printf" -> L.build_call printf_func [| str_format_str ; (expr builder e) |] "printf" builder
          | "print" | "printb" -> L.build_call printf_func [| int_format_str ; (expr builder e) |] "printf" builder
          | "printbig" -> L.build_call printbig_func [| (expr builder e) |] "printbig" builder
          (* | "malloc" -> let e' = primary_expr builder e in
            L.build_array_malloc (L.type_of e') "malloc" builder *)
          | "malloc" -> (match e with
                A.Binop(e1, A.Mult, e2) ->
                  let e1' = add_expr builder e1
                  and e2' = add_expr builder e2 in
                    if L.type_of e1' = i32_t then L.build_array_malloc (L.type_of e2') e1' "arr_malloc" builder
                    else L.build_array_malloc (L.type_of e1') e2' "malloc" builder
              | _ -> let e' = primary_expr builder e in L.build_malloc (L.type_of e') "malloc" builder)
          | "free" -> L.build_free (expr builder e) builder (* this had better be a void* pointer! *)
          | "atoi" -> L.build_call atoi_func [| (expr builder e) |] "atoi" builder
          | "strdup" -> L.build_call strdup_func [| (expr builder e) |] "strdup" builder
          | _ -> raise (Failure("CODEGEN: improper built-in function call!!!"))
        ) (* end BuiltInCall match statement *)
      | A.BuildArray (a, n) ->
            let ptr = lookup_type a in
            let t = strip ptr
            and n' = primary_expr builder n in
            (* malloc returns in i8_t ptr *)
            let tmp = L.build_array_malloc (ltype_of_typ t) n' "ary.malloc" builder in
            (* need to cast it to a ptr_t *)
            L.build_bitcast tmp (ltype_of_typ ptr) "ary.cast" builder
      | e -> primary_expr builder e

    (* Construct code for a postfix expression; return its value *)
    and postfix_expr builder = function
        A.Call (f, act) ->
            (* FIXME: using string_of_expr for now, but need a comprehensive solution to compound LHS *)
            let f' = A.string_of_expr f in
            let (fdef, fdecl) = StringMap.find f' function_decls in
            let actuals = List.rev (List.map (expr builder) (List.rev act)) in
            let result = (match fdecl.A.typ with A.Void -> "" | _ -> f' ^ "_result") in
            L.build_call fdef (Array.of_list actuals) result builder
      | A.ArrayAccess (a, i) ->
          let n = L.build_zext_or_bitcast (postfix_expr builder i) i64_t "ary.cast" builder in
          (* SOLUTION: need to use i64_t array constants, still figuring out how to do this *)
          let idx = L.build_gep (expr builder a) [| n |] "ary.ptr" builder in
          L.build_load idx "ary.val" builder
      | A.StructAccess (s, m) ->
          let (stype, location) = (match s with
            | A.Id id -> (lookup_type id, lookup id)
            | A.Dereference sp ->
                (strip(lookup_type (A.string_of_expr sp)), expr builder s)
            | A.ArrayAccess (a, _) ->
              (strip(strip(lookup_type (A.string_of_expr a))) , expr builder s)
            | _ -> raise (Failure("CODEGEN: illegal struct-type " ^ A.string_of_expr s))
          ) in
            let members = snd (struct_lookup (A.string_of_typ stype)) in
              let rec get_idx n lst i = match lst with
                | [] -> raise (Failure("CODEGEN: id " ^ m ^ " is not a member of struct " ^ A.string_of_expr s))
                | hd::tl -> if (hd=n) then i else get_idx n tl (i+1)
              in let idx = (get_idx m (List.map (fun (_,nm,_,_) -> nm) members) 0) in
              let ptr = L.build_struct_gep location idx ("struct.ptr") builder in
          L.build_load ptr ("struct.val."^m) builder
      | A.StructPointerAccess (s, m) ->
        let (stype, location) = (match s with
            | ArrayAccess(a,_) ->
              (match lookup_type (A.string_of_expr a) with
               | Pointer(Pointer(t)) -> (t, expr builder s)
               | Pointer(t) -> (t, expr builder s)
               | _ -> raise (Failure("CODGEN: something wrong with arrays D:"))
              )
            | _ -> ((strip(lookup_type (A.string_of_expr s))), expr builder s)
          ) in
          (* let stype = (strip(lookup_type (A.string_of_expr s)))
          and location = expr builder s in *)
            let members = snd (struct_lookup (A.string_of_typ stype)) in
              let rec get_idx n lst i = match lst with
                | [] -> raise (Failure("CODEGEN: id " ^ m ^ " is not a member of struct " ^ A.string_of_expr s))
                | hd::tl -> if (hd=n) then i else get_idx n tl (i+1)
        in let idx = (get_idx m (List.map (fun (_,nm,_,_) -> nm) members) 0) in
        let ptr = L.build_struct_gep location idx ("struct.ptr") builder in
        L.build_load ptr ("struct.val."^m) builder  
      | e -> built_in_expr builder e

    (* Construct code for a unary expression; return its value *)
    and unary_expr builder = function
        A.Unop(((A.Neg | A.Not) as op), e) ->
          let e' = postfix_expr builder e in
          (match op with
             A.Neg     -> L.build_neg e' "tmp" builder
           | A.Not     -> L.build_not e' "tmp" builder)
      | e -> postfix_expr builder e

    (* Construct code for casting the expression to a given type *)
    and cast_expr builder = function
        A.Cast (t, e) ->
        L.build_bitcast (cast_expr builder e) (ltype_of_typ t) "tcast" builder
      | e -> unary_expr builder e

    (* Construct code for a relational expression; return its value *)
    and relational_expr builder = function
        A.Binop(e1, ((A.Less|A.Greater|A.Leq|A.Geq) as op), e2) ->
          let e1' = relational_expr builder e1
          and e2' = add_expr builder e2 in
          (match op with
             A.Less -> L.build_icmp L.Icmp.Slt
           | A.Greater -> L.build_icmp L.Icmp.Sgt
           | A.Leq -> L.build_icmp L.Icmp.Sle
           | A.Geq -> L.build_icmp L.Icmp.Sge
           | _ -> raise (Failure("CODEGEN: improper relational expr!!!"))
          ) e1' e2' "relational_expr" builder
      | e -> add_expr builder e

    (* Construct code for an equality expression; return its value *)
    and equality_expr builder = function
        A.Binop(e1, ((A.Equal | A.Neq) as op) , e2)->
          let e1' = equality_expr builder e1 in
          let e2' = if e2 = A.Nullexpr then L.const_null (L.type_of e1')
                else relational_expr builder e2 in
          if e2 != A.Nullexpr then
            (match op with
                 A.Equal -> L.build_icmp L.Icmp.Eq
               | A.Neq -> L.build_icmp L.Icmp.Ne
               | _ -> raise (Failure("CODEGEN: improper equality expr!!!"))
            ) e1' e2' "equality_expr" builder
          else if (L.is_null e1') || (L.is_undef e1') then L.const_int i1_t 1
          else L.const_int i1_t 0
      | e -> relational_expr builder e

    (* Construct code for a logical and expression; return its value *)
    and logical_and_expr builder = function
        A.Binop(e1, A.And, e2) ->
          let e1' = logical_and_expr builder e1
          and e2' = equality_expr builder e2 in
          L.build_and e1' e2' "logical_and_expr" builder
      | e -> equality_expr builder e

(* Construct code for a logical or expression; return its value *)
    and logical_or_expr builder = function
        A.Binop(e1, A.Or, e2) ->
          let e1' = logical_or_expr builder e1
          and e2' = logical_and_expr builder e2 in
          L.build_or e1' e2' "logical_or_expr" builder
      | e -> logical_and_expr builder e

    (* Construct code for an assignment expression; return its value *)
    and assignment_expr builder = function
        A.Assign (A.Dereference(p), _, r) ->
          let r' = expr builder r
          and p' = postfix_expr builder p in
          ignore (L.build_store r' p' builder); r'
      | A.Assign (A.ArrayAccess(a,i), _, r) ->
          let r' = expr builder r in
          let n = L.build_zext_or_bitcast (postfix_expr builder i) i64_t "ary.cast" builder in
          let idx = L.build_gep (expr builder a) [| n |] "ary.ptr" builder in
          ignore(L.build_store r' idx builder); r'
      | A.Assign (A.StructAccess(s,m), _, r) ->
          let r' = expr builder r in
          let name = lookup_type (A.string_of_expr s) in
            let members = snd (struct_lookup (A.string_of_typ name)) in
              let rec get_idx n lst i = match lst with
                | [] -> raise (Failure("CODEGEN: id " ^ m ^ " is not a member of struct " ^ A.string_of_expr s))
                | hd::tl -> if (hd=n) then i else get_idx n tl (i+1)
              in
              let idx = get_idx m (List.map (fun (_,s,_,_) -> s) members) 0 in
            let ptr = L.build_struct_gep (lookup (A.string_of_expr s)) idx "struct.ptr" builder in
          ignore(L.build_store r' ptr builder); r'
      | A.Assign (A.StructPointerAccess(s,m), _, r) ->
          let r' = expr builder r
          and name = strip ( lookup_type (A.string_of_expr s) )
          and location = expr builder s in
            let members = snd (struct_lookup (A.string_of_typ name)) in
              let rec get_idx n lst i = match lst with
                | [] -> raise (Failure("CODEGEN: id " ^ m ^ " is not a member of struct " ^ A.string_of_expr s))
                | hd::tl -> if (hd=n) then i else get_idx n tl (i+1)
              in
              let idx = get_idx m (List.map (fun (_,sn,_,_) -> sn) members) 0 in
            let ptr = L.build_struct_gep location idx "struct.ptr" builder in
          ignore(L.build_store r' ptr builder); r'
      | A.Assign (l, op, r) ->
        let r' = expr builder r in ignore (
        (match op with
            A.Asn -> L.build_store r' (lookup (A.string_of_expr l)) builder
          | A.ModAsn ->
            let reg = lookup (A.string_of_expr l) in
            let tmp = L.build_srem (L.build_load reg "tmp" builder) r' "srem" builder in
            L.build_store tmp reg builder
        )); r' (* QUESTION: what is supposed to be returned here ? *)
      | e -> logical_or_expr builder e

    (* Construct code for an expression; return its value *)
    and expr builder = function
      | e -> assignment_expr builder e

    in (* this in, it is the end, man *)
(***********************************************************)
(* END of mutually recursive chain of expression functions *)
(***********************************************************)

    (* Invoke "f builder" if the current block doesn't already
       have a terminal (e.g., a branch). *)
    let add_terminal builder f =
      match L.block_terminator (L.insertion_block builder) with
	      Some _ -> ()
      | None -> ignore (f builder) in

    (* Build the code for the given statement; return the builder for
       the statement's successor *)
    let rec stmt builder = function
        A.Block sl ->
          (* Go through all global and local vars, initialize them from lazy records *)
          let eval_lazy _ id_val =
            if (id_val.flag && id_val.eval) then
              ignore( id_val.eval <- false; (* the only way to overwrite mutables in OCaml *)
                      L.build_store (expr builder id_val.expr) (lookup id_val.name) builder); in
              ignore(StringMap.iter eval_lazy local_lazy_vars;);
                     (*StringMap.iter eval_lazy global_lazy_vars); *)
          List.fold_left stmt builder sl
      | A.Expr e -> ignore (expr builder e); builder
      | A.Return e -> ignore (match fdecl.A.typ with
    	      A.Void -> L.build_ret_void builder
    	    | _ -> L.build_ret (expr builder e) builder); builder
      | A.If (predicate, then_stmt, else_stmt) ->
         let bool_val = expr builder predicate in
	         let merge_bb = L.append_block context "merge" the_function in

          	 let then_bb = L.append_block context "then" the_function in
          	 add_terminal (stmt (L.builder_at_end context then_bb) then_stmt)
          	   (L.build_br merge_bb);

          	 let else_bb = L.append_block context "else" the_function in
          	 add_terminal (stmt (L.builder_at_end context else_bb) else_stmt)
          	   (L.build_br merge_bb);

          	 ignore (L.build_cond_br bool_val then_bb else_bb builder);
          	 L.builder_at_end context merge_bb

      | A.While (predicate, body) ->
        	  let pred_bb = L.append_block context "while" the_function in
        	  ignore (L.build_br pred_bb builder);

        	  let body_bb = L.append_block context "while_body" the_function in
        	  add_terminal (stmt (L.builder_at_end context body_bb) body)
        	    (L.build_br pred_bb);

        	  let pred_builder = L.builder_at_end context pred_bb in
        	  let bool_val = expr pred_builder predicate in

        	  let merge_bb = L.append_block context "merge" the_function in
        	  ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder);
        	  L.builder_at_end context merge_bb

      | A.For (e1, e2, e3, body) -> stmt builder
	    ( A.Block [A.Expr e1 ; A.While (e2, A.Block [body ; A.Expr e3]) ] )
    in

    (* Build the code for each statement in the function *)
    let builder = stmt builder (A.Block fdecl.A.body) in

    (* Add a return if the last block falls off the end *)
    add_terminal builder (match fdecl.A.typ with
        A.Void -> L.build_ret_void
      | t -> L.build_ret (L.const_int (ltype_of_typ t) 0))
  in

  List.iter build_function_body program.functions;
  the_module
