#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* ============================= Data Structures  ================================== */

char *prop_names[10];
struct node* nodeIndex[1000];
struct node* dummyItem;
struct node* item;
struct edge* edgeIndex[1000];
struct edge* dummyItemEdge;
struct edge* itemEdge;
struct prop* propIndex[1000];
struct prop* dummyItemProp;
struct prop* itemProp;

/* Property Data Structure: Stores a key-value property pair for either a given node or edge */
struct prop {
        int prop_id;
        char* property;
        char* value;
};

/* Node Data Structure: Stores the node object as well as an array of pointers to its property objects */
struct node {
        int node_id;
        int prop_cnt;
        char* node_label;
        struct prop *property[10];
};

/* Edge Data Structure: Stores the edge object as well as pointers to its src (source) and trg (target) node objects */
struct edge {
        int edge_id;
        char* edge_label;
        struct node *src;
        struct node *trg;
};


/* ============================= Creation Functions ================================== */

/* Creates a Node Object and inserts into Hashable Index for Reference */
int insert_node(char* key) {

   struct node *item = (struct node*) malloc(sizeof(struct node));
   item->node_label = strdup(key);
   item->prop_cnt = 0;

   int hashIndex = 0;
   printf("\n\nInsert node: %s ", key);

   while(nodeIndex[hashIndex] != NULL && nodeIndex[hashIndex]->node_id != -1) {
      ++hashIndex;
   }

   item->node_id = hashIndex;
   nodeIndex[hashIndex] = item;
   printf("@ index %i\n\n", hashIndex);
   return hashIndex;
}

/* Creates a Property Object (owned by Node or Edge) and inserts into a Global Hashable Index for Reference */
struct prop *insert_prop(char* key, char* value) {

   struct prop *item = (struct prop*) malloc(sizeof(struct prop));
   item->value = strdup(strtok(value, "\n"));
   item->property = strdup(strtok(key, "\n"));

   int hashIndex = 0;

   while(propIndex[hashIndex] != NULL && propIndex[hashIndex]->property != key) {
      ++hashIndex;
   }

   item->prop_id = hashIndex;
   propIndex[hashIndex] = item;
   return item;
}


/* Creates a Edge Object (owns references to src and trg node pointers) and inserts into Global Hashable Index for Reference */
struct edge *insert_edge(struct node *src, struct node *trg, char* label) {

   struct edge *itemEdge = (struct edge*) malloc(sizeof(struct edge));
   itemEdge->src = src;
   itemEdge->trg = trg;
   itemEdge->edge_label = strdup(label);

   int hashIndex = 0;
   printf("Insert edge: %s\n", label);

   while(edgeIndex[hashIndex] != NULL && edgeIndex[hashIndex]->edge_label != label) {
      ++hashIndex;
   }

   itemEdge->edge_id = hashIndex;
   edgeIndex[hashIndex] = itemEdge;
  printf("inserted new edge @ index: %i\n", hashIndex);
   return itemEdge;
}


/* Count the number of occurance of a character when passed a line of text and the character */
int countChars( char* s, char c )
{
    return *s == '\0'
              ? 0
              : countChars( s + 1, c ) + (*s == c);
}

/* Accepts each line of text from RDMS File (CSV) and processes into Node Objects */
char* create_node(char* line)
{
        int i=0;
        char *array[10];

        int c = countChars(line, ',');
        array[i] = strtok(line,",");

        while(array[i]!=NULL)
        {
                array[++i] = strtok(NULL,",");
        }

        int node_index = insert_node(array[0]);
        int prop_cnt = nodeIndex[node_index]->prop_cnt;
        for (i = 1; i <= c; ++i)
        {
                printf("Inserted -> Prop: %s -> Val: %s\n", prop_names[i], array[i]);
                nodeIndex[node_index]->property[prop_cnt] = insert_prop(prop_names[i], array[i]);
                prop_cnt++;
                nodeIndex[node_index]->prop_cnt = prop_cnt;
        }

        return "done";
}


/* ================================= Graph Functions ============================== */

/* Search for Node based on user input of node_label - uses Hashable Index */
struct node *search() {
   int hashIndex = 0;
   char line[100];
   fgets(line, 100, stdin);

   while(nodeIndex[hashIndex] != NULL) {

      if(strcmp(strtok(nodeIndex[hashIndex]->node_label, "\n"), strtok(line, "\n")) == 0){
         printf("\nSearch Complete -> Found: %s @ index %i", line, hashIndex);
         return nodeIndex[hashIndex];
       }

      printf("Node label %s is compared to %s with value %i\n", nodeIndex[hashIndex]->node_label, line, strcmp(nodeIndex[hashIndex]->node_label, line));
      hashIndex++;
   }
   return NULL;
}

/* Search for Edge based on user input of edge_label - uses Hashable Index */
struct edge *search_edge() {
   int hashIndex = 0;
   char line[100];
   fgets(line, 100, stdin);

   while(edgeIndex[hashIndex] != NULL) {

      if(strcmp(strtok(edgeIndex[hashIndex]->edge_label, "\n"), strtok(line, "\n")) == 0){
         printf("\nSearch Complete -> Found: %s @ index %i", line, hashIndex);
         return edgeIndex[hashIndex];
       }

      printf("Edge label %s is compared to %s with value %i\n", edgeIndex[hashIndex]->edge_label, line, strcmp(edgeIndex[hashIndex]->edge_label, line));
      hashIndex++;
   }
   return NULL;
}

/* Return a pointer to the node reference that matches a particular node_label id */
struct node *get_node(char* node_label) {
   int hashIndex = 0;

   while(nodeIndex[hashIndex] != NULL) {

      if(strcmp(strtok(nodeIndex[hashIndex]->node_label, "\n"), strtok(node_label, "\n")) == 0){
         return nodeIndex[hashIndex];
       }
      hashIndex++;
   }
   return NULL;
}
                                                 
/* Increases the size of the char* pointer to allocate the new item being added */
char* append(char* str1, char* str2) {
   char * str3 = (char *) malloc(1 + strlen(str1)+ strlen(str2) );
   strcpy(str3, str1);
   strcat(str3, str2);
   return str3;
}

/* Display full contents of a graph - iterating and accessing all elements in node, edge, and property indices (through pointer references) */
void display() {
   int i = 0;
   int n_cnt = 0;
   int e_cnt = 0;

   for(i = 0; i<1000; i++) {
      if(nodeIndex[i] != NULL)
         n_cnt++;
   }

   printf("%i \nNode (index, node id)\n\n", n_cnt);
   for(i = 0; i < n_cnt; i++) {
     printf("Node: (%d,%s) \n", nodeIndex[i]->node_id, nodeIndex[i]->node_label);
   }
   printf("\n");

   for(i = 0; i<1000; i++) {
      if(edgeIndex[i] != NULL)
         e_cnt++;
   }

   printf("Graph Summary - Edges: %i \nEdge (index, label, src, trg)\n\n", e_cnt);
   for(i = 0; i < e_cnt; i++) {
     struct node *temp1 = edgeIndex[i]->src;
     struct node *temp2 = edgeIndex[i]->trg;
     printf("Edge: (%d,%s,%s,%s) \n", i, edgeIndex[i]->edge_label, temp1->node_label, temp2->node_label);
   }
   printf("\n");

}

/* Converts the contents of the current graph into a JSON readible format for export */
void display_JSON() {
   int i = 0;
   int f = 0;
   int n_cnt = 0;
   int e_cnt = 0;
   char* output;
   char* tab = "     ";
   char* ttab = "               ";

  char* start = "{\n\n";
   output = start;
   output = append(output, tab);
   output = append(output, "nodes : [\n");
   output = append(output, ttab);

   for(i = 0; i<1000; i++) {
      if(nodeIndex[i] != NULL)
         n_cnt++;
   }

   for(i = 0; i < n_cnt; i++) {
     char* name = nodeIndex[i]->node_label;
     output = append(output, name);
     output = append(output, " : { ");
     int prop_cnt = nodeIndex[i]->prop_cnt;
     int g;
     for(g = 0; g < prop_cnt-1; g++) {
         struct prop *temp= nodeIndex[i]->property[g];
         char* p = temp->property;
         output = append(output, p);
         output = append(output, " : ");
         char* v = temp->value;
         output = append(output, v);
         output = append(output, ", ");
     }
     struct prop *temp= nodeIndex[i]->property[prop_cnt-1];
     char* p = temp->property;
     output = append(output, p);
     output = append(output, " : ");
     char* v = temp->value;
     output = append(output, v);
     output = append(output, " },\n");
     output = append(output, ttab);
   }
   output = append(output, "\n");
   output = append(output, tab);
   output = append(output, "        ],\n");
   output = append(output, "\n\n");
   output = append(output, tab);
   output = append(output, "edges : [\n");
   output = append(output, ttab);

   for(i = 0; i<1000; i++) {
      if(edgeIndex[i] != NULL)
         e_cnt++;
   }

   for(f = 0; f < e_cnt; f++) {
     struct node *temp1 = edgeIndex[f]->src;
     char* src = temp1->node_label;
     struct node *temp2 = edgeIndex[f]->trg;
     char* trg = temp2->node_label;
     char* name = edgeIndex[f]->edge_label;
     output = append(output, name);
     output = append(output, " : { ");
     output = append(output, "source");
     output = append(output, " : ");
     output = append(output, src);
     output = append(output, ", ");
     output = append(output, "target");
     output = append(output, " : ");
     output = append(output, trg);
     output = append(output, " },\n");
     output = append(output, ttab);
   }

   output = append(output, "\n");
   output = append(output, tab);
   output = append(output, "        ]\n");
   output = append(output,"\n}");
   printf("%s", output);
   printf("\n");

}


/* Display contents of a node object by accepting the node pointer as parameter */
void display_node(struct node *node_id) {
   int prop_cnt = node_id->prop_cnt;
   printf("\nProperty Count: %i\n\n", prop_cnt);

   int i;
   for(i = 0; i < prop_cnt; i++) {
         struct prop *temp= node_id->property[i];
         printf("Property: %s, Value: %s\n", temp->property, temp->value);
   }

   printf("\n");
}

/* Display contents of a edge object by accepting the edge pointer as a parameter */
void display_edge(struct edge *edge_input) {

    struct node *temp1 = edge_input->src;
    struct node *temp2 = edge_input->trg;
                                                                                                                                                                           
    printf("\nEdge (index, label, src, trg)\n\n");
    printf("Edge: (%d,%s,%s,%s) \n", edge_input->edge_id, edge_input->edge_label, temp1->node_label, temp2->node_label);
   printf("\n");
}



/* Main script that accepts a user input file and generates all of the Node Objects and Property (Prop) Objects that is owns. (can use sample: t_node.csv, t_node2.csv) */
void read_nodes() {
   int line_count;
   char buffer[1024];
   char *line;
   char fileparam[80];
   scanf("%s", fileparam);
   FILE *fstream = fopen(fileparam,"r");
   if(fstream == NULL)
   {
      printf("\n file opening failed ");
      read_nodes();
   }

   int i = 0;
   line_count = 0;
   line = fgets(buffer,sizeof(buffer),fstream);
   printf("first line: %s\n", line);
   prop_names[i] = strtok(line,",");
   while(prop_names[i]!=NULL){
      prop_names[++i] = strtok(NULL,",");
   }

   int t = 0;
   for(t = 0; t < i; t++){
        prop_names[t] = strdup(prop_names[t]);
   }

   while((line=fgets(buffer,sizeof(buffer),fstream))!=NULL)
   {
     line_count++;
     create_node(line);
   }
   printf("\nSuccessfully imported %i nodes",line_count);
}

/* Read and Import the file provided for Edges into memory and reference appropriate Node Objects (can use sample: t_edge.csv) */
void read_edges() {
   int line_count;
   char buffer[1024];
   char *line;
   char fileparam[80];
   scanf("%s", fileparam);
   FILE *fstream = fopen(fileparam,"r");
   if(fstream == NULL)
   {
      printf("\n file opening failed ");
      read_nodes();
   }

   int i = 0;
   line_count = 0;
   line = fgets(buffer,sizeof(buffer),fstream);
   printf("first line: %s\n", line);

   while((line=fgets(buffer,sizeof(buffer),fstream))!=NULL)
   {
     printf("processing line %s\n", line);

     line_count++;
     int i = 0;
     char *array[10];
     array[i] = strtok(line,",");

     while(array[i]!=NULL)
     {
         array[++i] = strtok(NULL,",");
     }

     printf("search for %s\n", array[1]);
     struct node *temp1 = get_node(array[1]);
     printf("search for %s\n", array[2]);
     struct node *temp2 = get_node(array[2]);
     insert_edge(temp1, temp2, array[0]);
   }
   printf("Successfully imported %i edges",line_count);
}


void menu() {

    printf("\nManagement Console: (at any point type 'help' show this menu)\n\n");
    printf("Option 1: Import node file\n");
    printf("Option 2: Import edge file\n");
    printf("Option 3: Query node\n");
    printf("Option 4: Query edge\n");
    printf("Option 5: Search path\n");
    printf("Option 6: Print summary statistics\n");
    printf("Option 7: Export graph to JSON\n\n");
    printf("Please make your selection now: ");
}

void logic(char* input) {
  if(strcmp(input, "help") == 1){
    menu();
  }
  else {
  switch (*input) {
  case '1':
    printf("\nPlease provide the path to your node files (csv):\n");
    read_nodes();
    break;
  case '2':
    printf("Please provide the path to your edge files (csv):\n");
    read_edges();
    printf("\n=============================================================\n\n");
    break;
  case '3':
    printf("Enter the node id: ");
    struct node *temp_node = search();
    display_node(temp_node); 
    printf("\n=============================================================\n\n");
    break;
  case '4':
    printf("Enter the edge id: ");
    struct edge *temp_edge = search_edge();
    display_edge(temp_edge);
    printf("\n=============================================================\n\n");
    break;
  case '5':
    printf("Enter node1: ");
    printf("Enter node2: ");
    printf("\n=============================================================\n\n");
    break;
  case '6':
    printf("\nGraph Summary - Nodes: ");
    display();
    printf("\n=============================================================\n\n");
    break;
  case '7':
    printf("\nJSON Export: \n");
    display_JSON();
    printf("\n=============================================================\n\n");
    break;
  default:
    printf("\nPlease make another selection\n");
    menu();
    break;}
  }
}


/* ========================== Main Function ======================================= */

int main()
{

    printf("\n================================= Welcome to C+ Graph =============================\n\n");
    printf("C+ Graph is an in-memory application that allows you to\n transform RBMS files (SQL) into a native graph abstraction.\nWe use a Property Graph architecture so once you\nhave imported your data into C+ Graph you will be able to\ncreate graph queries and search using node or edge properties.\n");
    char line[1024];
    printf("\n=========================== please hit enter to continue ==========================\n\n");
    fgets(line, 1024, stdin);
    menu();

    while (fgets(line, 1024, stdin))
    {
       char* tmp = strdup(line);
       logic(tmp);
       free(tmp);
    }
   return 0;
}
                                                                                                                                                                           
                                                                                                                                                                           
                                                                                                                                                                           

