(* Based on the MicroC llvm. Modified by Niles, Jack, and Chelci *)

module L = Llvm
module A = Ast

(*Custom modules*)
(*module Asgn = Assign *)

module StringMap = Map.Make(String)

let translate (globals, functions) =
  let context = L.global_context () in
  let the_module = L.create_module context "LOON"
  and i32_t  = L.i32_type  context
  and i8_t   = L.i8_type   context
  and i1_t   = L.i1_type   context
  and void_t = L.void_type context in

  (* Define the array type *)
  let arr_type = L.pointer_type (L.pointer_type i8_t) in

  let rec ltype_of_typ = function
      A.Int -> (*ignore(print_endline("int gets called..."));*) i32_t
    | A.Char -> i8_t
    | A.Bool -> i1_t
    | A.String -> (*ignore(print_endline("str gets called..."));*) L.pointer_type i8_t
    | A.Array -> arr_type
    | A.Pair typ -> L.pointer_type (L.struct_type context [| L.pointer_type i8_t; ltype_of_typ typ |] )
    | A.Json -> arr_type
    | A.Void -> void_t in

  (* Declare each global variable; remember its value in a map *)
  let global_vars =
    let global_var m (t, n) =
      let init = L.const_int (ltype_of_typ t) 0
      in StringMap.add n (L.define_global n init the_module) m in
    List.fold_left global_var StringMap.empty globals in
(*  print_endline (string_of_bool (StringMap.mem "helloTest" global_vars)); *)

  (* Declare printf(), which the print built-in function will call *)
  let printf_t = L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
  let printf_func = L.declare_function "printf" printf_t the_module in

  (*Declare scanf(), which reads from stdin *)
  let loon_scanf_t = L.function_type (ltype_of_typ A.String) [| |] in
  let loon_scanf_func = L.declare_function "loon_scanf" loon_scanf_t the_module in


  (* Declare the built-in printbig() function *)
  let printbig_t = L.function_type i32_t [| i32_t |] in
  let printbig_func = L.declare_function "printbig" printbig_t the_module in

  (* Define each function (arguments and return type) so we can call it *)
  let function_decls =
    let function_decl m fdecl =
      let name = fdecl.A.fname
      and formal_types =
        Array.of_list (List.map (fun (t,_) -> ltype_of_typ t) fdecl.A.formals) in
      let ftype = L.function_type (ltype_of_typ fdecl.A.primitive) formal_types in
    StringMap.add name (L.define_function name ftype the_module, fdecl) m in
  List.fold_left function_decl StringMap.empty functions in

  (* Fill in the body of the given function *)
  let build_function_body fdecl =
    let (the_function, _) = StringMap.find fdecl.A.fname function_decls in
    let builder = L.builder_at_end context (L.entry_block the_function) in

    let format_str str_val = L.build_global_stringptr str_val "fmt" builder in

    (* Construct the function's "locals": formal arguments and locally
       declared variables.  Allocate each on the stack, initialize their
       value, if appropriate, and remember their values in the "locals" map *)
	(* Map of ids -> unmodified ocaml expression *)
	let id_vals_map = ref StringMap.empty in

    let local_vars =
      let add_formal m (t, n) p = L.set_value_name n p;
    let local = L.build_alloca (ltype_of_typ t) n builder in
    ignore (L.build_store p local builder);
    StringMap.add n local m in

      let add_local m (t, n) =
    let local_var = L.build_alloca (ltype_of_typ t) n builder in
    ignore(id_vals_map := StringMap.add n (A.zero_of_typ t) !id_vals_map );
    StringMap.add n local_var m in

      let formals = List.fold_left2 add_formal StringMap.empty fdecl.A.formals
          (Array.to_list (L.params the_function)) in
      List.fold_left add_local formals fdecl.A.locals in

    (* Return the value for a variable or formal argument *)
    let lookup n = try StringMap.find n local_vars
                   with Not_found -> StringMap.find n global_vars

    (* Map with:
    key - Id name
    value - A.val_type list
    List contains value type for each element in a given array *)
    and arr_types_map = ref StringMap.empty
    (* id to list of types *)
    and json_types_map = ref StringMap.empty

    (* id to map of keys -> index *)
    and json_lookup_map = ref StringMap.empty in
    let add_arr_types id = function
        | A.Val_list types_list -> StringMap.add id types_list !arr_types_map
        | _ -> ignore(print_endline("ADD_ARR_TYPES: ERROR - Bad input to array types map")); StringMap.empty
    and get_arr_types id = StringMap.find id !arr_types_map
    and is_arr id = StringMap.mem id !arr_types_map
    and add_json_types id types_list = StringMap.add id types_list !json_types_map
    and add_json_keys id l =
        let new_string_map = StringMap.empty in
        let add_to_map (map, index) next =
            (StringMap.add next index map, index + 1)
        in
        let new_string_map = fst (List.fold_left add_to_map (new_string_map, 0) l) in
        json_lookup_map := StringMap.add id new_string_map !json_lookup_map

    (* Stack containing lists of value types for each array.
       Should only be one array's list of types on stack at any given time *)
    and arr_types_stack = ref (A.Val_list [])
    and json_types_stack = ref []
    and json_keys_stack = ref []
    and key_lookup = ref StringMap.empty
    and current_key = ref ""
    in

    (* Construct code for an expression; return its value *)
    let rec expr builder = function
        A.Literal i -> L.const_int i32_t i
      | A.BoolLit b -> L.const_int i1_t (if b then 1 else 0)
      | A.CharLit c -> L.const_int i8_t (int_of_char c)
      (* StringLit constructs a private address that points to the argument value's contents *)
      | A.StringLit s ->
            L.build_global_stringptr s "str" builder
      | A.PairLit (k, v) -> (* need to eval both k and v *)
            (* Evaluate both k and v *)
            let key_string = expr builder k and
            value = expr builder v in

            let key_string_as_string =
                match k with
                | A.StringLit s -> s
                | _ -> "error" in
            ignore(current_key:= key_string_as_string);

            (* Define our bespoke pair type *)
            let pair_type = L.struct_type context [| L.pointer_type i8_t; L.type_of value |] in

            (* Allocate an object of type pair *)
            let allocated_struct = L.build_alloca pair_type "pair" builder in

            let place_for_key =
                L.build_in_bounds_gep allocated_struct [| (L.const_int i32_t 0); (L.const_int i32_t 0) |] "key_addr" builder in

            ignore(L.build_store key_string place_for_key builder);

            let place_for_value =
                L.build_in_bounds_gep allocated_struct [| (L.const_int i32_t 0); (L.const_int i32_t 1) |] "val_addr" builder in

            ignore(L.build_store value place_for_value builder);

            allocated_struct

      | A.Noexpr -> L.const_int i32_t 0
      | A.ArrayLit l -> let arr_size = L.const_int i32_t (List.length l) in
            (* Allocate space for values and types*)
            let arr_space =  L.build_array_alloca (L.pointer_type i8_t) arr_size "arr" builder in

            (* Function to load each individual value *)
            let load_object (indx , temp_types_list) expr_val =
                let llvm_indx = [| L.const_int i32_t indx|]
                and llvm_expr = (expr builder expr_val) in

                (* Allocate space for given type and store*)
                let val_type = L.type_of llvm_expr in
                let stored_val = L.build_alloca val_type "arr_val" builder in
                ignore(L.build_store llvm_expr stored_val builder);

                (* Get pointer to the new value and cast it to i8 pointer *)
                let void_elem_ptr = L.build_bitcast stored_val (L.pointer_type i8_t) "cast_val" builder
                and arr_indx = L.build_in_bounds_gep arr_space llvm_indx "arr_pos" builder in

                (* Store the pointer to the value in the arr, return updated types list and next indx *)
                ignore(L.build_store void_elem_ptr arr_indx builder) ;

                (* If val is an array, get its list of types - else get val_type of primitive *)
                let indx_elem_type =
                    if val_type = arr_type then (

                (* Get stack of types *)
                let top_of_stack = (match !arr_types_stack with
                                        | A.Val_list ts -> ts
                                        | _ -> []) in

                (* Get top of stack and make tail new stack *)
                    let l_of_types = List.hd top_of_stack in
                    ignore(arr_types_stack := A.Val_list (List.tl top_of_stack));
                    l_of_types
                    ) else( A.Val (L.pointer_type val_type) ) in
                        match temp_types_list with
                        | A.Val_list lts -> (indx + 1, A.Val_list (indx_elem_type :: lts))
                        | _ -> (indx +1, A.Val_list [])  in

                let old_stack = (match !arr_types_stack with
                                        | A.Val_list ts -> ts
                                        | _ -> []) in
                let res_list = (match(snd (List.fold_left load_object (0, A.Val_list []) l)) with
                    | A.Val_list f_list -> f_list
                    | _ -> [] ) in
        ignore(arr_types_stack := A.Val_list (A.Val_list (List.rev res_list) :: old_stack) );
        arr_space
      | A.JsonLit l ->
            let rec unzip_keys_and_vals = function
                [] -> ([], [])
              | (k, v) :: rest ->
                      let everything_else = unzip_keys_and_vals rest in
                      (k :: (fst everything_else), v :: (snd everything_else))
            in
            let unzipped = unzip_keys_and_vals l in
            let keys = fst unzipped
            and vals = snd unzipped in

            let keys_as_strings = List.map (fun x -> match x with
                | A.StringLit s -> s
                | _ -> "Undefined") keys in

            let arr_size = L.const_int i32_t (List.length l) in
            (* Allocate space for values and types*)
            let arr_space =  L.build_array_alloca (L.pointer_type i8_t) arr_size "arr" builder in

            (* Function to load vals*)
            let load_object (indx , temp_types_list) expr_val =
                let llvm_indx = [| L.const_int i32_t indx|]
                (* HANDLE IDS HERE *)
                and llvm_expr = (expr builder expr_val) in

                (* Allocate space for given type and store*)
                let val_type = L.type_of llvm_expr in
                let stored_val = L.build_alloca val_type "arr_val" builder in
                ignore(L.build_store llvm_expr stored_val builder);

                (* Get pointer to the new value and cast it to i8 pointer *)
                let void_elem_ptr = L.build_bitcast stored_val (L.pointer_type i8_t) "cast_val" builder
                and arr_indx = L.build_in_bounds_gep arr_space llvm_indx "arr_pos" builder in

                (* Store the pointer to the value in the arr, return updated types list and next indx *)
                ignore(L.build_store void_elem_ptr arr_indx builder);

                let indx_elem_type = L.pointer_type val_type in

                (indx + 1, indx_elem_type :: temp_types_list)
            in

            (* Get the list of types for this array *)
            let res_list = snd (List.fold_left load_object (0, []) vals)
            in

            (* Push the list of types for this array onto stack and return address of this array literal *)
            ignore(json_types_stack := List.rev res_list);
            ignore(json_keys_stack := keys_as_strings);
            arr_space
      | A.Id s -> L.build_load (lookup s) s builder
      | A.Binop (e1, op, e2) ->
        let e1' = expr builder e1
        and e2' = expr builder e2
        (* Semantic checking ensures that two are of same type - can insert additional check for float/int conversions *)
        and check_expr_type e_1 e_2 = (match e_1 with
                A.StringLit s1 -> ignore(s1); "i8*"
              | A.Literal i -> ignore(i); "i32"
              | A.Id id ->
                    if StringMap.mem id !json_types_map then (
                        match e_2 with
                            A.Id id -> if StringMap.mem id !json_types_map then ("json+json")
                                        else ("json+pair")
                          | _ -> "error"
                    )
                    else if StringMap.mem id !key_lookup then (
                    "pair+pair")
                    else
                    L.string_of_lltype (L.element_type (L.type_of (lookup id)))
            (* Arrays will also go here *)
              | _ -> ignore(print_endline ("ERROR: CHECK_EXPR_TYPE: Invalid operand") ); "null") in
        (match (check_expr_type e1 e2) with
            "i8*" ->
                let concat_str estr1 estr2 = (A.string_of_expr estr1) ^ (A.string_of_expr estr2) in
                let m_op =
                    match op with
                        A.Add -> L.build_global_stringptr

                        (* Should never happen given semantically correct tree
                        - Should still replace with L.NULL eventually *)
                    | _ ->
                        ignore(print_endline ("Not PLUS op which is a problem..." ) );
                        L.build_global_stringptr
                in
                m_op (concat_str e1 e2) "str" builder
    (* Everything else is an int/float *)
          | "i32" ->
            (match op with
              (* Overload add to perform string concat*)
                A.Add     -> L.build_add
              | A.Sub     -> L.build_sub
              | A.Mult    -> L.build_mul
              | A.Div     -> L.build_sdiv
              | A.And     -> L.build_and
              | A.Or      -> L.build_or
              | A.Equal   -> L.build_icmp L.Icmp.Eq
              | A.Neq     -> L.build_icmp L.Icmp.Ne
              | A.Less    -> L.build_icmp L.Icmp.Slt
              | A.Leq     -> L.build_icmp L.Icmp.Sle
              | A.Greater -> L.build_icmp L.Icmp.Sgt
              | A.Geq     -> L.build_icmp L.Icmp.Sge
            ) e1' e2' "tmp" builder
          | "pair+pair" ->
                let get_val pair =
                    let pointer_to_value =
                        L.build_in_bounds_gep pair [| (L.const_int i32_t 0); (L.const_int i32_t 1) |] "val_addr" builder in

                    let return_value = L.build_load pointer_to_value "" builder in
                    return_value
                in

                let first_val = get_val e1'
                and second_val = get_val e2' in
                let first_id = match e1 with
                    A.Id id -> id
                  | _ -> "error"
                in
                let second_id = match e2 with
                    A.Id id -> id
                  | _ -> "error"
                in
                let first_key_string = StringMap.find first_id !key_lookup
                and second_key_string = StringMap.find second_id !key_lookup in

                let keys = [first_key_string; second_key_string]
                and vals = [first_val; second_val] in

                let arr_size = L.const_int i32_t (List.length keys) in (* Allocate space for values and types*)
                let arr_space =  L.build_array_alloca (L.pointer_type i8_t) arr_size "arr" builder in

                (* Function to load vals*)
                let load_object (indx , temp_types_list) llvm_expr =
                    let llvm_indx = [| L.const_int i32_t indx|] in

                    (* Allocate space for given type and store*)
                    let val_type = L.type_of llvm_expr in
                    let stored_val = L.build_alloca val_type "arr_val" builder in
                    ignore(L.build_store llvm_expr stored_val builder);

                    (* Get pointer to the new value and cast it to i8 pointer *)
                    let void_elem_ptr = L.build_bitcast stored_val (L.pointer_type i8_t) "cast_val" builder
                    and arr_indx = L.build_in_bounds_gep arr_space llvm_indx "arr_pos" builder in

                    (* Store the pointer to the value in the arr, return updated types list and next indx *)
                    ignore(L.build_store void_elem_ptr arr_indx builder);

                    let indx_elem_type = L.pointer_type val_type in

                    (indx + 1, indx_elem_type :: temp_types_list)
                in

                (* Get the list of types for this array *)
                let res_list = snd (List.fold_left load_object (0, []) vals)
                in

                (* Push the list of types for this array onto stack and return address of this array literal *)
                ignore(json_types_stack := List.rev res_list);
                ignore(json_keys_stack := keys);
                arr_space

          | "json+pair" ->
                let get_val pair =
                    let pointer_to_value =
                        L.build_in_bounds_gep pair [| (L.const_int i32_t 0); (L.const_int i32_t 1) |] "val_addr" builder in

                    let return_value = L.build_load pointer_to_value "" builder in
                    return_value
                in

                let pair_val = get_val e2' in
                let pair_id = match e2 with
                    A.Id id -> id
                  | _ -> "error"
                and json_id = match e1 with
                    A.Id id -> id
                  | _ -> "error"
                in
                let key_string = StringMap.find pair_id !key_lookup in

                let json_lookup = StringMap.find json_id !json_lookup_map in
                let json_types = StringMap.find json_id !json_types_map in
                let fold_key_func key _ acc = key :: acc in

                let load_val acc key_string =
                    let index = StringMap.find key_string json_lookup in

                    (* Gets type to cast to *)
                    let type_of_res = List.nth json_types index in
                    (* Function to get element pointer to desired element in an array *)

                    let llvm_index = L.const_int i32_t index in
                    let elemptr = L.build_gep (L.build_load (lookup json_id) "" builder) [|llvm_index|] "" builder in
                    let arr_val = L.build_load elemptr "" builder in
                    let cast_val = L.build_bitcast arr_val type_of_res "cast" builder in
                    L.build_load cast_val "" builder :: acc
                in
                let json_keys = StringMap.fold fold_key_func json_lookup [] in
                let json_vals = List.fold_left load_val [] json_keys in

                let keys = List.rev(key_string :: json_keys)

                and vals = List.rev(pair_val :: List.rev json_vals) in

                let arr_size = L.const_int i32_t (List.length keys) in
                (* Allocate space for values and types*)
                let arr_space =  L.build_array_alloca (L.pointer_type i8_t) arr_size "arr" builder in

                (* Function to load vals*)
                let load_object (indx , temp_types_list) llvm_expr =
                    let llvm_indx = [| L.const_int i32_t indx|] in

                    (* Allocate space for given type and store*)
                    let val_type = L.type_of llvm_expr in
                    let stored_val = L.build_alloca val_type "arr_val" builder in
                    ignore(L.build_store llvm_expr stored_val builder);

                    (* Get pointer to the new value and cast it to i8 pointer *)
                    let void_elem_ptr = L.build_bitcast stored_val (L.pointer_type i8_t) "cast_val" builder
                    and arr_indx = L.build_in_bounds_gep arr_space llvm_indx "arr_pos" builder in

                    (* Store the pointer to the value in the arr, return updated types list and next indx *)
                    ignore(L.build_store void_elem_ptr arr_indx builder);

                    let indx_elem_type = L.pointer_type val_type in

                    (indx + 1, indx_elem_type :: temp_types_list)
                in

                (* Get the list of types for this array *)
                let res_list = snd (List.fold_left load_object (0, []) vals)
                in

                (* Push the list of types for this array onto stack and return address of this array literal *)
                ignore(json_types_stack := List.rev res_list);
                ignore(json_keys_stack := keys);
                arr_space
          | _ -> ignore(print_endline ("NO SUITABLE BINARY OPERATIONS FOUND FOR LEFT OPERAND")); L.const_null i32_t)
      | A.Unop(op, e) ->
          let e' = expr builder e in
            (match op with
              A.Neg     -> L.build_neg e' "tmp" builder
            | A.Not     -> L.build_not e' "tmp" builder
            | A.Deref   ->
                let pointer_to_value =
                    L.build_in_bounds_gep e' [| (L.const_int i32_t 0); (L.const_int i32_t 1) |] "key_addr" builder in

                let return_value = L.build_load pointer_to_value "" builder in
                return_value
        )
      | A.Access(id, indx_lst) ->
        (* If id is a polymorphic array, cast pointer type accordingly*)
        if StringMap.mem id !json_types_map then (
            (* Get the first value in the indx_list as a string *)
            (* Assume that only primitives in json, never need more than first *)
            let index_string = A.string_of_expr (List.hd indx_lst)
            and types = StringMap.find id !json_types_map in
            let lookup_map = StringMap.find id !json_lookup_map in
            let index = StringMap.find index_string lookup_map in

            (* Gets type to cast to *)
            let type_of_res = List.nth types index in
            (* Function to get element pointer to desired element in an array *)

            let llvm_index = L.const_int i32_t index in
            let elemptr = L.build_gep (L.build_load (lookup id) "" builder) [|llvm_index|] "" builder in
            let arr_val = L.build_load elemptr "" builder in
            let cast_val = L.build_bitcast arr_val type_of_res "cast" builder in
            L.build_load cast_val "" builder)

        (* If id is a polymorphic array (it should be, this is really just a safety check), cast pointer type accordingly*)
        else (
            (* Function to get element pointer to desired element in an array *)
            let pos_finder prev_pos indx =
               (* Load Array value ( i8* ) *)
               let llv_of_indx = expr builder indx
               and load_of_orig = L.build_load prev_pos "" builder in

               (* Cast loaded i8* to an i8*** because it must be address of an array *)
               let cast_of_load = L.build_bitcast load_of_orig (L.pointer_type arr_type) "temp_cast" builder in

               (* Get address of element at desired index position *)
               L.build_gep (L.build_load cast_of_load "" builder) [|llv_of_indx|] "" builder in

            (* Get the initial value to index through *)
            let first_indx = List.hd indx_lst in
            let init_res = L.build_gep (L.build_load (lookup id) "" builder) [|(expr builder first_indx)|] "" builder in

            (* Fold list of index positions in order to get element pointer to final index position
            NOTE: If List.tl indx_list yields the empty list, the result of the below call is equal to init_res *)
            let final_pos = List.fold_left pos_finder init_res (List.tl indx_lst) in
            let arr_val = L.build_load final_pos "" builder in (*ignore(print_endline("ACCESS: Loaded value is: " ^ (L.string_of_llvalue arr_val) ^ " with type " ^ (L.string_of_lltype (L.type_of arr_val))));*)

            if StringMap.mem id !arr_types_map then
            (* First invoke function to map each expr in list to ocaml int *)
            (let expr_to_int e_i = match e_i with
                               A.Literal i -> i
                               | A.Id s ->  let oc_val = StringMap.find s !id_vals_map in
											(match oc_val with
												A.Literal s_i -> (*ignore(print_endline("Id is literal with val: " ^ (string_of_int s_i)));*) 												s_i
                                           		| _ -> ignore(print_endline("EXPR_TO_INT: Not a Literal - array index not possible")); -1
                                            )
                                | _ -> ignore(print_endline("ACCESS: ERROR: Bad type passed into access element")); 0 in

            let indxs_int_lst = List.map expr_to_int indx_lst in

            (* Gets type to cast the element pointer to *)
            let type_of_res = A.get_val_type context indxs_int_lst (A.Val_list(get_arr_types id)) in
            let cast_val = L.build_bitcast arr_val (type_of_res) "cast" builder in

            L.build_load cast_val "" builder) else arr_val)

      | A.Assign (s, lst, e) -> let e' = expr builder e in
            (* Check if you are assigning to array index position *)
            if(lst = []) then (

                (* No access assignment, simply load into id's address space *)
				ignore(id_vals_map := StringMap.add s e !id_vals_map );
                ignore (L.build_store e' (lookup s) builder);
                (* In case of array assignment, pop the stack for list of val_types and then add to map *)
                if L.element_type (L.type_of (lookup s)) = L.pointer_type (L.pointer_type i8_t) then (
                ignore(
                  match e with
                    A.JsonLit _ ->
                        ignore(json_types_map := add_json_types s !json_types_stack );
                        ignore(json_types_stack := []);
                        ignore(add_json_keys s !json_keys_stack);
                        ignore(json_keys_stack := []);
                  | A.Binop _ ->
                        ignore(json_types_map := add_json_types s !json_types_stack );
                        ignore(json_types_stack := []);
                        ignore(add_json_keys s !json_keys_stack);
                        ignore(json_keys_stack := []);
                  | A.ArrayLit _ ->
                        let popped_val = match !arr_types_stack with
                        | A.Val_list lstv -> lstv
                        | _ -> [] in

                        ignore(arr_types_map := add_arr_types s (List.hd popped_val) );
                        ignore(arr_types_stack := A.Val_list(List.tl popped_val));
                  | _ -> ()
                );

                e') else (
                    (* If not a Json object or an array, check if it's a pair (lit or id)
                    If it is, then add its key to key_lookup and return, else just return *)
                    match e with
                      | A.PairLit _ ->
                        (ignore(key_lookup := StringMap.add s !current_key !key_lookup); e')
                      | A.Id id -> ignore(print_endline id);
                        if StringMap.mem id !key_lookup then
                          (let key = StringMap.find id !key_lookup in
                              ignore(print_endline s);
                              ignore(print_endline key);
                              ignore(key_lookup := StringMap.add s key !key_lookup); e')
                        else
                              e'
                      | _ -> e'
                )

            ) else(
                (* Access assignment *)

                (* Function to get element pointer to desired element in an array *)
                let pos_finder prev_pos indx =
                    let llv_of_indx = expr builder indx
                    and load_of_orig = L.build_load prev_pos "" builder in

                    let cast_of_load = L.build_bitcast load_of_orig (L.pointer_type arr_type) "temp_cast" builder in
                    L.build_gep (L.build_load cast_of_load "" builder) [|llv_of_indx|] "" builder in

                (* Get the initial value to index through *)
                let first_indx = List.hd lst in
                let init_res = L.build_gep (L.build_load (lookup s) "" builder) [|(expr builder first_indx)|] "" builder in

                (* Fold list of index positions in order to get element pointer to final index position
                NOTE: If List.tl indx_list yields the empty list, the result of the below call is equal to init_res *)
                let final_pos = List.fold_left pos_finder init_res (List.tl lst) in

                (* Turn the list of indx positions into a list of ints *)
                let expr_to_int e_i = match e_i with
                               A.Literal i -> i
                               | A.Id s ->  let oc_val = StringMap.find s !id_vals_map in
											(match oc_val with
											A.Literal s_i -> (*ignore(print_endline("Id is literal with val: " ^ (string_of_int s_i)));*) 												s_i
                                            | _ -> ignore(print_endline("EXPR_TO_INT: Not a Literal - array index not possible")); -1
                                            )
                                | _ -> ignore(print_endline("ACCESS: ERROR: Bad type passed into access element")); 0 in

               let indxs_int_lst = List.map expr_to_int lst

                (* If new type is an array, pop types from stack - otherwise just wrap val type *)
               and new_type = let elt = L.type_of e' in
                              if elt = arr_type then(
                                  (*ignore(print_endline("ASSIGN: new value to store is array: " ^ (L.string_of_lltype elt)));*)
                                  let true_stack = match !arr_types_stack with
                                        | A.Val_list tlst -> tlst
                                        | _ -> [] in
                                  let new_types_list = List.hd true_stack in
                                  ignore(arr_types_stack := A.Val_list(List.tl true_stack));
                                  new_types_list)
                              else ((*ignore(print_endline("ASSIGN: new value to store is NOT array: " ^ (L.string_of_lltype elt))); *)
                                    A.Val(elt))

                (* Get the current list of types for id *)
                and types_lst = get_arr_types s in

                (* Set the array value type at the specified position to the new value's type *)
                let new_types_lst = A.set_val_type context types_lst new_type indxs_int_lst in

                (* Add the updated list of types for this id to the map - then allocate space for type of the new value *)
                ignore(arr_types_map := add_arr_types s (A.Val_list new_types_lst));

                let alloc_new_val = L.build_alloca (A.get_val_type context [] new_type) "assign_acc_val" builder in

                (* Store new value in allocated space, bitcast the pointer to it to void ptr *)
                ignore(L.build_store e' alloc_new_val builder);
                let cast_new_val = L.build_bitcast alloc_new_val (L.pointer_type i8_t) "assign_tmp_cast" builder in

                (* Store cast value into the index position of the original array *)
                ignore(L.build_store cast_new_val final_pos builder); e'
            )

      | A.Call ("printJSON", lst) | A.Call ("printb", lst) -> let rec print_builder (fmt_str, lst_init) = (if (List.length lst_init)=0 then
                            (fmt_str ^ "\n", [] )
                            else (let x = List.hd lst_init in
                                let str_new = match x with
                                A.StringLit s1 -> ignore(s1); fmt_str ^ "%s"
                                | A.CharLit c1 -> ignore(c1); fmt_str ^ "%c"
                                | A.Literal i1 -> ignore(i1); fmt_str ^ "%d"
                                | A.BoolLit b1 -> ignore(b1); fmt_str ^ "%d"
                                | A.Id id1 -> let int_type = L.pointer_type (ltype_of_typ A.Int)
                                        and bool_type = L.pointer_type (ltype_of_typ A.Bool)
                                        and str_type = L.pointer_type (ltype_of_typ A.String)
                                        and char_type = L.pointer_type (ltype_of_typ A.Char) in
                                        (*and id_type = L.type_of (lookup id1) in *) (*ignore(print_endline ("ID_CATCH: Before match int_type: " ^ (L.string_of_lltype int_type) ));*)
                                        let type_match llt = if llt = str_type then
                                            "%s"
                                        else if llt = int_type then "%d" else if llt = char_type then ("%c")
                                        else if llt = bool_type then ("%d")
                                        else ((*ignore(print_endline ("ID_CATCH: Bad match...")); *) "Bad")
                                    in fmt_str ^ (type_match (L.type_of (lookup id1)))
                                | A.Access (id, indx) -> if is_arr id then(

                                (* Get proper type *)
                                let expr_to_int e_i = match e_i with
                                    A.Literal i -> i
                                    | A.Id s ->  let oc_val = StringMap.find s !id_vals_map in
											(match oc_val with
												A.Literal s_i -> (*ignore(print_endline("Id is literal with val: " ^ (string_of_int s_i)));*)
															s_i
	                                            | _ -> ignore(print_endline("EXPR_TO_INT: Not a Literal - array index not possible")); -1
                                            )
                                    | _ -> ignore(print_endline("ACCESS: ERROR: Bad type passed into access element")); 0 in

                                let indxs_int_lst = List.map expr_to_int indx in

                                let llt_of_val = L.element_type (A.get_val_type context indxs_int_lst (A.Val_list(get_arr_types id))) in

                                fmt_str ^ (A.fmt_of_lltype (L.string_of_lltype llt_of_val)) ) else fmt_str ^ "%c"
                                | _ -> (*ignore(print_endline ("PRINT_BUILDER: Head is unknown type..."));*) fmt_str ^ "BAD"

                                in let res = print_builder (str_new, (List.tl lst_init)) in
                                ((fst res), (expr builder x) :: (snd res) ) )) in
                            let full_args = print_builder ("", lst) in
                                L.build_call printf_func (Array.of_list ((format_str (fst full_args)) :: (snd full_args) )) "printf" builder
      | A.Call ("printbig", [e]) ->
	  L.build_call printbig_func [| (expr builder e) |] "printbig" builder

      | A.Call ("loon_scanf", _ ) ->
          L.build_call loon_scanf_func [| |] "loon_scanf" builder

      (*| A.Call ("loon_scanf", [e]) ->failwith "why not? scanf"
	  (*L.build_call scanf_func [| (expr builder e) |] "loon_scanf" builder*)*)
      | A.Call (f, act) ->
            let (fdef, fdecl) = StringMap.find f function_decls in
            let actuals = List.rev (List.map (expr builder) (List.rev act)) in
            let result = (match fdecl.A.primitive with A.Void -> ""
                                            | _ -> f ^ "_result") in
         L.build_call fdef (Array.of_list actuals) result builder
    in


    (* Invoke "f builder" if the current block doesn't already
       have a terminal (e.g., a branch). *)
    let add_terminal builder f =
      match L.block_terminator (L.insertion_block builder) with
    Some _ -> ()
      | None -> ignore (f builder) in

    (* Build the code for the given statement; return the builder for
       the statement's successor *)
    let rec stmt builder = function
    A.Block sl -> List.fold_left stmt builder sl
      | A.Expr e -> ignore (expr builder e); builder
      | A.Return e -> ignore (match fdecl.A.primitive with
        A.Void -> L.build_ret_void builder
    | _ -> L.build_ret (expr builder e) builder); builder
      | A.If (predicate, then_stmt, else_stmt) ->
            let bool_val = expr builder predicate in
            let merge_bb = L.append_block context "merge" the_function in

            let then_bb = L.append_block context "then" the_function in
            add_terminal (stmt (L.builder_at_end context then_bb) then_stmt)
            (L.build_br merge_bb);

            let else_bb = L.append_block context "else" the_function in
            add_terminal (stmt (L.builder_at_end context else_bb) else_stmt)
            (L.build_br merge_bb);

            ignore (L.build_cond_br bool_val then_bb else_bb builder);
            L.builder_at_end context merge_bb

      | A.While (predicate, body) ->
            let pred_bb = L.append_block context "while" the_function in
            ignore (L.build_br pred_bb builder);

            let body_bb = L.append_block context "while_body" the_function in
            add_terminal (stmt (L.builder_at_end context body_bb) body)
            (L.build_br pred_bb);

            let pred_builder = L.builder_at_end context pred_bb in
            let bool_val = expr pred_builder predicate in

            let merge_bb = L.append_block context "merge" the_function in
            ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder);
            L.builder_at_end context merge_bb

      | A.For (e1, e2, e3, body) -> stmt builder
            ( A.Block [A.Expr e1 ; A.While (e2, A.Block [body ; A.Expr e3]) ] )
        in

    (* Build the code for each statement in the function *)
    let builder = stmt builder (A.Block fdecl.A.body) in

    (* Add a return if the last block falls off the end *)
    add_terminal builder (match fdecl.A.primitive with
        A.Void -> L.build_ret_void
      | t -> L.build_ret (L.const_int (ltype_of_typ t) 0))
  in
  List.iter build_function_body functions;
  the_module
