(* LOON ast.mli. Written by Chelci, Jack, Niles, and Kyle *)
module L = Llvm

type op =
    (* numerical operators *)
    | Add | Sub | Mult | Div | Equal
    (* Relational operators *)
    | Neq | Less | Leq | Greater | Geq
    (* boolean operators *)
    | And | Or

type uop =
    | Neg | Not | Deref

type typ =
    | Int | Bool | Void | String | Pair of typ | Char | Array | Json

type bind = typ * string

type expr =
    | Literal of int
    | BoolLit of bool
    | CharLit of char
    | StringLit of string
    | PairLit of expr * expr
    | Id of string
    | Noexpr
    | Binop of expr * op * expr
    | Unop of uop * expr
	| Assign of string * expr list * expr
    | Call of string * expr list
	| Access of string * expr list
    | ArrayLit of expr list
    | JsonLit of (expr * expr) list

type stmt =
    | Block of stmt list
    | Expr of expr
    | If of expr * stmt * stmt
    | For of expr * expr * expr * stmt
    | While of expr * stmt
    | Return of expr

type func_decl = {
    primitive	: typ;
    fname 		: string;
    formals 	:	bind list;
    locals		: bind list;
    body 		: stmt list;
}

type program = bind list * func_decl list

(** Wrapper around array value types *)
type val_type = 
		| Val of L.lltype
		| Val_list of val_type list 

val string_of_program : bind list * func_decl list -> string

val zero_of_typ : typ -> expr

(* Pretty-printing functions *)

val string_of_op : op -> string

val string_of_uop : uop -> string

val string_of_expr : expr -> string

val string_of_stmt : stmt -> string

val string_of_typ : typ -> string

val string_of_vdecl : bind -> string

val string_of_fdecl : func_decl -> string

val fmt_of_lltype : string -> string

val get_val_type : L.llcontext -> int list -> val_type -> L.lltype 

val set_val_type :  L.llcontext -> val_type list -> val_type -> int list -> val_type list
