module VGA_LED_Emulator(
 input logic 	    clk50, reset,
 input logic [9:0]  posx,posy,
 input logic [3:0] s_shape,
 input logic s_invert,
 input logic [2:0] s_count,
 input logic [2:0] id,
 input logic layer,
 input logic [3:0] life_1, life_2,
 input logic [7:0] bR,bG,bB,
 output logic [7:0] VGA_R, VGA_G, VGA_B,
 output logic 	    VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);
 
 
	parameter N_SPRITES = 8;
	parameter N_GROUND = 9;

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
	
   always_ff @(posedge clk50 or posedge reset)
		if (reset) hcount <= 0;
		else if (endOfLine) hcount <= 0;
		else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 			     vcount;
   logic 			     endOfField;
   
	always_ff @(posedge clk50 or posedge reset)
		if (reset)
			vcount <= 0;
		else if (endOfLine)
			if (endOfField)
				vcount <= 0;
			else
				vcount <= vcount + 10'd 1;
	


   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge


/*------------------------------------------------------------------------------------------------*/
	
	//Declarations
/******************************************************************************/
	genvar i;

	//Sprites
	logic [2:0] count[N_SPRITES];	//count of sprite
	logic [3:0] shape[N_SPRITES];	//Shape of sprite
	logic [2:0] count_gnd[N_GROUND];	//count of sprite

	logic [7:0] Rpix[2];					//color red
	logic [7:0] Gpix[2];					//color green
	logic [7:0] Bpix[2];					//color blue
	logic [1:0] Apix;						//active bit
	logic inv[N_SPRITES]; 			//invert sprite

	//Memory

	logic [4:0] x;
	logic [8:0] y;
	logic [4:0] gx, gy;
	logic invert;
	logic [4:0] code[2];

	//Sprite Control
	logic [9:0] cur_x, cur_y;
	logic [1:0] enable;		//enable to display sprite on screen

	//Icons
	logic [3:0] xc;	 			//coordinate x
	logic [4:0] yc; 			//coordinate y
	logic enable_c;
	logic [2:0] life_p1,life_p2;
	logic icon_a;		 		//output active bit
	logic [7:0] Ri,Gi,Bi;	//output colors
	

	//Position input data	
	logic [9:0]px[N_SPRITES];
	logic [9:0]py[N_SPRITES];

	logic [9:0]px_gnd[N_GROUND];
	logic [9:0]py_gnd[N_GROUND];

/******************************************************************************/

	//Input data
/******************************************************************************/
	logic [3:0]id_ground;

	logic [N_SPRITES-1:0]reg_enable;
	logic [N_GROUND-1:0]reg_enb_gnd;

	assign id_ground[2:0] = id;
	assign id_ground[3] = s_shape[0];

	decoder #(N_SPRITES) (id, reg_enable);
	decoder #(N_GROUND) (id_ground, reg_enb_gnd);

	generate
		for(i = 0; i < N_GROUND; i++) begin: loop_gnd
			register #(10) pos_x0(clk50,reset,reg_enb_gnd[i]&(~layer),posx,px_gnd[i]);
			register #(10) pos_y0(clk50,reset,reg_enb_gnd[i]&(~layer),posy,py_gnd[i]);
			register #(3) cnt0(clk50,reset,reg_enb_gnd[i]&(~layer),s_count,count_gnd[i]);
		end
	endgenerate	
	
	generate
		for(i = 0; i < N_SPRITES; i++) begin: loop_modules
			register #(10) pos_x1(clk50,reset,reg_enable[i]&layer,posx,px[i]);
			register #(10) pos_y1(clk50,reset,reg_enable[i]&layer,posy,py[i]);
			register #(3) cnt1(clk50,reset,reg_enable[i]&layer,s_count,count[i]);
			register #(1) inv1(clk50,reset,reg_enable[i]&layer,s_invert,inv[i]);
			register #(4) shp1(clk50,reset,reg_enable[i]&layer,s_shape,shape[i]);
		end
	endgenerate
	
	assign life_p1 = life_1;
	assign life_p2 = life_2;
/******************************************************************************/

	//Modules instantiation
/******************************************************************************/
	logic [7:0] R,G,B;	//output colors
	
	assign cur_x = hcount[10:1];
	assign cur_y = vcount;
	
	groundcontrol gndctrl(cur_x, cur_y, px_gnd, py_gnd, count_gnd,
		gx, gy, enable[0]);

	spritecontrol spritectrl(cur_x, cur_y, px, py, shape, count, inv,
		x, y, enable[1], invert);
	
	icon_control ictrl(.*);

	icons picon(.*);

	color_gnd cmemgnd(gx,gy,code[0]);
	color_mem cmem(x,y,invert,code[1]);

	active_gnd amemgnd(gx,gy,Apix[0]);
	active_mem amem(x,y,invert,Apix[1]);

	color_code code_col_0(code[0],Rpix[0],Gpix[0],Bpix[0]);
	color_code code_col_1(code[1],Rpix[1],Gpix[1],Bpix[1]);
	
	
	background back(.*);
/******************************************************************************/

	//Pixel Selection for printing
/******************************************************************************/
	logic ivisible;
	logic [1:0] visible;
	logic [1:0] enb;

	assign visible = enable & Apix;
	assign ivisible = enable_c & icon_a;

	assign enb[0] = ivisible | (~visible[1])&visible[0];
	assign enb[1] = ivisible | visible[1];
/******************************************************************************/

	//Pixel printing
/******************************************************************************/
	always_comb begin
		case(enb)
			2'b00: {VGA_R, VGA_G, VGA_B} = {R, G, B}; 				// Background
			2'b01: {VGA_R, VGA_G, VGA_B} = {Rpix[0], Gpix[0], Bpix[0]};	//Layer 0
			2'b10: {VGA_R, VGA_G, VGA_B} = {Rpix[1], Gpix[1], Bpix[1]};	//Layer 1
			2'b11: {VGA_R, VGA_G, VGA_B} = {Ri, Gi, Bi};				//Icons layer
		endcase
	end

/******************************************************************************/

/*------------------------------------------------------------------------------------------------*/
	
endmodule // VGA_LED_Emulator



//Primitime modules
/******************************************************************************/
module decoder #(parameter N = 3) (input [N-1:0] DataIn, output logic [(1<<N)-1:0] DataOut);
	 always_comb begin
		 DataOut <= 1 << DataIn;
	  end
endmodule


module register(clock,reset,enable,datain,dataout);
parameter size = 16;

input logic clock;
input logic reset;
input enable;
input logic [size-1:0] datain;
output logic [size-1:0] dataout;

	always_ff @(posedge clock) begin
		if (reset)
			dataout <= 0;
		else if(enable)
			dataout <= datain;
	end

endmodule

/******************************************************************************/


//Image display control
/******************************************************************************/
module icon_control(
input logic [9:0] cur_x, cur_y,
input logic [3:0] life_1,life_2,
output logic [3:0] xc,
output logic [4:0] yc,
output logic enable_c);

	logic [1:0] inlife;
	logic in_x1, in_x2, in_y;
	logic [3:0] xc1, xc2;
	logic [4:0] yc1, yc2;
	
	assign enable_c =  inlife[0] | inlife[1];

	assign xc1 = cur_x - 10'd41;
	assign xc2 = cur_x - 10'd541;

	assign yc1 = cur_y - 10'd21;
	assign yc2 = cur_y - 10'd5;
	
	assign yc = (inlife[1]) ? yc2 : yc1;
	assign xc = (inlife[1]) ? xc2 : xc1;

	assign in_x1 = (cur_x > 10'd40)&
				   (cur_x < (10'd57 + 10'd16*(life_1-4'd1)));
	
	assign in_x2 = (cur_x > 10'd540)&
				   (cur_x < (10'd557 + 10'd16*(life_2-4'd1)));

	assign in_y = (cur_y > 10'd20) & (cur_y < 10'd37);


	assign inlife[0] = in_x1 & in_y & (life_1[0] | life_1[1] | life_1[2] | life_1[3]);
	assign inlife[1] = in_x2 & in_y & (life_2[0] | life_2[1] | life_2[2] | life_2[3]);

endmodule // iconcontrol


module spritecontrol(
input logic [9:0] cur_x, cur_y,
input logic [9:0] px[8],
input logic [9:0] py[8],
input logic [3:0] shape[8],
input logic [2:0] count[8],
input logic inv[8],
output logic [4:0] x,
output logic [8:0] y,
output logic enable,
output logic invert);

parameter N_SPRITES = 8;

	logic visible[N_SPRITES];		//visibility of sprite
	logic inSprite[N_SPRITES];
	logic [8:0]sh_shape[N_SPRITES];
	
	genvar i;

	
	assign invert =/**/
					(inSprite[7] & inv[7]) |
					(inSprite[6] & inv[6]) |
					(inSprite[5] & inv[5]) |
					(inSprite[4] & inv[4]) |
					(inSprite[3] & inv[3]) |
					(inSprite[2] & inv[2]) |
					(inSprite[1] & inv[1]) |
					(inSprite[0] & inv[0]);


	assign enable =/**/
					(inSprite[7] & visible[7]) | 
					(inSprite[6] & visible[6]) |
					(inSprite[5] & visible[5]) |
					(inSprite[4] & visible[4]) |
					(inSprite[3] & visible[3]) |
					(inSprite[2] & visible[2]) |
					(inSprite[1] & visible[1]) |
					(inSprite[0] & visible[0]);

	logic [9:0] ax, ay;

	assign ay =  /**/
					(inSprite[7]) ? (py[7] - sh_shape[7]) :
					(inSprite[6]) ? (py[6] - sh_shape[6]) :
					(inSprite[5]) ? (py[5] - sh_shape[5]) :
					(inSprite[4]) ? (py[4] - sh_shape[4]) :
					(inSprite[3]) ? (py[3] - sh_shape[3]) :
					(inSprite[2]) ? (py[2] - sh_shape[2]) :
					(inSprite[1]) ? (py[1] - sh_shape[1]) :
					(inSprite[0]) ? (py[0] - sh_shape[0]) :
					8'd0;

	assign ax =  /**/
					(inSprite[7]) ? px[7] :
					(inSprite[6]) ? px[6] :
					(inSprite[5]) ? px[5] :
					(inSprite[4]) ? px[4] :
					(inSprite[3]) ? px[3] :
					(inSprite[2]) ? px[2] :
					(inSprite[1]) ? px[1] :
					(inSprite[0]) ? px[0] :
					4'd0;

	assign y = cur_y + 16 - ay;
	assign x = cur_x + 16 - ax;
	
	
	generate
		for(i = 0; i < N_SPRITES; i++) begin: loop_spr
			assign inSprite[i] =((cur_x > (px[i] - 10'd17))&
								(cur_x < (px[i] + 10'd16 + 10'd32*(count[i]-1)))&
								(cur_y > (py[i] - 10'd17))&
								(cur_y < (py[i] + 10'd16)));
		end
	endgenerate
	
	generate
		for(i = 0; i < N_SPRITES; i++) begin: loop_sha
			assign sh_shape[i][8:5] = shape[i];
			assign sh_shape[i][4:0] = 5'd0;
		end
	endgenerate
	
	
	generate
		for(i = 0; i < N_SPRITES; i++) begin: loop_vis
			assign visible[i] = (count[i][0] | count[i][1] | count[i][2]);
		end
	endgenerate
	
endmodule // spritecontrol


module groundcontrol(cur_x, cur_y, px, py, count, x, y, enable);
parameter N_GROUND = 9;

input logic [9:0] cur_x, cur_y;
input logic [9:0] px[N_GROUND];
input logic [9:0] py[N_GROUND];
input logic [2:0] count[N_GROUND];
output logic [4:0] x;
output logic [4:0] y;
output logic enable;

	logic visible[N_GROUND];		//visibility of sprite
	logic inSprite[N_GROUND];
	logic [9:0] ax, ay;

	genvar i;
	
	assign enable =/**/
					(inSprite[8] & visible[8]) | 
					(inSprite[7] & visible[7]) | 
					(inSprite[6] & visible[6]) |
					(inSprite[5] & visible[5]) |
					(inSprite[4] & visible[4]) |
					(inSprite[3] & visible[3]) |
					(inSprite[2] & visible[2]) |
					(inSprite[1] & visible[1]) |
					(inSprite[0] & visible[0]);

	assign ay =/**/
					(inSprite[8]) ? py[8] :
					(inSprite[7]) ? py[7] :
					(inSprite[6]) ? py[6] :
					(inSprite[5]) ? py[5] :
					(inSprite[4]) ? py[4] :
					(inSprite[3]) ? py[3] :
					(inSprite[2]) ? py[2] :
					(inSprite[1]) ? py[1] :
					(inSprite[0]) ? py[0] :
					4'd0;

	assign ax =/**/
					(inSprite[8]) ? px[8] :
				 	(inSprite[7]) ? px[7] :
					(inSprite[6]) ? px[6] :
					(inSprite[5]) ? px[5] :
					(inSprite[4]) ? px[4] :
					(inSprite[3]) ? px[3] :
					(inSprite[2]) ? px[2] :
					(inSprite[1]) ? px[1] :
					(inSprite[0]) ? px[0] :
					4'd0;	
	
	assign y = cur_y + 16 - ay;
	assign x = cur_x + 16 - ax;

	generate
		for(i = 0; i < N_GROUND; i++) begin: loop_spr
			assign inSprite[i] =((cur_x > (px[i] - 10'd17))&
								(cur_x < (px[i] + 10'd16 + 10'd32*(count[i]-1)))&
								(cur_y > (py[i] - 10'd17))&
								(cur_y < (py[i] + 10'd16)));
		end
	endgenerate
	
	generate
		for(i = 0; i < N_GROUND; i++) begin: loop_vis
			assign visible[i] = (count[i][0] | count[i][1] | count[i][2]);
		end
	endgenerate
	
endmodule // spritecontrol
/******************************************************************************/


//icons
/******************************************************************************/
module icons(
input logic clk50,
input logic [3:0] xc,
input logic [4:0] yc,
output logic [7:0] Ri,Gi,Bi,
output logic icon_a);

	logic [15:0] a;
	logic [63:0] c;
	logic [4:0] icon_c;
	
	assign icon_c = c[xc*4 +: 4];
	assign icon_a = a[xc];
	
	active_icon act_ic(clk50, yc, a);
	
	colors_icon col_ic(clk50, yc, c);
	
	icon_code code(icon_c, Ri, Gi, Bi);
	
endmodule


module colors_icon(
input logic clk ,
input logic [4:0] address,
output logic [63:0] data_out);

	always_ff @(posedge clk)
	begin
		data_out <= mem[address];
	end
	
	logic [63:0] mem[0:31] = '{
		//grandma
		64'h11111000000,
		64'h111111110000,
		64'h123221111000,
		64'h23222121000,
		64'h222222221000,
		64'h24422211000,
		64'h22222211000,
		64'h15555511000,
		64'h555555551100,
		64'h5555555555100,
		64'h2255555522100,
		64'h2226666222000,
		64'h2266666622000,
		64'h66006600000,
		64'h422002240000,
		64'h4444004444000,
		//grandpa
		64'h1111100000,
		64'h11111110110,
		64'h27221111110,
		64'h227222121000,
		64'h2212221121000,
		64'h111222211000,
		64'h22222200000,
		64'h888898880000,
		64'h8888888888000,
		64'h88888898888800,
		64'h22888888882200,
		64'h222aaaaaa22200,
		64'h22aaaaaaaa2200,
		64'haaa00aaa0000,
		64'hbbb0000bbb000,
		64'hbbbb0000bbbb00};
endmodule


module active_icon(
input logic clk ,
input logic [4:0] address,
output logic [15:0] data_out);

	always_ff @(posedge clk)
	begin
		data_out <= mem[address];
	end
	
	logic [15:0] mem[0:31] = '{
		//grandma
		16'h7c0,
		16'hff0,
		16'hff8,
		16'h7f8,
		16'hff8,
		16'h7f8,
		16'h7f8,
		16'h7f8,
		16'hffc,
		16'h1ffc,
		16'h1ffc,
		16'h1ff8,
		16'h1ff8,
		16'h660,
		16'he70,
		16'h1e78,
		//grandpa			
		16'h3e0,
		16'h7f6,
		16'h7fe,
		16'hff8,
		16'h1ff8,
		16'hff8,
		16'h7e0,
		16'hff0,
		16'h1ff8,
		16'h3ffc,
		16'h3ffc,
		16'h3ffc,
		16'h3ffc,
		16'he70,
		16'h1c38,
		16'h3c3c};

endmodule
/******************************************************************************/


//Sprites
/******************************************************************************/

//Top Level
/*---------------------------*/
//General Sprites Color
module color_mem(
input logic [4:0]x,
input logic [8:0]y,
input logic invert,
output logic [4:0] code);

	logic [7:0]posx;

	logic [6:0]s_x;
	
	logic [159:0] col;

	assign s_x[6:2] = x;
	assign s_x[1:0] = 2'd0;
	assign posx = (invert) ? (155 - (s_x + x)):(s_x + x);
	assign code = col[posx +: 5];
	
	twoport_color color(y,col);
endmodule

//Ground Sprites Color
module color_gnd(
input logic [4:0]gx,
input logic [4:0]gy,
output logic [4:0] code);

	logic [7:0]posx;

	logic [6:0]s_x;
	
	logic [159:0] col;

	assign s_x[6:2] = gx;
	assign s_x[1:0] = 2'd0;
	assign posx = s_x + gx;
	assign code = col[posx +: 5];
	
	ground_color color(gy,col);

endmodule

//General Sprites Active
module active_mem(
input logic [4:0]x,
input logic [8:0]y,
input invert,
output logic Apix);

	logic [4:0] xa;
	logic [31:0] ac;
	
	assign xa = (invert) ? (31-x) : (x);

	assign Apix = ac[xa];
	
	//active memory
	twoport_active active(y,ac);

endmodule

//Ground Sprites Active
module active_gnd(
input logic [4:0]gx,
input logic [4:0]gy,
output logic Agnd);

	logic [31:0] ac;

	assign Agnd = ac[gx];
	
	//active memory
	ground_active active(gy,ac);

endmodule
/*---------------------------*/


//Memory Modules
/*---------------------------*/
//General Sprites Color memory
module twoport_color(
input logic [8:0] aa,
output logic [159:0] qa);
	
	assign qa = mem[aa];
	
	//Sprite memory
	logic [159:0] mem[0:191] = '{
		//ground
		160'h0,
		160'h0,
		160'h0,
		160'h0,
		160'h84210000000000000000000000000,
		160'h21084210840000000084000000008000,
		160'h842108421084421042108421084210842108421,
		160'h842108421084421084210842108421084108421,
		160'h8421084421086318c6318c6318c421086210823,
		160'h18c6318c6310c6318c6318c6318c4210c6318c63,
		160'h18c6318c6310c6318c6318c6318c421084318c43,
		160'h18c6318c4218c6318c6318c4218c6318c6318842,
		160'h18c631884318c6318c6318c4318c6318c6318c63,
		160'h18c631086318c6318c6318c4318c6318c6318c63,
		160'h108421086318c6318c6318c421086310c4210842,
		160'h188421086318c6318c6318c42108421084210842,
		160'h18c6318c631086318c6318c621084318c6318c63,
		160'h18c6318c6318c4210842108421084318c6318c63,
		160'h18c6318c6318c6318c6318c631086318c6318c63,
		160'h18c6318c6318c6318c6318c631086318c6318c63,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		//grandma
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108429484210842108421084,
		160'h21084210842108421085318a4210842108421084,
		160'h210842108421084210a6318c5210842108421084,
		160'h21084210842108421085318a4210842108421084,
		160'h210842108421085294a631484210842108421084,
		160'h2108421084210a6318c6318a4210842108421084,
		160'h2108421084214c6318c6318c5210842108421084,
		160'h2108421084214c739ce739cc5210842108421084,
		160'h2108421084214e7420e7420e7290842108421084,
		160'h2108421084214e7420e7420e7290842108421084,
		160'h2108421084214e739ce739ce7290842108421084,
		160'h2108421084214e739ce739ce7290842108421084,
		160'h2108421084214a739d2939ce5414842108421084,
		160'h21084210852a10539ce739ca8420a42108421084,
		160'h21084210a842108294a529508421052108421084,
		160'h21084210a8421084210842108421052108421084,
		160'h21084210a739508421084210829ce52108421084,
		160'h21084210a739508421084210829ce52108421084,
		160'h21084210a739508421084210529ce52108421084,
		160'h21084210a739ca8421084210539ce52108421084,
		160'h21084210a73954a5294a5294529ce52108421084,
		160'h21084210852a94a5294a52945214a42108421084,
		160'h21084210842a94a294a552945210842108421084,
		160'h21084210842a94a2908552945210842108421084,
		160'h21084210842a94a2908552945210842108421084,
		160'h210842108549ce72908539ce9290842108421084,
		160'h21084214a949ce72908539ce9494a42108421084,
		160'h210842a5294a529290854a5294a5252108421084,
		160'h21085294a5294a529085294a5294a52908421084,
		//grandma jump
		160'h2108421084210842108429484210842108421084,
		160'h21084210842108421085318a4210842108421084,
		160'h210842108421084210a6318c5210842108421084,
		160'h21084294842108421085318a421084294a421084,
		160'h2108539ca421085294a6314842108539ce521084,
		160'h210a739ce5210a6318c6318a4210a739ce729084,
		160'h210a739ce5214c6318c6318c5210a739ce729084,
		160'h2108539ce7294c739ce739cc5214e739ce521084,
		160'h2108429ce7394e7420e7420e729ce739ca421084,
		160'h21084214e7414e7420e7420e729ce73948421084,
		160'h21084210a8414e739ce739ce72a0e72908421084,
		160'h2108421085414e739ce739ce72a1052108421084,
		160'h21084210842a0a739d2939ce5420a42108421084,
		160'h21084210842150539ce739ca8414842108421084,
		160'h210842108421508294a529508414842108421084,
		160'h2108421084210a84210842108290842108421084,
		160'h2108421084210a84210842108290842108421084,
		160'h2108421084214a84210842108290842108421084,
		160'h21084210842a1084210842108290842108421084,
		160'h2108421085421084210842108290842108421084,
		160'h21084210aa5294a5294a5294a290842108421084,
		160'h21084210aa5294a5294a5294a290842108421084,
		160'h21084210a939ce5294a55294a290842108421084,
		160'h210842152939ce5210855294a290842108421084,
		160'h210842a5294a525210855294a290842108421084,
		160'h210842a5294a4a5210a939ce5210842108421084,
		160'h21084214a529484210a939ce5210842108421084,
		160'h210842108421084210a94a525210842108421084,
		160'h210842108421084210a94a525210842108421084,
		160'h210842108421084210a94a525210842108421084,
		160'h210842108421084210a94a525210842108421084,
		160'h21084210842108421085294a4210842108421084,
		//grandpa
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842948421084210842108421084,
		160'h210842108421085318a421084210842108421084,
		160'h210842108421085318c521084210842108421084,
		160'h210842108421084298c529084210842108421084,
		160'h210842108421085318c631484210842108421084,
		160'h2108421084210a6318c6318a4210842108421084,
		160'h2108421084214c6318c6318c5210842108421084,
		160'h2108421084214c739ce739cc5210842108421084,
		160'h2108421084214e75ace75ace7290842108421084,
		160'h2108421084214e75ace75ace7290842108421084,
		160'h2108421084214e739ce739ce7290842108421084,
		160'h2108421084214e739cc639ce7290842108421084,
		160'h2108421084214a7398c631ce5614842108421084,
		160'h21084210852b18539ce739cac630a42108421084,
		160'h21084210ac6318c294a52958c631852108421084,
		160'h21084210ac6318c6318c6318c631852108421084,
		160'h21084210a73958c6358c6b18c29ce52108421084,
		160'h21084210a73958c6318c6318c29ce52108421084,
		160'h21084210a7395ce739ce739c529ce52108421084,
		160'h21084210a739cae739ce739c539ce52108421084,
		160'h21084210a7395ce739ce739c529ce52108421084,
		160'h21084210852b9ce739ce739c5214a42108421084,
		160'h21084210842b9ce294a5739c5210842108421084,
		160'h21084210842b9ce29085739c5210842108421084,
		160'h21084210842b9ce29085739c5210842108421084,
		160'h21084210857bdef290857bdef290842108421084,
		160'h21084214a57bdef290857bdef294a42108421084,
		160'h210842bdef7bdef290857bdef7bde52108421084,
		160'h21085294a5294a529085294a5294a52908421084,
		//grandpa jump
		160'h2108421084210842948421084210842108421084,
		160'h210842108421085318a421084210842108421084,
		160'h210842108421085318c521084210842108421084,
		160'h210842948421084298c52908421084294a421084,
		160'h2108539ca421085318c6314842108539ce521084,
		160'h210a739ce5210a6318c6318a4210a739ce729084,
		160'h210a739ce5214c6318c6318c5210a739ce729084,
		160'h2108539ce7294c739ce739cc5214e739ce521084,
		160'h2108429ce7394e75ace75ace729ce739ca421084,
		160'h21084214e7614e75ace75ace729ce73948421084,
		160'h21084210ac614e739ce739ce72b0e72908421084,
		160'h2108421085614e739cc639ce72b1852108421084,
		160'h21084210842b0a7398c631ce5630a42108421084,
		160'h21084210842158539ce739cac614842108421084,
		160'h21084210842158c294a52958c614842108421084,
		160'h2108421084210ac6318c6318c290842108421084,
		160'h2108421084210ac6358c6b18c290842108421084,
		160'h2108421084214ac6318c6318c290842108421084,
		160'h21084210842b9ce739ce739ce290842108421084,
		160'h2108421085739ce739ce739ce290842108421084,
		160'h21084210af7bdce739ce739ce290842108421084,
		160'h21084210af7bdee739ce739ce290842108421084,
		160'h21084210af7bdef294a5739ce290842108421084,
		160'h21084215ef7bde521085739ce290842108421084,
		160'h210842bdef7bde521085739ce290842108421084,
		160'h210842bdef7bca5210af7bde5210842108421084,
		160'h21084214a529484210af7bde5210842108421084,
		160'h210842108421084210af7bde5210842108421084,
		160'h210842108421084210af7bde5210842108421084,
		160'h210842108421084210af7bde5210842108421084,
		160'h210842108421084210af7bde5210842108421084,
		160'h21084210842108421085294a4210842108421084,
		//wheelchair
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421231810842108421084,
		160'h21084210842108421084246739ca318423194084,
		160'h2108421084210842108424a739ce739ce4421084,
		160'h2108421084210842108424e739ce739ce2421084,
		160'h2108421084210842421024e739ce739ce2421084,
		160'h2108421084242539ce739ce739ce739ce0421084,
		160'h2108421084242308408424e739ca5294a0421084,
		160'h2108421084810842108424e3294e739ca0421084,
		160'h21084210848c23194e739ca739ce739ca0421084,
		160'h21084210848ce739ce739ca739ce739ce3121084,
		160'h21084210848ce739ce739ce739ce739ce539c484,
		160'h21084210848ce739ce739ce319ce7394e3024a04,
		160'h21084210848ce739ce739ce739ce739ce3024644,
		160'h21084210848c2339ce739ce739ce739ce3089250,
		160'h21084210848ca738c2139ce728ca719465091211,
		160'h210842121194271242339c09094e528c67321211,
		160'h21084212109424421253946339ca2424a7184611,
		160'h210842420489264246118ca739ca5084e7281211,
		160'h21084244909125089210946538c62424e7121230,
		160'h210842449094652212319425284a248ce7089250,
		160'h210848409294093842518ca7194a3184e3024664,
		160'h210848c0939ce109c610946308c65024a3124624,
		160'h21084890939c6448ca739c49084a518c61084884,
		160'h24210812139ca4424a108ca308ca508909194084,
		160'h24e738109394e04844848ce72212729121381084,
		160'h2121194611944848908494e32212129ce5021084,
		160'h2108421232946319108494a52210842108421084,
		160'h2108421084846318908494651210842108421084,
		160'h2108421084210842108424a44210842108421084};

endmodule

//Ground Color memory
module ground_color(
input logic [4:0] aa,
output logic [159:0] qa);
	
	assign qa = mem[aa];
	
	//Sprite memory
	logic [159:0] mem[0:31] = '{
		//ground
		160'h0,
		160'h0,
		160'h0,
		160'h0,
		160'h84210000000000000000000000000,
		160'h21084210840000000084000000008000,
		160'h842108421084421042108421084210842108421,
		160'h842108421084421084210842108421084108421,
		160'h8421084421086318c6318c6318c421086210823,
		160'h18c6318c6310c6318c6318c6318c4210c6318c63,
		160'h18c6318c6310c6318c6318c6318c421084318c43,
		160'h18c6318c4218c6318c6318c4218c6318c6318842,
		160'h18c631884318c6318c6318c4318c6318c6318c63,
		160'h18c631086318c6318c6318c4318c6318c6318c63,
		160'h108421086318c6318c6318c421086310c4210842,
		160'h188421086318c6318c6318c42108421084210842,
		160'h18c6318c631086318c6318c621084318c6318c63,
		160'h18c6318c6318c4210842108421084318c6318c63,
		160'h18c6318c6318c6318c6318c631086318c6318c63,
		160'h18c6318c6318c6318c6318c631086318c6318c63,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084,
		160'h2108421084210842108421084210842108421084};

endmodule

//General Sprites Active memory
module twoport_active(
input logic [7:0] aa,
output logic [31:0] qa);
		
	assign qa = mem[aa];
		
	logic [31:0] mem[0:191] = '{
		//ground
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		//grandma
		32'h0,
		32'h0,
		32'h0,
		32'hc000,
		32'h1e000,
		32'h3f000,
		32'h1e000,
		32'h1fc000,
		32'h3fe000,
		32'h7ff000,
		32'h7ff000,
		32'h7ff800,
		32'h7ff800,
		32'h7ff800,
		32'h7ff800,
		32'h7ffc00,
		32'h1fffe00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h1fff600,
		32'hfff000,
		32'hf9f000,
		32'hf9f000,
		32'h1f9f800,
		32'h7f9fe00,
		32'hff9ff00,
		32'h1ff9ff80,
		//grandma jump
		32'hc000,
		32'h1e000,
		32'h3f000,
		32'hc01e0e0,
		32'h1e1fc1f0,
		32'h3f3fe3f8,
		32'h3f7ff3f8,
		32'h1ffff7f0,
		32'hfffffe0,
		32'h7ffffc0,
		32'h3ffff80,
		32'h1ffff00,
		32'hfffe00,
		32'h7ffc00,
		32'h7ffc00,
		32'h3ff800,
		32'h3ff800,
		32'h7ff800,
		32'hfff800,
		32'h1fff800,
		32'h3fff800,
		32'h3fff800,
		32'h3fff800,
		32'h7f1f800,
		32'hff1f800,
		32'hff3f000,
		32'h7c3f000,
		32'h3f000,
		32'h3f000,
		32'h3f000,
		32'h3f000,
		32'h1e000,
		//grandpa
		32'h0,
		32'h0,
		32'h0,
		32'hc0000,
		32'h1e0000,
		32'h1f0000,
		32'hf8000,
		32'h1fc000,
		32'h3fe000,
		32'h7ff000,
		32'h7ff000,
		32'h7ff800,
		32'h7ff800,
		32'h7ff800,
		32'h7ff800,
		32'h7ffc00,
		32'h1fffe00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h3ffff00,
		32'h1fff600,
		32'hfff000,
		32'hf9f000,
		32'hf9f000,
		32'h1f9f800,
		32'h7f9fe00,
		32'hff9ff00,
		32'h1ff9ff80,
		//grandpa jump
		32'hc0000,
		32'h1e0000,
		32'h1f0000,
		32'hc0f80e0,
		32'h1e1fc1f0,
		32'h3f3fe3f8,
		32'h3f7ff3f8,
		32'h1ffff7f0,
		32'hfffffe0,
		32'h7ffffc0,
		32'h3ffff80,
		32'h1ffff00,
		32'hfffe00,
		32'h7ffc00,
		32'h7ffc00,
		32'h3ff800,
		32'h3ff800,
		32'h7ff800,
		32'hfff800,
		32'h1fff800,
		32'h3fff800,
		32'h3fff800,
		32'h3fff800,
		32'h7f1f800,
		32'hff1f800,
		32'hff3f000,
		32'h7c3f000,
		32'h3f000,
		32'h3f000,
		32'h3f000,
		32'h3f000,
		32'h1e000,
		//wheelchair
		32'h0,
		32'h0,
		32'h0,
		32'h3800,
		32'h7ffc,
		32'h7fe0,
		32'h7fe0,
		32'h77fe0,
		32'h7fffe0,
		32'h7c7fe0,
		32'h807fe0,
		32'hffffe0,
		32'hfffff0,
		32'hfffffc,
		32'hfffff6,
		32'hfffff6,
		32'hfffffb,
		32'hfffffb,
		32'h3f7dff3,
		32'h3e3fe7f,
		32'h6a7fffb,
		32'h5bbfe73,
		32'h5f3fefb,
		32'hddffff6,
		32'hdffff76,
		32'h9efdffc,
		32'h7be7ff9c,
		32'h79ecf3b8,
		32'h3fc8f3f0,
		32'h3f8f000,
		32'hf8f000,
		32'h6000};
endmodule

//Ground Sprites Active memory
module ground_active(
input logic [4:0] aa,
output logic [31:0] qa);
		
	assign qa = mem[aa];
		
	logic [31:0] mem[0:31] = '{
		//ground
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'hffffffff,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0,
		32'h0};
endmodule
/*---------------------------*/

/******************************************************************************/


//Code Converters
/******************************************************************************/
module icon_code(
input [4:0]code,
output logic [7:0] R, G, B);


	always_comb begin
		case(code)
			5'h0:
			begin
				R = 8'd195;
				G = 8'd255;
				B = 8'd255;
			end
			5'h1:
			begin
				R = 8'd167;
				G = 8'd167;
				B = 8'd167;
			end
			5'h2:
			begin
				R = 8'd255;
				G = 8'd215;
				B = 8'd177;
			end
			5'h3:
			begin
				R = 8'd122;
				G = 8'd255;
				B = 8'd170;
			end
			5'h4:
			begin
				R = 8'd255;
				G = 8'd0;
				B = 8'd0;
			end
			5'h5:
			begin
				R = 8'd62;
				G = 8'd226;
				B = 8'd136;
			end
			5'h6:
			begin
				R = 8'd20;
				G = 8'd77;
				B = 8'd255;
			end
			5'h7:
			begin
				R = 8'd51;
				G = 8'd226;
				B = 8'd255;
			end
			5'h8:
			begin
				R = 8'd255;
				G = 8'd255;
				B = 8'd255;
			end
			5'h9:
			begin
				R = 8'd255;
				G = 8'd255;
				B = 8'd0;
			end
			5'hA:
			begin
				R = 8'd138;
				G = 8'd87;
				B = 8'd23;
			end
			5'hB:
			begin
				R = 8'd0;
				G = 8'd0;
				B = 8'd0;
			end
			default:
			begin
				R = 8'd0;
				G = 8'd0;
				B = 8'd0;
			end
		endcase
	end

endmodule
	
module color_code(
input [4:0]code,
output logic [7:0] R, G, B);

	always_comb begin
		case(code)
			5'h0:
			begin
				R = 8'd148;
				G = 8'd206;
				B = 8'd0;
			end
			5'h1:
			begin
				R = 8'd49;
				G = 8'd92;
				B = 8'd5;
			end
			5'h2:
			begin
				R = 8'd69;
				G = 8'd36;
				B = 8'd1;
			end
			5'h3:
			begin
				R = 8'd120;
				G = 8'd70;
				B = 8'd14;
			end
			5'h4:
			begin
				R = 8'd195;
				G = 8'd254;
				B = 8'd252;
			end
			5'h5:
			begin
				R = 8'd40;
				G = 8'd40;
				B = 8'd40;
			end
			5'h6:
			begin
				R = 8'd167;
				G = 8'd167;
				B = 8'd167;
			end
			5'h7:
			begin
				R = 8'd255;
				G = 8'd215;
				B = 8'd117;
			end
			5'h8:
			begin
				R = 8'd62;
				G = 8'd226;
				B = 8'd136;
			end
			5'h9:
			begin
				R = 8'd255;
				G = 8'd0;
				B = 8'd0;
			end
			5'hA:
			begin
				R = 8'd20;
				G = 8'd77;
				B = 8'd255;
			end
			5'hB:
			begin
				R = 8'd65;
				G = 8'd226;
				B = 8'd253;
			end
			5'hC:
			begin
				R = 8'd255;
				G = 8'd255;
				B = 8'd255;
			end
			5'hD:
			begin
				R = 8'd255;
				G = 8'd253;
				B = 8'd56;
			end
			5'hE:
			begin
				R = 8'd138;
				G = 8'd87;
				B = 8'd23;
			end
			5'hF:
			begin
				R = 8'd0;
				G = 8'd0;
				B = 8'd0;
			end
			5'h10:
			begin
				R = 8'd165;
				G = 8'd206;
				B = 8'd207;
			end
			5'h11:
			begin
				R = 8'd136;
				G = 8'd156;
				B = 8'd155;
			end
			5'h12:
			begin
				R = 8'd100;
				G = 8'd109;
				B = 8'd104;
			end
			5'h13:
			begin
				R = 8'd61;
				G = 8'd63;
				B = 8'd60;
			end
			default:
			begin
				R = 8'd0;
				G = 8'd0;
				B = 8'd0;
			end
		endcase
	end

endmodule
/******************************************************************************/

module background(
input logic [9:0] cur_x,
input logic [9:0] cur_y,
output logic [7:0] R, G, B);

	logic [4:0] posx;
	logic [12:0] posy;
	
	logic [31:0] col;
	logic [1:0] code;

	
	assign posx[4:1] = cur_x[4:1];
	assign posx[0] = 1'b0;

	
	assign posy = cur_y[9:1]*20 + cur_x[9:5];
	assign code = col[posx +: 2];
	
	mem_back back(posy,col);

	back_code bcode(.*);

endmodule

module back_code(
input [1:0]code,
output logic [7:0] R, G, B);

	always_comb begin
		case(code)
			2'b00:
			begin
				R = 8'h35;
				G = 8'hbc;
				B = 8'hff;
			end
			2'b01:
			begin
				R = 8'ha7;
				G = 8'he3;
				B = 8'hfc;
			end
			2'b10:
			begin
				R = 8'ha4;
				G = 8'he4;
				B = 8'hff;
			end
			2'b11:
			begin
				R = 8'hfb;
				G = 8'hfc;
				B = 8'hf9;
			end
			default:
			begin
				R = 8'd0;
				G = 8'd0;
				B = 8'd0;
			end
		endcase // code
	end
endmodule

module mem_back(
input logic [12:0] aa,
output logic [31:0] qa);
		
	assign qa = mem[aa];
		
	logic [31:0] mem [0:4799] = '{
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aaaa000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haafffa00,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hefffffa0,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffffff8,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffff8,
						32'h2d,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa90000,
						32'hfffffffe,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2bffa000,
						32'hfffffffe,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffff800,
						32'hfffffffe,
						32'hbb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffffe00,
						32'hffffffff,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff00,
						32'hbfffffff,
						32'h7a800abb,
						32'hb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hfbffffff,
						32'heee8adef,
						32'haf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h7fffffc0,
						32'hfffbfffe,
						32'hffbabbff,
						32'h2bb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heeffffc0,
						32'hfffffbb7,
						32'hfffeeeff,
						32'haef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffffa8,
						32'hfffffeef,
						32'hfffffbff,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h82a00000,
						32'hffeffefe,
						32'hffffffbf,
						32'hffffffff,
						32'h2abf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heaff0000,
						32'hffffffff,
						32'hffffffff,
						32'heffffbff,
						32'h2aaefe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aaaa000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbffc000,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hbff7abb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haafffa00,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefff000,
						32'hffffffff,
						32'hffffffbf,
						32'hfffffdff,
						32'hbfffdfaf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hefffffa0,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffff000,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffffff8,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffc00,
						32'hffffbfff,
						32'hffffffff,
						32'hffffffff,
						32'hffffffef,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffff8,
						32'h2d,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffc00,
						32'hffffffff,
						32'hffffffbf,
						32'hfffffdfe,
						32'hffffffff,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa90000,
						32'hfffffffe,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffbff800,
						32'hffffbfff,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffef,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2bffa000,
						32'hfffffffe,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffaff000,
						32'hffffaeff,
						32'hfeffffbf,
						32'hfffffefb,
						32'hbbffffbb,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffff800,
						32'hfffffffe,
						32'hbb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfeabe000,
						32'hfffeab7b,
						32'hffffff7b,
						32'hffffeeee,
						32'haeeefaab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffffe00,
						32'hffffffff,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h7a098000,
						32'heee82aef,
						32'h7b7ffeae,
						32'hffffb9ab,
						32'h2bbba8ae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff00,
						32'hbfffffff,
						32'h7a800abb,
						32'hb,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'hbba002aa,
						32'heffffe2b,
						32'hbbbb98ae,
						32'h2aa802b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hfbffffff,
						32'heee8adef,
						32'haf,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'haa00000a,
						32'hbbbbb802,
						32'hadeaa0a9,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h7fffffc0,
						32'hfffbfffe,
						32'hffbabbff,
						32'h2bb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h6eeea000,
						32'h2aaa000a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heeffffc0,
						32'hfffffbb7,
						32'hfffeeeff,
						32'haef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa9a0000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffffa8,
						32'hfffffeef,
						32'hfffffbff,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h82a00000,
						32'hffeffefe,
						32'hffffffbf,
						32'hffffffff,
						32'h2abf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heaff0000,
						32'hffffffff,
						32'hffffffff,
						32'heffffbff,
						32'h2aaefe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbffc000,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hbff7abb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefff000,
						32'hffffffff,
						32'hffffffbf,
						32'hfffffdff,
						32'hbfffdfaf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffff000,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffc00,
						32'hffffbfff,
						32'hffffffff,
						32'hffffffff,
						32'hffffffef,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffc00,
						32'hffffffff,
						32'hffffffbf,
						32'hfffffdfe,
						32'hffffffff,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffbff800,
						32'hffffbfff,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffef,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffaff000,
						32'hffffaeff,
						32'hfeffffbf,
						32'hfffffefb,
						32'hbbffffbb,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfeabe000,
						32'hfffeab7b,
						32'hffffff7b,
						32'hffffeeee,
						32'haeeefaab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h7a098000,
						32'heee82aef,
						32'h7b7ffeae,
						32'hffffb9ab,
						32'h2bbba8ae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'hbba002aa,
						32'heffffe2b,
						32'hbbbb98ae,
						32'h2aa802b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'haa00000a,
						32'hbbbbb802,
						32'hadeaa0a9,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'h27f,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h6eeea000,
						32'h2aaa000a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'hbff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa9a0000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfff80000,
						32'h2eff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffd0000,
						32'h290bbff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he0000000,
						32'hfffe002b,
						32'h2bf9bbff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfa000000,
						32'hfffe82bf,
						32'hbffeefff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hff800000,
						32'hfbbaebff,
						32'hfffebbff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hff800000,
						32'hfffeafbb,
						32'hfffbafff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'hffff7fff,
						32'hfffebbbf,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heebb9000,
						32'hfffeffff,
						32'hffebaeff,
						32'hab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbdff800,
						32'hffffffff,
						32'hffbfbfff,
						32'h2a82bd,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefffe00,
						32'hfffeffff,
						32'hffffffff,
						32'h2afeaff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffeaa0,
						32'hffffffff,
						32'hffffffff,
						32'hafffbff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffbffa,
						32'hffffffff,
						32'hffffffff,
						32'hbffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe800000,
						32'haf,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hffffffff,
						32'hffffffff,
						32'hffffefff,
						32'h2bffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffa00000,
						32'h2ff,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hfffffffe,
						32'hffffffff,
						32'hfeffffff,
						32'h2affffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'haa2bff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfebbfbaa,
						32'hfffef7ff,
						32'hffffefff,
						32'h1bbffbff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hffaa0009,
						32'hbfebfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8aee900,
						32'hffefbefe,
						32'hfaef7bff,
						32'haefabbf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe800000,
						32'hfeffa02f,
						32'hbfffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha02aa000,
						32'hffbaabbb,
						32'heabbeaff,
						32'ha68aee,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2ba,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'hbbfffaab,
						32'hffffefff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hedea0aaa,
						32'ha0aa52ae,
						32'h29a,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2bff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfff80000,
						32'hffffeffe,
						32'hffebbbfb,
						32'h2a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbba80000,
						32'h802a80ab,
						32'h2a,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'hbfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffebe80,
						32'hffffffff,
						32'hbf7ffaaf,
						32'ha0af,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa800000,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'h2a8affff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffbfff8,
						32'hffffbfff,
						32'hffffffbf,
						32'hbfaff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hea80027e,
						32'hffafffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffffe,
						32'hffffffff,
						32'hffffffef,
						32'h2fffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'hbfe80bff,
						32'hffffffff,
						32'h2f,
						32'h0,
						32'h0,
						32'h0,
						32'hffbbfffe,
						32'hffffffff,
						32'hfbffffff,
						32'h2fffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hfffeaaff,
						32'hfffbffee,
						32'hbf,
						32'h0,
						32'h0,
						32'h0,
						32'hfeab7ff8,
						32'hfffbeeff,
						32'haaffffbb,
						32'hbf6ff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'hfffbffbf,
						32'hfaeefeff,
						32'habf,
						32'h0,
						32'h0,
						32'h0,
						32'haa0aaaa0,
						32'hbfeeabbb,
						32'h80aefaaf,
						32'h2a2a6,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffafa000,
						32'hffffffef,
						32'hdffeabff,
						32'h282bef,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hef6802aa,
						32'haa02a,
						32'h28,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefffe00,
						32'hffefffff,
						32'hffffefff,
						32'h2febfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa800000,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hffffffff,
						32'hfffffbff,
						32'hbffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heeffff80,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haadffe00,
						32'hfefbbfff,
						32'hbfffeeff,
						32'h2fdbfea,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h82aaa800,
						32'hfbaaeeea,
						32'h2bbeabef,
						32'ha8a9a0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hda00aaa0,
						32'h2a80abb,
						32'ha00,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2aa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfa000000,
						32'h2fff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'haa2bfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hba000000,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hfaa0009f,
						32'hbfebffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffa00000,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'heffa02ff,
						32'hffffffff,
						32'hb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfff82900,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'hbfffaabf,
						32'hfffefffb,
						32'h2f,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffebfa0,
						32'h2aa82ab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hff800000,
						32'hfffeffef,
						32'hfebbbfbf,
						32'h2af,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbbc0000,
						32'hfffbffea,
						32'h2aeeeadf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffebe800,
						32'hfffffffb,
						32'hf7ffaaff,
						32'ha0afb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha7ff0000,
						32'hffe6eb9f,
						32'hbbbffb77,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffbfff80,
						32'hfffbffff,
						32'hfffffbff,
						32'hbfafff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hefff8000,
						32'heeffbfff,
						32'hfeeffeef,
						32'h2af,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffffe0,
						32'hffffffff,
						32'hfffffeff,
						32'h2ffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbb7b8000,
						32'hffbffbbf,
						32'hfa7fffff,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbbfffe0,
						32'hffffffff,
						32'hbfffffff,
						32'h2ffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heaee0000,
						32'hffbbfaae,
						32'heaefbbbb,
						32'h2bb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heab7ff80,
						32'hffbeefff,
						32'haffffbbf,
						32'hbf6ffa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha2a80000,
						32'hefaee0aa,
						32'h82bae9ee,
						32'h2a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0aaaa00,
						32'hfeeabbba,
						32'haefaafb,
						32'h2a2a68,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hba29800a,
						32'haa82ab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf6802aa8,
						32'haa02ae,
						32'h280,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha8000000,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha8000000,
						32'haa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aaaa000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aaaa00,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haafffa00,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aafffa0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hefffffa0,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haefffffa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbffffff8,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hbbffffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe0a4000,
						32'h2bbf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffff8,
						32'h2d,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hdfffffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffafe800,
						32'haaa0aaff,
						32'h0,
						32'h0,
						32'h0,
						32'haa90000,
						32'hfffffffe,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he0aa9000,
						32'hbfffffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'hef000000,
						32'hfefffa82,
						32'hbbbab7ff,
						32'ha,
						32'h0,
						32'h0,
						32'h2bffa000,
						32'hfffffffe,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he2bffa00,
						32'hefffffff,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'hffc00000,
						32'hf9bae7e9,
						32'heffeddff,
						32'h2ae,
						32'h0,
						32'h0,
						32'hbffff800,
						32'hfffffffe,
						32'hbb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hebffff80,
						32'hbfffffff,
						32'hb,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'hbfeffffb,
						32'hbbffbbfb,
						32'habff,
						32'h0,
						32'h0,
						32'hbffffe00,
						32'hffffffff,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbffffe0,
						32'hefffffff,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'hdee00000,
						32'heffeefee,
						32'h9fffffff,
						32'h2bffe,
						32'h0,
						32'h0,
						32'hffffff00,
						32'hbfffffff,
						32'h7a800abb,
						32'hb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffff0,
						32'hbbffffff,
						32'hb7a800ab,
						32'h0,
						32'h0,
						32'h0,
						32'hbb800000,
						32'heefeabba,
						32'hbbeeeeff,
						32'haefa,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hfbffffff,
						32'heee8adef,
						32'haf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffff8,
						32'hffbfffff,
						32'hfeee8ade,
						32'ha,
						32'h0,
						32'h0,
						32'haa000000,
						32'hebb82aa8,
						32'haeba7bbb,
						32'haa0,
						32'h0,
						32'h0,
						32'h7fffffc0,
						32'hfffbfffe,
						32'hffbabbff,
						32'h2bb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he7fffffc,
						32'hffffbfff,
						32'hbffbabbf,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h8a600280,
						32'h2aa0aaee,
						32'h0,
						32'h0,
						32'h0,
						32'heeffffc0,
						32'hfffffbb7,
						32'hfffeeeff,
						32'haef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h7eeffffc,
						32'hffffffbb,
						32'hffffeeef,
						32'hae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aa,
						32'h0,
						32'h0,
						32'h0,
						32'hffffffa8,
						32'hfffffeef,
						32'hfffffbff,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hfffffffa,
						32'hffffffee,
						32'hffffffbf,
						32'haf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h82a00000,
						32'hffeffefe,
						32'hffffffbf,
						32'hffffffff,
						32'h2abf,
						32'h0,
						32'h0,
						32'h0,
						32'he82a0000,
						32'hfffeffef,
						32'hfffffffb,
						32'hffffffff,
						32'h2ab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heaff0000,
						32'hffffffff,
						32'hffffffff,
						32'heffffbff,
						32'h2aaefe,
						32'h0,
						32'h0,
						32'h0,
						32'hfeaff000,
						32'hffffffff,
						32'hffffffff,
						32'heeffffbf,
						32'h2aaef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbffc000,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hbff7abb,
						32'h0,
						32'h0,
						32'h0,
						32'hffbffc00,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffff,
						32'hbff7ab,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefff000,
						32'hffffffff,
						32'hffffffbf,
						32'hfffffdff,
						32'hbfffdfaf,
						32'h0,
						32'h0,
						32'h0,
						32'hffefff00,
						32'hffffffff,
						32'hfffffffb,
						32'hffffffdf,
						32'hbfffdfa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffff000,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffff,
						32'h2,
						32'h0,
						32'h0,
						32'hffffff00,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'h2bffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffc00,
						32'hffffbfff,
						32'hffffffff,
						32'hffffffff,
						32'hffffffef,
						32'h2,
						32'h0,
						32'h0,
						32'hffffffc0,
						32'hfffffbff,
						32'hffffffff,
						32'hffffffff,
						32'h2ffffffe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffc00,
						32'hffffffff,
						32'hffffffbf,
						32'hfffffdfe,
						32'hffffffff,
						32'ha,
						32'h0,
						32'h0,
						32'hffffffc0,
						32'hffffffff,
						32'heffffffb,
						32'hffffffdf,
						32'hafffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffbff800,
						32'hffffbfff,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffef,
						32'ha,
						32'h0,
						32'h0,
						32'hfffbff80,
						32'hfffffbff,
						32'hffffffff,
						32'hffffffff,
						32'habfffffe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffaff000,
						32'hffffaeff,
						32'hfeffffbf,
						32'hfffffefb,
						32'hbbffffbb,
						32'h2,
						32'h0,
						32'h0,
						32'hfffaff00,
						32'hfffffaef,
						32'hbfeffffb,
						32'hbfffffef,
						32'h2bbffffb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfeabe000,
						32'hfffeab7b,
						32'hffffff7b,
						32'hffffeeee,
						32'haeeefaab,
						32'h0,
						32'h0,
						32'h0,
						32'hbfeabe00,
						32'hbfffeab7,
						32'heffffff7,
						32'hbffffeee,
						32'haeeefaa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h7a098000,
						32'heee82aef,
						32'h7b7ffeae,
						32'hffffb9ab,
						32'h2bbba8ae,
						32'h0,
						32'h0,
						32'h0,
						32'hf7a09800,
						32'heeee82ae,
						32'hb7b7ffea,
						32'heffffb9a,
						32'h2bbba8a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'hbba002aa,
						32'heffffe2b,
						32'hbbbb98ae,
						32'h2aa802b,
						32'h0,
						32'h0,
						32'h0,
						32'hae800000,
						32'hbbba002a,
						32'heeffffe2,
						32'hbbbbb98a,
						32'h2aa802,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'haa00000a,
						32'hbbbbb802,
						32'hadeaa0a9,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'ha8000000,
						32'h2aa00000,
						32'h9bbbbb80,
						32'haadeaa0a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h6eeea000,
						32'h2aaa000a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha6eeea00,
						32'h2aaa000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa9a0000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa9a000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hff800000,
						32'h9f,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfff80000,
						32'h2ff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffe0000,
						32'hbbf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffff4000,
						32'ha42eff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hffff800a,
						32'hafe6eff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe800000,
						32'hffffa0af,
						32'h2fffbbff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'hfeeebaff,
						32'hbfffaeff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffe00000,
						32'hffffabee,
						32'hbffeebff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2ba0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfff80000,
						32'hffffdfff,
						32'hbfffaeef,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h27ffe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfbaee400,
						32'hffffbfff,
						32'hfffaebbf,
						32'h2a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he0000000,
						32'hbffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfef7fe00,
						32'hffffffff,
						32'h7fefefff,
						32'haa0af,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'h2effff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffbfff80,
						32'hffffbfff,
						32'hffffffff,
						32'habfabf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfd000000,
						32'h90bbffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffaa8,
						32'hffffffff,
						32'hffffffff,
						32'h2bffeff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe002be0,
						32'hf9bbffff,
						32'h2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hffffeffe,
						32'hffffffff,
						32'hffffffff,
						32'h2ffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe82bffa,
						32'hfeefffff,
						32'hbf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he0000000,
						32'hffffffff,
						32'hffffffff,
						32'hfffffbff,
						32'haffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hbaebffff,
						32'hfebbfffb,
						32'h2ff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'hffffffff,
						32'hffffffff,
						32'hffbfffff,
						32'habfffff,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hfeafbbff,
						32'hfbafffff,
						32'h2ff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hffaefeea,
						32'hffffbdff,
						32'hfffffbff,
						32'h6effeff,
						32'h0,
						32'h0,
						32'h0,
						32'he0000000,
						32'hff7fffff,
						32'hfebbbfff,
						32'haff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hae80000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbe2bba40,
						32'hfffbefbf,
						32'hfebbdeff,
						32'h2bbeaef,
						32'h0,
						32'h0,
						32'h0,
						32'hbb900000,
						32'hfeffffee,
						32'hebaeffff,
						32'habff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haffe8000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he80aa800,
						32'hffeeaaee,
						32'hbaaefabf,
						32'h29a2bb,
						32'h0,
						32'h0,
						32'h0,
						32'hdff80000,
						32'hfffffffb,
						32'hbfbfffff,
						32'h2a82bdff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffa000,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hbb7a82aa,
						32'ha82a94ab,
						32'ha6,
						32'h0,
						32'h0,
						32'h0,
						32'hfffe0000,
						32'hfefffffe,
						32'hffffffff,
						32'hafeaffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'hffffe000,
						32'haa2b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heeea0000,
						32'ha00aa02a,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'hffeaa000,
						32'hffffffff,
						32'hffffffff,
						32'hfffbffff,
						32'ha,
						32'h0,
						32'h0,
						32'h9f80000,
						32'hffffaa00,
						32'hbfebf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haaa00000,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffbffa00,
						32'hffffffff,
						32'hffffffff,
						32'hffffffff,
						32'hb,
						32'h0,
						32'h0,
						32'h2ffe8000,
						32'hfffeffa0,
						32'hbfffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hffffffff,
						32'hffefffff,
						32'hffffffff,
						32'h2b,
						32'h0,
						32'h0,
						32'habffe000,
						32'hffbbfffa,
						32'h2ffffef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfffffe80,
						32'hffffffff,
						32'hffffffff,
						32'hfffffffe,
						32'h2a,
						32'h0,
						32'h0,
						32'hfefff800,
						32'hfbffffef,
						32'h2affebbb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbbfbaa00,
						32'hfef7fffe,
						32'hffefffff,
						32'hbffbffff,
						32'h1b,
						32'h0,
						32'h80000000,
						32'hffbffebe,
						32'hafffffff,
						32'hafbf7ffa,
						32'ha0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haee90000,
						32'hefbefef8,
						32'hef7bffff,
						32'hefabbffa,
						32'ha,
						32'h0,
						32'hf8000000,
						32'hfffffbff,
						32'hbfffffbf,
						32'hffffffff,
						32'hbfa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2aa00000,
						32'hbaabbba0,
						32'hbbeaffff,
						32'ha68aeeea,
						32'h0,
						32'h0,
						32'hfe000000,
						32'hffffffff,
						32'hefffffff,
						32'hffffffff,
						32'h2fff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hea0aaa00,
						32'haa52aeed,
						32'h29aa0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'hffffbbff,
						32'hffffffff,
						32'hfffbffff,
						32'h2fff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha8000000,
						32'h2a80abbb,
						32'h2a80,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hfffeab7f,
						32'hbbfffbee,
						32'hffaaffff,
						32'hbf6,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'haaa,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'hbbaa0aaa,
						32'hafbfeeab,
						32'ha680aefa,
						32'h2a2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa800000,
						32'h2aef6802,
						32'h28000aa0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haaa8000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2ba,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2ba,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2bff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2bff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'hbfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he8000000,
						32'hbfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'h2a8affff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'h2a8affff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hea80027e,
						32'hffafffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hea80027e,
						32'hffafffff,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'hbfe80bff,
						32'hffffffff,
						32'h2f,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'hbfe80bff,
						32'hffffffff,
						32'h2f,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hfffeaaff,
						32'hfffbffee,
						32'hbf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hfffeaaff,
						32'hfffbffee,
						32'hbf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'hfffbffbf,
						32'hfaeefeff,
						32'habf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfe000000,
						32'hfffbffbf,
						32'hfaeefeff,
						32'habf,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffafa000,
						32'hffffffef,
						32'hdffeabff,
						32'h282bef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffafa000,
						32'hffffffef,
						32'hdffeabff,
						32'h282bef,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefffe00,
						32'hffefffff,
						32'hffffefff,
						32'h2febfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hfefffe00,
						32'hffefffff,
						32'hffffefff,
						32'h2febfff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hffffffff,
						32'hfffffbff,
						32'hbffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffffff80,
						32'hffffffff,
						32'hfffffbff,
						32'hbffffff,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'heeffff80,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffe,
						32'h0,
						32'h2ba0000,
						32'h0,
						32'h0,
						32'heeffff80,
						32'hffffffff,
						32'hffffffff,
						32'hbfffffe,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haadffe00,
						32'hfefbbfff,
						32'hbfffeeff,
						32'h2fdbfea,
						32'h0,
						32'h2bffa000,
						32'h0,
						32'h0,
						32'haadffe00,
						32'hfefbbfff,
						32'hbfffeeff,
						32'h2fdbfea,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2ba0,
						32'h0,
						32'h0,
						32'h82aaa800,
						32'hfbaaeeea,
						32'h2bbeabef,
						32'ha8a9a0,
						32'h0,
						32'haefff829,
						32'h0,
						32'h0,
						32'h82aaa800,
						32'hfbaaeeea,
						32'h2bbeabef,
						32'ha8a9a0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h2bffa,
						32'h0,
						32'h0,
						32'h0,
						32'hda00aaa0,
						32'h2a80abb,
						32'ha00,
						32'ha0000000,
						32'habfffebf,
						32'h2aa82,
						32'h0,
						32'h0,
						32'hda00aaa0,
						32'h2a80abb,
						32'ha00,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h82900000,
						32'haefff,
						32'h0,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2aa,
						32'h0,
						32'hea0bbc00,
						32'hdffffbff,
						32'h2aeeea,
						32'h0,
						32'h0,
						32'ha0000000,
						32'h2aa,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hebfa0000,
						32'ha82abfff,
						32'h2a,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h9fa7ff00,
						32'h77ffe6eb,
						32'habbbffb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hc0000000,
						32'hbffea0bb,
						32'heeadffff,
						32'h2ae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hffefff80,
						32'hefeeffbf,
						32'haffeeffe,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf0000000,
						32'h6eb9fa7f,
						32'hffb77ffe,
						32'habbb,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hbfbb7b80,
						32'hffffbffb,
						32'hfffa7fff,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hf8000000,
						32'hfbfffeff,
						32'hffeefeef,
						32'h2affee,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haeeaee00,
						32'hbbffbbfa,
						32'hbbeaefbb,
						32'h2,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'hb8000000,
						32'hffbbfbb7,
						32'hfffffffb,
						32'hafffa7,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haaa2a800,
						32'heeefaee0,
						32'h2a82bae9,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'he0000000,
						32'hbfaaeeae,
						32'hfbbbbffb,
						32'h2bbeae,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'ha000000,
						32'habba2980,
						32'haa82,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h80000000,
						32'hee0aaa2a,
						32'hae9eeefa,
						32'h2a82b,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa80000,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h9800a000,
						32'ha82abba2,
						32'ha,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'haa80,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0,
						32'h0};						
endmodule // mem_back

