// Primary Developer for CU_Sprites.sv: Blayne Kettlewell
// Date: 5/11/2016

module CU_SPRITES_MOD(
	input logic reset, clk65, clk260, audio_clk, write, chipSelect,
	input logic [31:0] writeData,
	input logic [17:0] address,
	output logic [7:0] vgaR, vgaG, vgaB,
	output logic vgaClk, vgaHS, vgaVS, vgaBlank_n, vgaSync_n,
	inout  wire AUD_ADCLRCK,
   input  wire AUD_ADCDAT,
   inout  wire AUD_DACLRCK,
   output logic AUD_DACDAT,
   output logic AUD_XCK,
   inout  wire AUD_BCLK,
   output logic AUD_I2C_SCLK,
   inout  wire AUD_I2C_SDAT,
   output logic AUD_MUTE
);

// Wire and register declarations
typedef enum logic [3:0] {P_IDLE, P_SAMPLE_PATTERN_INDEX, P_READ_PATTERN_DATA, R_SAMPLE_PATTERN_DATA} p_state_t;

p_state_t currState, nextState;

logic [31:0] writeDataSync;

logic localWriteEnable;

logic [4:0] pixelXOffsetSync, pixelYOffsetSync, pixelXOffsetSyncTwo, pixelYOffsetSyncTwo = 5'd0;
logic [7:0] nameColumnOffsetSync, nameColumnOffsetSyncTwo = 8'd0;
logic [9:0] nameRowOffsetSync, nameRowOffsetSyncTwo = 10'd0;

logic [31:0] redLineTwoCurrSync, redLineOneCurrSync, redLineZeroCurrSync;
logic [31:0] greenLineTwoCurrSync, greenLineOneCurrSync, greenLineZeroCurrSync;
logic [31:0] blueLineTwoCurrSync, blueLineOneCurrSync, blueLineZeroCurrSync;

logic [31:0] redLineTwoQ, redLineOneQ, redLineZeroQ;
logic [31:0] greenLineTwoQ, greenLineOneQ, greenLineZeroQ;
logic [31:0] blueLineTwoQ, blueLineOneQ, blueLineZeroQ;

logic [31:0] redLineTwoNextSync, redLineOneNextSync, redLineZeroNextSync;
logic [31:0] greenLineTwoNextSync, greenLineOneNextSync, greenLineZeroNextSync;
logic [31:0] blueLineTwoNextSync, blueLineOneNextSync, blueLineZeroNextSync;

logic readNameStrobe;
logic sampleIndexStrobe, sampleIndexStrobeSync;
logic readPatternStrobe;
logic samplePatternData, samplePatternDataSync;
logic updateOffsetStrobe;
logic updateLineBuffer;
logic nextRow, nextColumn;

logic [10:0] hCount;
logic [9:0] vCount;
logic hActiveWindow, vActiveWindow;
logic newFrameSync;
logic [4:0] lineIndex;
logic [4:0] pixelIndex;
logic [4:0] displayColumnSync, displayRowSync;
logic [7:0] patternIndexQ, patternIndexQSync;
logic nextColUpdateZero, nextColUpdateOne, resetColToZero;
logic [15:0] patternIndexMult;
logic [12:0] patternIndex;
logic endOfRows;
logic endOfVerticalPatterns;

logic [7:0] nameRAMDataSync;
logic [15:0] rdNameRAMAddrSync, wrNameRAMAddrSync;
logic rdNameRAMStrobeSync, wrNameRAMStrobeSync;

logic [12:0] rdRedTwoRAMAddrSync, rdRedOneRAMAddrSync, rdRedZeroRAMAddrSync;
logic [12:0] rdGreenTwoRAMAddrSync, rdGreenOneRAMAddrSync, rdGreenZeroRAMAddrSync;
logic [12:0] rdBlueTwoRAMAddrSync, rdBlueOneRAMAddrSync, rdBlueZeroRAMAddrSync;

logic rdRedTwoRAMStrobeSync, rdRedOneRAMStrobeSync, rdRedZeroRAMStrobeSync;
logic rdGreenTwoRAMStrobeSync, rdGreenOneRAMStrobeSync, rdGreenZeroRAMStrobeSync;
logic rdBlueTwoRAMStrobeSync, rdBlueOneRAMStrobeSync, rdBlueZeroRAMStrobeSync;

logic [12:0] wrRedTwoRAMAddrSync, wrRedOneRAMAddrSync, wrRedZeroRAMAddrSync;
logic [12:0] wrGreenTwoRAMAddrSync, wrGreenOneRAMAddrSync, wrGreenZeroRAMAddrSync;
logic [12:0] wrBlueTwoRAMAddrSync, wrBlueOneRAMAddrSync, wrBlueZeroRAMAddrSync;

logic wrRedTwoRAMStrobeSync, wrRedOneRAMStrobeSync, wrRedZeroRAMStrobeSync;
logic wrGreenTwoRAMStrobeSync, wrGreenOneRAMStrobeSync, wrGreenZeroRAMStrobeSync;
logic wrBlueTwoRAMStrobeSync, wrBlueOneRAMStrobeSync, wrBlueZeroRAMStrobeSync;

logic [31:0] redTwoRAMDataSync, redOneRAMDataSync, redZeroRAMDataSync;
logic [31:0] greenTwoRAMDataSync, greenOneRAMDataSync, greenZeroRAMDataSync;
logic [31:0] blueTwoRAMDataSync, blueOneRAMDataSync, blueZeroRAMDataSync;

logic [31:0] redTwoSpriteColZeroSync, redTwoSpriteColOneSync, redTwoSpriteColTwoSync, redTwoSpriteColThreeSync;
logic [31:0] redOneSpriteColZeroSync, redOneSpriteColOneSync, redOneSpriteColTwoSync, redOneSpriteColThreeSync;
logic [31:0] redZeroSpriteColZeroSync, redZeroSpriteColOneSync, redZeroSpriteColTwoSync, redZeroSpriteColThreeSync;

logic [127:0] wrRedTwoSpriteDataSync, wrRedOneSpriteDataSync, wrRedZeroSpriteDataSync;
logic [127:0] redTwoSpriteZeroLineQ, redOneSpriteZeroLineQ, redZeroSpriteZeroLineQ;

logic [31:0] greenTwoSpriteColZeroSync, greenTwoSpriteColOneSync, greenTwoSpriteColTwoSync, greenTwoSpriteColThreeSync;
logic [31:0] greenOneSpriteColZeroSync, greenOneSpriteColOneSync, greenOneSpriteColTwoSync, greenOneSpriteColThreeSync;
logic [31:0] greenZeroSpriteColZeroSync, greenZeroSpriteColOneSync, greenZeroSpriteColTwoSync, greenZeroSpriteColThreeSync;

logic [127:0] wrGreenTwoSpriteDataSync, wrGreenOneSpriteDataSync, wrGreenZeroSpriteDataSync;
logic [127:0] greenTwoSpriteZeroLineQ, greenOneSpriteZeroLineQ, greenZeroSpriteZeroLineQ;

logic [31:0] blueTwoSpriteColZeroSync, blueTwoSpriteColOneSync, blueTwoSpriteColTwoSync, blueTwoSpriteColThreeSync;
logic [31:0] blueOneSpriteColZeroSync, blueOneSpriteColOneSync, blueOneSpriteColTwoSync, blueOneSpriteColThreeSync;
logic [31:0] blueZeroSpriteColZeroSync, blueZeroSpriteColOneSync, blueZeroSpriteColTwoSync, blueZeroSpriteColThreeSync;

logic [127:0] wrBlueTwoSpriteDataSync, wrBlueOneSpriteDataSync, wrBlueZeroSpriteDataSync;
logic [127:0] blueTwoSpriteZeroLineQ, blueOneSpriteZeroLineQ, blueZeroSpriteZeroLineQ;

logic [6:0] wrSpriteZeroRAMAddrSync, rdSpriteZeroRAMAddrSync;
logic [31:0] spriteConfigData;
logic [9:0] spriteZeroPixelXOffsetSync, spriteZeroPixelYOffsetSync = 10'd0;
logic [9:0] spriteZeroPixelXOffsetSyncTwo, spriteZeroPixelYOffsetSyncTwo;
logic [6:0] spriteZeroXPixel, spriteZeroXPixelSyncTwo, spriteZeroYPixel;
logic spriteZeroInXBoundary, spriteZeroInYBoundary;
logic spriteZeroVisible, spriteZeroInBoundary, spriteZeroTransparentPixel,  spriteZeroActive;
logic wrSpriteZeroRAMSync, rdSpriteZeroRAMSync;
logic [6:0] spritePixelX, spritePixelY;
shortint translatedXOrigin, translatedYOrigin;
shortint vecXNormalized, vecYNormalized;

logic [7:0] redRawPatternChan, greenRawPatternChan, blueRawPatternChan;
logic [7:0] redRawSpriteZeroChan, greenRawSpriteZeroChan, blueRawSpriteZeroChan;
logic [7:0] redRawChan, greenRawChan, blueRawChan, redRawOrBlankChan, greenRawOrBlankChan, blueRawOrBlankChan;

logic blankScreenSync;

logic redTwoActive, redOneActive, redZeroActive;
logic greenTwoActive, greenOneActive, greenZeroActive;
logic blueTwoActive, blueOneActive, blueZeroActive;
logic nameTableActive, nameTableMaxBound, nameTableMinBound;
logic nameOffsetXActive, nameOffsetYActive;
logic pixelOffsetXActive, pixelOffsetYActive;

logic vgaControlActive;

logic spriteRedTwoColZeroActive, spriteRedTwoColOneActive,  spriteRedTwoColTwoActive,  spriteRedTwoColThreeActive; 
logic spriteRedOneColZeroActive, spriteRedOneColOneActive,  spriteRedOneColTwoActive,  spriteRedOneColThreeActive; 
logic spriteRedZeroColZeroActive, spriteRedZeroColOneActive,  spriteRedZeroColTwoActive,  spriteRedZeroColThreeActive; 

logic spriteGreenTwoColZeroActive, spriteGreenTwoColOneActive,  spriteGreenTwoColTwoActive,  spriteGreenTwoColThreeActive; 
logic spriteGreenOneColZeroActive, spriteGreenOneColOneActive,  spriteGreenOneColTwoActive,  spriteGreenOneColThreeActive; 
logic spriteGreenZeroColZeroActive, spriteGreenZeroColOneActive,  spriteGreenZeroColTwoActive,  spriteGreenZeroColThreeActive; 

logic spriteBlueTwoColZeroActive, spriteBlueTwoColOneActive,  spriteBlueTwoColTwoActive,  spriteBlueTwoColThreeActive; 
logic spriteBlueOneColZeroActive, spriteBlueOneColOneActive,  spriteBlueOneColTwoActive,  spriteBlueOneColThreeActive; 
logic spriteBlueZeroColZeroActive, spriteBlueZeroColOneActive,  spriteBlueZeroColTwoActive,  spriteBlueZeroColThreeActive; 

logic spriteConfigActive;
logic spriteZeroPixelXOffsetActive;
logic spriteZeroPixelYOffsetActive;
logic spriteZeroOriginXTransActive;
logic spriteZeroOriginYTransActive;
logic spriteZeroRotateXVecActive;
logic spriteZeroRotateYVecActive;
logic spriteZeroAttributeActive;
logic soundControlActive;

parameter RED_TWO_START    = 20'h00000,
			 RED_ONE_START    = 20'h02000,
			 RED_ZERO_START   = 20'h04000,
			 GREEN_TWO_START  = 20'h06000,
			 GREEN_ONE_START  = 20'h08000,
			 GREEN_ZERO_START = 20'h0A000,
			 BLUE_TWO_START   = 20'h0C000,
			 BLUE_ONE_START   = 20'h0E000,
			 BLUE_ZERO_START  = 20'h10000,
			 NAME_TABLE_START = 20'h12000;

assign redTwoActive    = address[17:13] == 5'b00000;
assign redOneActive    = address[17:13] == 5'b00001;
assign redZeroActive   = address[17:13] == 5'b00010;
assign greenTwoActive  = address[17:13] == 5'b00011;
assign greenOneActive  = address[17:13] == 5'b00100;
assign greenZeroActive = address[17:13] == 5'b00101;
assign blueTwoActive   = address[17:13] == 5'b00110;
assign blueOneActive   = address[17:13] == 5'b00111;
assign blueZeroActive  = address[17:13] == 5'b01000;
assign nameTableMinBound = (address[17:16] == 2'b01) & (address[15:12] != 4'b0001) & (address[15:12] != 4'b0000);
assign nameTableMaxBound = (address[17:16] == 2'b10 & address[15:13] == 3'b000);
assign nameTableActive    = nameTableMinBound | nameTableMaxBound;
assign nameOffsetXActive  = address == 18'b10_0010_0000_0000_0000;
assign nameOffsetYActive  = address == 18'b10_0010_0000_0000_0001;
assign pixelOffsetXActive = address == 18'b10_0010_0000_0000_0010;
assign pixelOffsetYActive = address == 18'b10_0010_0000_0000_0011;

assign vgaControlActive             = address == 18'b10_0010_0000_0000_0100;

assign spriteRedTwoColZeroActive    = address == 18'b10_0010_0001_0000_0000;
assign spriteRedTwoColOneActive     = address == 18'b10_0010_0001_0000_0001;
assign spriteRedTwoColTwoActive     = address == 18'b10_0010_0001_0000_0010;
assign spriteRedTwoColThreeActive   = address == 18'b10_0010_0001_0000_0011;
assign spriteRedOneColZeroActive    = address == 18'b10_0010_0001_0000_0100;
assign spriteRedOneColOneActive     = address == 18'b10_0010_0001_0000_0101;
assign spriteRedOneColTwoActive     = address == 18'b10_0010_0001_0000_0110;
assign spriteRedOneColThreeActive   = address == 18'b10_0010_0001_0000_0111;
assign spriteRedZeroColZeroActive   = address == 18'b10_0010_0001_0000_1000;
assign spriteRedZeroColOneActive    = address == 18'b10_0010_0001_0000_1001;
assign spriteRedZeroColTwoActive    = address == 18'b10_0010_0001_0000_1010;
assign spriteRedZeroColThreeActive  = address == 18'b10_0010_0001_0000_1011;

assign spriteGreenTwoColZeroActive    = address == 18'b10_0010_0001_0000_1100;
assign spriteGreenTwoColOneActive     = address == 18'b10_0010_0001_0000_1101;
assign spriteGreenTwoColTwoActive     = address == 18'b10_0010_0001_0000_1110;
assign spriteGreenTwoColThreeActive   = address == 18'b10_0010_0001_0000_1111;
assign spriteGreenOneColZeroActive    = address == 18'b10_0010_0001_0001_0000;
assign spriteGreenOneColOneActive     = address == 18'b10_0010_0001_0001_0001;
assign spriteGreenOneColTwoActive     = address == 18'b10_0010_0001_0001_0010;
assign spriteGreenOneColThreeActive   = address == 18'b10_0010_0001_0001_0011;
assign spriteGreenZeroColZeroActive   = address == 18'b10_0010_0001_0001_0100;
assign spriteGreenZeroColOneActive    = address == 18'b10_0010_0001_0001_0101;
assign spriteGreenZeroColTwoActive    = address == 18'b10_0010_0001_0001_0110;
assign spriteGreenZeroColThreeActive  = address == 18'b10_0010_0001_0001_0111;

assign spriteBlueTwoColZeroActive    = address == 18'b10_0010_0001_0001_1000;
assign spriteBlueTwoColOneActive     = address == 18'b10_0010_0001_0001_1001;
assign spriteBlueTwoColTwoActive     = address == 18'b10_0010_0001_0001_1010;
assign spriteBlueTwoColThreeActive   = address == 18'b10_0010_0001_0001_1011;
assign spriteBlueOneColZeroActive    = address == 18'b10_0010_0001_0001_1100;
assign spriteBlueOneColOneActive     = address == 18'b10_0010_0001_0001_1101;
assign spriteBlueOneColTwoActive     = address == 18'b10_0010_0001_0001_1110;
assign spriteBlueOneColThreeActive   = address == 18'b10_0010_0001_0001_1111;
assign spriteBlueZeroColZeroActive   = address == 18'b10_0010_0001_0010_0000;
assign spriteBlueZeroColOneActive    = address == 18'b10_0010_0001_0010_0001;
assign spriteBlueZeroColTwoActive    = address == 18'b10_0010_0001_0010_0010;
assign spriteBlueZeroColThreeActive  = address == 18'b10_0010_0001_0010_0011;

assign spriteConfigActive            = address == 18'b10_0010_0001_0010_0100;

assign spriteZeroPixelXOffsetActive  = address == 18'b10_0010_0001_0010_0101;
assign spriteZeroPixelYOffsetActive  = address == 18'b10_0010_0001_0010_0110;
assign spriteZeroOriginXTransActive  = address == 18'b10_0010_0001_0010_0111;
assign spriteZeroOriginYTransActive  = address == 18'b10_0010_0001_0010_1000;
assign spriteZeroRotateXVecActive    = address == 18'b10_0010_0001_0010_1001;
assign spriteZeroRotateYVecActive    = address == 18'b10_0010_0001_0010_1010;
assign spriteZeroAttributeActive     = address == 18'b10_0010_0001_0010_1011;

assign soundControlActive            = address == 18'b10_0010_0001_0010_1100;

// Sound signals
logic [1:0] sample_end;
logic [1:0] sample_req;
logic [15:0] audio_output;
logic [15:0] audio_input;

//included for sound
logic [15:0] M_sound;
logic [14:0] addr_sound;
logic [3:0]  soundControl;

//The sound portion of this code is a modification of Tutorial of Howard Mao zhehaomao.com/blog/fpga/2014/01/15/sockit-8.html
//Sound Implementation Modified By:Chandan Kanungo, integrated to CU_Sprites by Blayne Kettlewell

// Instantiate the XVGA emulator
VGA_SPRITE_EMULATOR vgaEmulator(.*);

// Instatiate all the Megafunction RAM

//storing sound in memory
sound soundRamZero (.clock(clk65), .address(addr_sound), .q(M_sound));

i2c_av_config av_config (
    .clk (clk65),
    .reset (reset),
    .i2c_sclk (AUD_I2C_SCLK),
    .i2c_sdat (AUD_I2C_SDAT)
);

audio_codec ac (
    .clk (audio_clk),
    .reset (reset),
    .sample_end (sample_end),
    .sample_req (sample_req),
    .audio_output (audio_output),
    .audio_input (audio_input),
    .channel_sel (2'b10),

    .AUD_ADCLRCK (AUD_ADCLRCK),
    .AUD_ADCDAT (AUD_ADCDAT),
    .AUD_DACLRCK (AUD_DACLRCK),
    .AUD_DACDAT (AUD_DACDAT),
    .AUD_BCLK (AUD_BCLK)
);

audio_effects ae (
    .clk (audio_clk),
    .sample_end (sample_end[1]),
    .sample_req (sample_req[1]),
    .audio_output (audio_output),
    .audio_input  (audio_input),
    .control (soundControl),
	 .addr_sound(addr_sound),
	 .M_sound(M_sound)
);


// SPRITES
// Red Color Channel RAMs
RAM_DUAL_U128 spriteZeroRedTwoRAM( .clock(clk65),
													.data(wrRedTwoSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(redTwoSpriteZeroLineQ));

RAM_DUAL_U128 spriteZeroRedOneRAM( .clock(clk65),
													.data(wrRedOneSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(redOneSpriteZeroLineQ));

RAM_DUAL_U128 spriteZeroRedZeroRAM( .clock(clk65),
													.data(wrRedZeroSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(redZeroSpriteZeroLineQ));
// Green Color Channel RAMs
RAM_DUAL_U128 spriteZeroGreenTwoRAM( .clock(clk65),
													.data(wrGreenTwoSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(greenTwoSpriteZeroLineQ));

RAM_DUAL_U128 spriteZeroGreenOneRAM( .clock(clk65),
													.data(wrGreenOneSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(greenOneSpriteZeroLineQ));

RAM_DUAL_U128 spriteZeroGreenZeroRAM( .clock(clk65),
													.data(wrGreenZeroSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(greenZeroSpriteZeroLineQ));

// Blue Color Channel RAMs
RAM_DUAL_U128 spriteZeroBlueTwoRAM( .clock(clk65),
													.data(wrBlueTwoSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(blueTwoSpriteZeroLineQ));

RAM_DUAL_U128 spriteZeroBlueOneRAM( .clock(clk65),
													.data(wrBlueOneSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(blueOneSpriteZeroLineQ));

RAM_DUAL_U128 spriteZeroBlueZeroRAM( .clock(clk65),
													.data(wrBlueZeroSpriteDataSync),
													.rdaddress(rdSpriteZeroRAMAddrSync),
													.rden(rdSpriteZeroRAMSync),
													.wraddress(wrSpriteZeroRAMAddrSync),
													.wren(wrSpriteZeroRAMSync),
													.q(blueZeroSpriteZeroLineQ));


// PATTERNS
// Red Color Channel RAMs
RAM_DUAL_U32 patternRedTwoRAM( .clock(clk65),
													.data(redTwoRAMDataSync),
													.rdaddress(rdRedTwoRAMAddrSync),
													.rden(rdRedTwoRAMStrobeSync),
													.wraddress(wrRedTwoRAMAddrSync),
													.wren(wrRedTwoRAMStrobeSync),
													.q(redLineTwoQ));

RAM_DUAL_U32 patternRedOneRAM(	.clock(clk65),
												 	.data(redOneRAMDataSync),
												 	.rdaddress(rdRedOneRAMAddrSync),
												 	.rden(rdRedOneRAMStrobeSync),
												 	.wraddress(wrRedOneRAMAddrSync),
													.wren(wrRedOneRAMStrobeSync),
													.q(redLineOneQ));

RAM_DUAL_U32 patternRedZeroRAM(.clock(clk65),
												 	.data(redZeroRAMDataSync),
												 	.rdaddress(rdRedZeroRAMAddrSync),
												 	.rden(rdRedZeroRAMStrobeSync),
												 	.wraddress(wrRedZeroRAMAddrSync),
													.wren(wrRedZeroRAMStrobeSync),
													.q(redLineZeroQ));

// Green Color Channel RAMs
RAM_DUAL_U32 patternGreenTwoRAM( .clock(clk65),
													.data(greenTwoRAMDataSync),
													.rdaddress(rdGreenTwoRAMAddrSync),
													.rden(rdGreenTwoRAMStrobeSync),
													.wraddress(wrGreenTwoRAMAddrSync),
													.wren(wrGreenTwoRAMStrobeSync),
													.q(greenLineTwoQ));

RAM_DUAL_U32 patternGreenOneRAM(	.clock(clk65),
												 	.data(greenOneRAMDataSync),
												 	.rdaddress(rdGreenOneRAMAddrSync),
												 	.rden(rdGreenOneRAMStrobeSync),
												 	.wraddress(wrGreenOneRAMAddrSync),
													.wren(wrGreenOneRAMStrobeSync),
													.q(greenLineOneQ));

RAM_DUAL_U32 patternGreenZeroRAM(.clock(clk65),
												 	.data(greenZeroRAMDataSync),
												 	.rdaddress(rdGreenZeroRAMAddrSync),
												 	.rden(rdGreenZeroRAMStrobeSync),
												 	.wraddress(wrGreenZeroRAMAddrSync),
													.wren(wrGreenZeroRAMStrobeSync),
													.q(greenLineZeroQ));

// Blue Color Channel RAMs
RAM_DUAL_U32 patternBlueTwoRAM( .clock(clk65),
													.data(blueTwoRAMDataSync),
													.rdaddress(rdBlueTwoRAMAddrSync),
													.rden(rdBlueTwoRAMStrobeSync),
													.wraddress(wrBlueTwoRAMAddrSync),
													.wren(wrBlueTwoRAMStrobeSync),
													.q(blueLineTwoQ));

RAM_DUAL_U32 patternBlueOneRAM(.clock(clk65),
												 	.data(blueOneRAMDataSync),
												 	.rdaddress(rdBlueOneRAMAddrSync),
												 	.rden(rdBlueOneRAMStrobeSync),
												 	.wraddress(wrBlueOneRAMAddrSync),
													.wren(wrBlueOneRAMStrobeSync),
													.q(blueLineOneQ));

RAM_DUAL_U32 patternBlueZeroRAM(.clock(clk65),
												 	.data(blueZeroRAMDataSync),
												 	.rdaddress(rdBlueZeroRAMAddrSync),
												 	.rden(rdBlueZeroRAMStrobeSync),
												 	.wraddress(wrBlueZeroRAMAddrSync),
													.wren(wrBlueZeroRAMStrobeSync),
													.q(blueLineZeroQ));

// Name Table RAM
RAM_DUAL_U8 nameTableRAM( .clock(clk65),
													.data(nameRAMDataSync),
													.rdaddress(rdNameRAMAddrSync),
													.rden(rdNameRAMStrobeSync) ,
													.wraddress(wrNameRAMAddrSync),
													.wren(wrNameRAMStrobeSync),
													.q(patternIndexQ));

// Instantiate shift multipliers

ShiftMultiply32_U16  multiplyPatternIndex (.A(patternIndexQSync), .Y(patternIndexMult));

ShiftMultiply32_U8 multiplyScaleRedChan  (.A(redRawOrBlankChan)     , .Y(vgaR));
ShiftMultiply32_U8 multiplyScaleGreenChan(.A(greenRawOrBlankChan)   , .Y(vgaG));
ShiftMultiply32_U8 multiplyScaleBlueChan (.A(blueRawOrBlankChan)    , .Y(vgaB));

nameIndexCalculation nameIndexCalc(.*);

SpriteRotationCalculation spriteRotationCalc(.*);

assign localWriteEnable = chipSelect & write;

// This code block below controls writing to RAM from the Avalon master
always_ff @(posedge clk65)	begin
	// Align incomming Avalon data
	writeDataSync <= writeData;
	nameRAMDataSync        <= writeData[7:0];
	redTwoRAMDataSync      <= writeData;
	redOneRAMDataSync      <= writeData;
	redZeroRAMDataSync     <= writeData;
	greenTwoRAMDataSync    <= writeData;
	greenOneRAMDataSync    <= writeData;
	greenZeroRAMDataSync   <= writeData;
	blueTwoRAMDataSync     <= writeData;
	blueOneRAMDataSync     <= writeData;
	blueZeroRAMDataSync    <= writeData;

  // RED COLOR CHANNEL
	if(localWriteEnable & redTwoActive) begin
		wrRedTwoRAMAddrSync   <= address - RED_TWO_START;
		wrRedTwoRAMStrobeSync <= 1'b1;
	end
	else begin
		wrRedTwoRAMAddrSync   <= wrRedTwoRAMAddrSync;
		wrRedTwoRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & redOneActive) begin
		wrRedOneRAMAddrSync   <= address - RED_ONE_START;
		wrRedOneRAMStrobeSync <= 1'b1;
	end
	else begin
		wrRedOneRAMAddrSync   <= wrRedOneRAMAddrSync;
		wrRedOneRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & redZeroActive) begin
		wrRedZeroRAMAddrSync   <= address - RED_ZERO_START;
		wrRedZeroRAMStrobeSync <= 1'b1;
	end
	else begin
		wrRedZeroRAMAddrSync   <= wrRedZeroRAMAddrSync;
		wrRedZeroRAMStrobeSync <= 1'b0;
	end

	// GREEN COLOR CHANNEL
	if(localWriteEnable & greenTwoActive) begin
		wrGreenTwoRAMAddrSync   <= address - GREEN_TWO_START;
		wrGreenTwoRAMStrobeSync <= 1'b1;
	end
	else begin
		wrGreenTwoRAMAddrSync   <= wrGreenTwoRAMAddrSync;
		wrGreenTwoRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & greenOneActive) begin
		wrGreenOneRAMAddrSync  	<= address - GREEN_ONE_START;
		wrGreenOneRAMStrobeSync <= 1'b1;
	end
	else begin
		wrGreenOneRAMAddrSync  	<= wrGreenOneRAMAddrSync;
		wrGreenOneRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & greenZeroActive) begin
		wrGreenZeroRAMAddrSync   <= address - GREEN_ZERO_START;
		wrGreenZeroRAMStrobeSync <= 1'b1;
	end
	else begin
		wrGreenZeroRAMAddrSync   <= wrGreenZeroRAMAddrSync;
		wrGreenZeroRAMStrobeSync <= 1'b0;
	end

	// BLUE COLOR CHANNEL
	if(localWriteEnable & blueTwoActive) begin
		wrBlueTwoRAMAddrSync   <= address - BLUE_TWO_START;
		wrBlueTwoRAMStrobeSync <= 1'b1;
	end
	else begin
		wrBlueTwoRAMAddrSync   <= wrBlueTwoRAMAddrSync;
		wrBlueTwoRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & blueOneActive) begin
		wrBlueOneRAMAddrSync   <= address - BLUE_ONE_START;
		wrBlueOneRAMStrobeSync <= 1'b1;
	end
	else begin
		wrBlueOneRAMAddrSync   <= wrBlueOneRAMAddrSync;
		wrBlueOneRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & blueZeroActive) begin
		wrBlueZeroRAMAddrSync   <= address - BLUE_ZERO_START;
		wrBlueZeroRAMStrobeSync <= 1'b1;
	end
	else begin
		wrBlueZeroRAMAddrSync   <= wrBlueZeroRAMAddrSync;
		wrBlueZeroRAMStrobeSync <= 1'b0;
	end

	if(localWriteEnable & nameTableActive) begin
		wrNameRAMAddrSync       <= address - NAME_TABLE_START;
		wrNameRAMStrobeSync     <= 1'b1;
	end
	else begin
		wrNameRAMAddrSync       <= wrNameRAMAddrSync;
		wrNameRAMStrobeSync     <= 1'b0;
	end

	// COMMAND REGISTERS

	// NAME COLUMN OFFSET CAPUTRE
	if(localWriteEnable & nameOffsetXActive) nameColumnOffsetSync <= writeData[7:0];
	else nameColumnOffsetSync <= nameColumnOffsetSync;
	// NAME ROW OFFSET CAPUTRE
	if(localWriteEnable & nameOffsetYActive) nameRowOffsetSync <= writeData[9:0];
	else nameRowOffsetSync <= nameRowOffsetSync;
	// PIXEL X OFFSET CAPUTRE
	if(localWriteEnable & pixelOffsetXActive) pixelXOffsetSync <= writeData[4:0];
	else pixelXOffsetSync <= pixelXOffsetSync;
	// PIXEL Y OFFSET CAPUTRE
	if(localWriteEnable & pixelOffsetYActive) pixelYOffsetSync <= writeData[4:0];
	else pixelYOffsetSync <= pixelYOffsetSync;

	// SPRITE REGISTERS
	// Red Two - columns 0 through 3
	if(localWriteEnable & spriteRedTwoColZeroActive) redTwoSpriteColZeroSync <= writeData;
	else redTwoSpriteColZeroSync <= redTwoSpriteColZeroSync;

	if(localWriteEnable & spriteRedTwoColOneActive) redTwoSpriteColOneSync <= writeData;
	else redTwoSpriteColOneSync <= redTwoSpriteColOneSync;

	if(localWriteEnable & spriteRedTwoColTwoActive) redTwoSpriteColTwoSync <= writeData;
	else redTwoSpriteColTwoSync <= redTwoSpriteColTwoSync;

	if(localWriteEnable & spriteRedTwoColThreeActive) redTwoSpriteColThreeSync <= writeData;
	else redTwoSpriteColThreeSync <= redTwoSpriteColThreeSync;

  // Red One - columns 0 through 3
	if(localWriteEnable & spriteRedOneColZeroActive) redOneSpriteColZeroSync <= writeData;
	else redOneSpriteColZeroSync <= redOneSpriteColZeroSync;

	if(localWriteEnable & spriteRedOneColOneActive) redOneSpriteColOneSync <= writeData;
	else redOneSpriteColOneSync <= redOneSpriteColOneSync;

	if(localWriteEnable & spriteRedOneColTwoActive) redOneSpriteColTwoSync <= writeData;
	else redOneSpriteColTwoSync <= redOneSpriteColTwoSync;

	if(localWriteEnable & spriteRedOneColThreeActive) redOneSpriteColThreeSync <= writeData;
	else redOneSpriteColThreeSync <= redOneSpriteColThreeSync;

	// Red Zero - columns 0 through 3
	if(localWriteEnable & spriteRedZeroColZeroActive) redZeroSpriteColZeroSync <= writeData;
	else redZeroSpriteColZeroSync <= redZeroSpriteColZeroSync;

	if(localWriteEnable & spriteRedZeroColOneActive) redZeroSpriteColOneSync <= writeData;
	else redZeroSpriteColOneSync <= redZeroSpriteColOneSync;

	if(localWriteEnable & spriteRedZeroColTwoActive) redZeroSpriteColTwoSync <= writeData;
	else redZeroSpriteColTwoSync <= redZeroSpriteColTwoSync;

	if(localWriteEnable & spriteRedZeroColThreeActive) redZeroSpriteColThreeSync <= writeData;
	else redZeroSpriteColThreeSync <= redZeroSpriteColThreeSync;

	// Green Two - columns 0 through 3
	if(localWriteEnable & spriteGreenTwoColZeroActive) greenTwoSpriteColZeroSync <= writeData;
	else greenTwoSpriteColZeroSync <= greenTwoSpriteColZeroSync;

	if(localWriteEnable & spriteGreenTwoColOneActive) greenTwoSpriteColOneSync <= writeData;
	else greenTwoSpriteColOneSync <= greenTwoSpriteColOneSync;

	if(localWriteEnable & spriteGreenTwoColTwoActive) greenTwoSpriteColTwoSync <= writeData;
	else greenTwoSpriteColTwoSync <= greenTwoSpriteColTwoSync;

	if(localWriteEnable & spriteGreenTwoColThreeActive) greenTwoSpriteColThreeSync <= writeData;
	else greenTwoSpriteColThreeSync <= greenTwoSpriteColThreeSync;

  // Green One - columns 0 through 3
	if(localWriteEnable & spriteGreenOneColZeroActive) greenOneSpriteColZeroSync <= writeData;
	else greenOneSpriteColZeroSync <= greenOneSpriteColZeroSync;

	if(localWriteEnable & spriteGreenOneColOneActive) greenOneSpriteColOneSync <= writeData;
	else greenOneSpriteColOneSync <= greenOneSpriteColOneSync;

	if(localWriteEnable & spriteGreenOneColTwoActive) greenOneSpriteColTwoSync <= writeData;
	else greenOneSpriteColTwoSync <= greenOneSpriteColTwoSync;

	if(localWriteEnable & spriteGreenOneColThreeActive) greenOneSpriteColThreeSync <= writeData;
	else greenOneSpriteColThreeSync <= greenOneSpriteColThreeSync;

	// Green Zero - columns 0 through 3
	if(localWriteEnable & spriteGreenZeroColZeroActive) greenZeroSpriteColZeroSync <= writeData;
	else greenZeroSpriteColZeroSync <= greenZeroSpriteColZeroSync;

	if(localWriteEnable & spriteGreenZeroColOneActive) greenZeroSpriteColOneSync <= writeData;
	else greenZeroSpriteColOneSync <= greenZeroSpriteColOneSync;

	if(localWriteEnable & spriteGreenZeroColTwoActive) greenZeroSpriteColTwoSync <= writeData;
	else greenZeroSpriteColTwoSync <= greenZeroSpriteColTwoSync;

	if(localWriteEnable & spriteGreenZeroColThreeActive) greenZeroSpriteColThreeSync <= writeData;
	else greenZeroSpriteColThreeSync <= greenZeroSpriteColThreeSync;

	// Blue Two - columns 0 through 3
	if(localWriteEnable & spriteBlueTwoColZeroActive) blueTwoSpriteColZeroSync <= writeData;
	else blueTwoSpriteColZeroSync <= blueTwoSpriteColZeroSync;

	if(localWriteEnable & spriteBlueTwoColOneActive) blueTwoSpriteColOneSync <= writeData;
	else blueTwoSpriteColOneSync <= blueTwoSpriteColOneSync;

	if(localWriteEnable & spriteBlueTwoColTwoActive) blueTwoSpriteColTwoSync <= writeData;
	else blueTwoSpriteColTwoSync <= blueTwoSpriteColTwoSync;

	if(localWriteEnable & spriteBlueTwoColThreeActive) blueTwoSpriteColThreeSync <= writeData;
	else blueTwoSpriteColThreeSync <= blueTwoSpriteColThreeSync;

  // Blue One - columns 0 through 3
	if(localWriteEnable & spriteBlueOneColZeroActive) blueOneSpriteColZeroSync <= writeData;
	else blueOneSpriteColZeroSync <= blueOneSpriteColZeroSync;

	if(localWriteEnable & spriteBlueOneColOneActive) blueOneSpriteColOneSync <= writeData;
	else blueOneSpriteColOneSync <= blueOneSpriteColOneSync;

	if(localWriteEnable & spriteBlueOneColTwoActive) blueOneSpriteColTwoSync <= writeData;
	else blueOneSpriteColTwoSync <= blueOneSpriteColTwoSync;

	if(localWriteEnable & spriteBlueOneColThreeActive) blueOneSpriteColThreeSync <= writeData;
	else blueOneSpriteColThreeSync <= blueOneSpriteColThreeSync;

	// Blue Zero - columns 0 through 3
	if(localWriteEnable & spriteBlueZeroColZeroActive) blueZeroSpriteColZeroSync <= writeData;
	else blueZeroSpriteColZeroSync <= blueZeroSpriteColZeroSync;

	if(localWriteEnable & spriteBlueZeroColOneActive) blueZeroSpriteColOneSync <= writeData;
	else blueZeroSpriteColOneSync <= blueZeroSpriteColOneSync;

	if(localWriteEnable & spriteBlueZeroColTwoActive) blueZeroSpriteColTwoSync <= writeData;
	else blueZeroSpriteColTwoSync <= blueZeroSpriteColTwoSync;

	if(localWriteEnable & spriteBlueZeroColThreeActive) blueZeroSpriteColThreeSync <= writeData;
	else blueZeroSpriteColThreeSync <= blueZeroSpriteColThreeSync;

	
	// VGA COMMAND REGISTER
	if(localWriteEnable & vgaControlActive) blankScreenSync <= writeData[0];
	else blankScreenSync <= blankScreenSync;

	// SPRITE CONFIGURATION
  if(localWriteEnable & spriteConfigActive) begin
		wrSpriteZeroRAMAddrSync <= writeData[6:0];
		wrSpriteZeroRAMSync     <= 1'b1;
	end
	else begin
		wrSpriteZeroRAMAddrSync <= wrSpriteZeroRAMAddrSync;
		wrSpriteZeroRAMSync     <= 1'b0;
	end
	
	// SPRITE LOCATION
	if(localWriteEnable & spriteZeroPixelXOffsetActive) spriteZeroPixelXOffsetSync <= writeData[9:0];
	else spriteZeroPixelXOffsetSync <= spriteZeroPixelXOffsetSync;

	if(localWriteEnable & spriteZeroPixelYOffsetActive) spriteZeroPixelYOffsetSync <= writeData[9:0];
	else spriteZeroPixelYOffsetSync <= spriteZeroPixelYOffsetSync;
	
	// SPRITE ROTATIONS
	if(localWriteEnable & spriteZeroOriginXTransActive) begin
		translatedXOrigin <= writeData[15:0];
	end
	else begin
		translatedXOrigin <= translatedXOrigin;
	end
	if(localWriteEnable & spriteZeroOriginYTransActive) begin
		translatedYOrigin <= writeData[15:0];
	end
	else begin
		translatedYOrigin <= translatedYOrigin;
	end
	if(localWriteEnable & spriteZeroRotateXVecActive) begin
		vecXNormalized <= writeData[15:0];
	end
	else begin
		vecXNormalized <= vecXNormalized;
	end
	if(localWriteEnable & spriteZeroRotateYVecActive) begin
		vecYNormalized <= writeData[15:0];
	end
	else begin
		vecYNormalized <= vecYNormalized;
	end

	// SPRITE ZERO ATTRIBUTES
	if(localWriteEnable & spriteZeroAttributeActive) spriteZeroVisible <= writeData[0];
	else spriteZeroVisible <= spriteZeroVisible;
	
		// SOUND COMMAND REGISTER
	if(localWriteEnable & soundControlActive) begin
		soundControl <= writeData[3:0];
	end
	else begin
		soundControl <= soundControl;
	end


end
// SOUND CONTROL

assign AUD_XCK = audio_clk;
assign AUD_MUTE = (soundControl != 4'b0);

// SPRITE ZERO read control and datapath
// NOTE sprite offset can not go beyond 1024 - 128 in the X direction
assign spriteZeroInXBoundary = (spriteZeroPixelXOffsetSyncTwo <= hCount) & (hCount < spriteZeroPixelXOffsetSyncTwo + 7'd64);
assign spriteZeroInYBoundary = (spriteZeroPixelYOffsetSyncTwo <= vCount) & (vCount < spriteZeroPixelYOffsetSyncTwo + 7'd64);
assign spriteZeroInBoundary = spriteZeroInXBoundary & spriteZeroInYBoundary;

assign spritePixelX = (hCount - spriteZeroPixelXOffsetSyncTwo); // Multiply by two for downsampling
assign spritePixelY = (vCount - spriteZeroPixelYOffsetSyncTwo); // Multiply by two for downsampling

always_ff @(posedge clk65) begin

	wrRedTwoSpriteDataSync  <= {redTwoSpriteColZeroSync, redTwoSpriteColOneSync, redTwoSpriteColTwoSync, redTwoSpriteColThreeSync};
	wrRedOneSpriteDataSync  <= {redOneSpriteColZeroSync, redOneSpriteColOneSync, redOneSpriteColTwoSync, redOneSpriteColThreeSync};
	wrRedZeroSpriteDataSync <= {redZeroSpriteColZeroSync, redZeroSpriteColOneSync, redZeroSpriteColTwoSync, redZeroSpriteColThreeSync};
	
	wrGreenTwoSpriteDataSync  <= {greenTwoSpriteColZeroSync, greenTwoSpriteColOneSync, greenTwoSpriteColTwoSync, greenTwoSpriteColThreeSync};
	wrGreenOneSpriteDataSync  <= {greenOneSpriteColZeroSync, greenOneSpriteColOneSync, greenOneSpriteColTwoSync, greenOneSpriteColThreeSync};
	wrGreenZeroSpriteDataSync <= {greenZeroSpriteColZeroSync, greenZeroSpriteColOneSync, greenZeroSpriteColTwoSync, greenZeroSpriteColThreeSync};
	
	wrBlueTwoSpriteDataSync  <= {blueTwoSpriteColZeroSync, blueTwoSpriteColOneSync, blueTwoSpriteColTwoSync, blueTwoSpriteColThreeSync};
	wrBlueOneSpriteDataSync  <= {blueOneSpriteColZeroSync, blueOneSpriteColOneSync, blueOneSpriteColTwoSync, blueOneSpriteColThreeSync};
	wrBlueZeroSpriteDataSync <= {blueZeroSpriteColZeroSync, blueZeroSpriteColOneSync, blueZeroSpriteColTwoSync, blueZeroSpriteColThreeSync};
	
	if (spriteZeroInBoundary & spriteZeroVisible) begin
		rdSpriteZeroRAMSync <= 1'b1;
	end
	else begin
		rdSpriteZeroRAMSync <= 1'b0;
	end
end

assign redRawSpriteZeroChan[2:0]   = {redTwoSpriteZeroLineQ[spriteZeroXPixelSyncTwo], redOneSpriteZeroLineQ[spriteZeroXPixelSyncTwo], redZeroSpriteZeroLineQ[spriteZeroXPixelSyncTwo]};
assign greenRawSpriteZeroChan[2:0] = {greenTwoSpriteZeroLineQ[spriteZeroXPixelSyncTwo], greenOneSpriteZeroLineQ[spriteZeroXPixelSyncTwo], greenZeroSpriteZeroLineQ[spriteZeroXPixelSyncTwo]};
assign blueRawSpriteZeroChan[2:0]  = {blueTwoSpriteZeroLineQ[spriteZeroXPixelSyncTwo], blueOneSpriteZeroLineQ[spriteZeroXPixelSyncTwo], blueZeroSpriteZeroLineQ[spriteZeroXPixelSyncTwo]};

assign spriteZeroTransparentPixel =  (redRawSpriteZeroChan[2:0] == 3'd0) & (greenRawSpriteZeroChan[2:0] == 3'b111) & (blueRawSpriteZeroChan[2:0] == 3'd0);
assign spriteZeroActive = ~spriteZeroTransparentPixel & spriteZeroInBoundary & spriteZeroVisible;

// TODO verify that the combinational path for rgb pixel value updates do not need to be registered
assign redRawChan = spriteZeroActive ? redRawSpriteZeroChan     : redRawPatternChan;
assign greenRawChan = spriteZeroActive ? greenRawSpriteZeroChan : greenRawPatternChan;
assign blueRawChan = spriteZeroActive ? blueRawSpriteZeroChan   : blueRawPatternChan;

assign redRawOrBlankChan   = (blankScreenSync & vgaBlank_n) ? redRawChan   : 3'd0;
assign greenRawOrBlankChan = (blankScreenSync & vgaBlank_n) ? greenRawChan : 3'd0;
assign blueRawOrBlankChan  = (blankScreenSync & vgaBlank_n) ? blueRawChan  : 3'd0;

// Synchronous update of RAM Reads, line buffers, and position offsets
always_ff @(posedge clk65 or posedge reset) begin
	if (reset) begin
		currState <= P_IDLE;
		rdNameRAMStrobeSync    <= 1'b0;
		sampleIndexStrobeSync  <= 1'b0;
		samplePatternDataSync  <= 1'b0;
		nameRowOffsetSyncTwo    <= 8'd0;
		nameColumnOffsetSyncTwo <= 8'd0;
		pixelYOffsetSyncTwo     <= 5'd0;
		pixelXOffsetSyncTwo     <= 5'd0;
		spriteZeroPixelXOffsetSyncTwo <= 10'd0;
		spriteZeroPixelYOffsetSyncTwo <= 10'd0;
		redLineTwoCurrSync     <= 32'd0;
		redLineOneCurrSync     <= 32'd0;
		redLineZeroCurrSync    <= 32'd0;
		greenLineTwoCurrSync   <= 32'd0;
		greenLineOneCurrSync   <= 32'd0;
		greenLineZeroCurrSync  <= 32'd0;
		blueLineTwoCurrSync    <= 32'd0;
		blueLineOneCurrSync    <= 32'd0;
		blueLineZeroCurrSync   <= 32'd0;
		redLineTwoNextSync     <= 32'd0;
		redLineOneNextSync     <= 32'd0;
		redLineZeroNextSync    <= 32'd0;
		greenLineTwoNextSync   <= 32'd0;
		greenLineOneNextSync   <= 32'd0;
		greenLineZeroNextSync  <= 32'd0;
		blueLineTwoNextSync    <= 32'd0;
		blueLineOneNextSync    <= 32'd0;
		blueLineZeroNextSync   <= 32'd0;
	end
	else begin
		// Default assignments
		currState <= nextState;
		sampleIndexStrobeSync <= sampleIndexStrobe;
		samplePatternDataSync <= samplePatternData;

		// Update RAM pattern address read locations
		rdRedTwoRAMAddrSync      <= patternIndex;
		rdRedOneRAMAddrSync      <= patternIndex;
		rdRedZeroRAMAddrSync     <= patternIndex;
		rdGreenTwoRAMAddrSync    <= patternIndex;
		rdGreenOneRAMAddrSync    <= patternIndex;
		rdGreenZeroRAMAddrSync   <= patternIndex;
		rdBlueTwoRAMAddrSync     <= patternIndex;
		rdBlueOneRAMAddrSync     <= patternIndex;
		rdBlueZeroRAMAddrSync    <= patternIndex;

		rdRedTwoRAMStrobeSync      <= readPatternStrobe;
		rdRedOneRAMStrobeSync      <= readPatternStrobe;
		rdRedZeroRAMStrobeSync     <= readPatternStrobe;
		rdGreenTwoRAMStrobeSync    <= readPatternStrobe;
		rdGreenOneRAMStrobeSync    <= readPatternStrobe;
		rdGreenZeroRAMStrobeSync   <= readPatternStrobe;
		rdBlueTwoRAMStrobeSync     <= readPatternStrobe;
		rdBlueOneRAMStrobeSync     <= readPatternStrobe;
		rdBlueZeroRAMStrobeSync    <= readPatternStrobe;

		// Update RAM name address read locations
		rdNameRAMStrobeSync        <= readNameStrobe;

		// Sample New Pattern Index
		if(sampleIndexStrobeSync) begin
			patternIndexQSync <= patternIndexQ;
		end
		else begin
			patternIndexQSync <= patternIndexQSync;
		end

		// Sample New Pattern Data
		if(samplePatternDataSync) begin
			redLineTwoNextSync    <= redLineTwoQ;
			redLineOneNextSync    <= redLineOneQ;
			redLineZeroNextSync   <= redLineZeroQ;
			greenLineTwoNextSync  <= greenLineTwoQ;
			greenLineOneNextSync  <= greenLineOneQ;
			greenLineZeroNextSync <= greenLineZeroQ;
			blueLineTwoNextSync   <= blueLineTwoQ;
			blueLineOneNextSync   <= blueLineOneQ;
			blueLineZeroNextSync  <= blueLineZeroQ;
		end
		else begin
			redLineTwoNextSync    <= redLineTwoNextSync;
			redLineOneNextSync    <= redLineOneNextSync;
			redLineZeroNextSync   <= redLineZeroNextSync;
			greenLineTwoNextSync  <= greenLineTwoNextSync;
			greenLineOneNextSync  <= greenLineOneNextSync;
			greenLineZeroNextSync <= greenLineZeroNextSync;
			blueLineTwoNextSync   <= blueLineTwoNextSync;
			blueLineOneNextSync   <= blueLineOneNextSync;
			blueLineZeroNextSync  <= blueLineZeroNextSync;
		end

		// Swap out the line buffers
		if(updateLineBuffer) begin
			redLineTwoCurrSync    <= redLineTwoNextSync;
			redLineOneCurrSync    <= redLineOneNextSync;
			redLineZeroCurrSync   <= redLineZeroNextSync;
			greenLineTwoCurrSync  <= greenLineTwoNextSync;
			greenLineOneCurrSync  <= greenLineOneNextSync;
			greenLineZeroCurrSync <= greenLineZeroNextSync;
			blueLineTwoCurrSync   <= blueLineTwoNextSync;
			blueLineOneCurrSync   <= blueLineOneNextSync;
			blueLineZeroCurrSync  <= blueLineZeroNextSync;
		end
		else begin
			redLineTwoCurrSync    <= redLineTwoCurrSync;
			redLineOneCurrSync    <= redLineOneCurrSync;
			redLineZeroCurrSync   <= redLineZeroCurrSync;
			greenLineTwoCurrSync  <= greenLineTwoCurrSync;
			greenLineOneCurrSync  <= greenLineOneCurrSync;
			greenLineZeroCurrSync <= greenLineZeroCurrSync;
			blueLineTwoCurrSync   <= blueLineTwoCurrSync;
			blueLineOneCurrSync   <= blueLineOneCurrSync;
			blueLineZeroCurrSync  <= blueLineZeroCurrSync;
		end

		// Synchronize screen movements to vgaVS
		if(updateOffsetStrobe) begin
			nameRowOffsetSyncTwo    <= nameRowOffsetSync;
			nameColumnOffsetSyncTwo <= nameColumnOffsetSync;
			pixelYOffsetSyncTwo     <= pixelYOffsetSync;
			pixelXOffsetSyncTwo     <= pixelXOffsetSync;
			spriteZeroPixelXOffsetSyncTwo <= spriteZeroPixelXOffsetSync;
			spriteZeroPixelYOffsetSyncTwo <= spriteZeroPixelYOffsetSync;
		end
		else begin
			nameRowOffsetSyncTwo    <= nameRowOffsetSyncTwo;
			nameColumnOffsetSyncTwo <= nameColumnOffsetSyncTwo;
			pixelYOffsetSyncTwo     <= pixelYOffsetSyncTwo;
			pixelXOffsetSyncTwo     <= pixelXOffsetSyncTwo;
		end
	end
end

always_comb begin
  // Default Statements
  nextState = currState;			// hold state
	sampleIndexStrobe = 1'b0;
	readPatternStrobe = 1'd0;
	samplePatternData = 1'd0;
	readNameStrobe    = 1'd0;
	updateLineBuffer  = 1'd0;

	lineIndex  = vCount[4:0] + pixelYOffsetSyncTwo; // Should this increment outside of the active window?
	pixelIndex = hCount[4:0] + pixelXOffsetSyncTwo;

	nextRow = ~hActiveWindow & vActiveWindow & (lineIndex == 5'd31);
	nextColumn = hActiveWindow;

	if(hActiveWindow & vActiveWindow) patternIndex = patternIndexMult[12:0] + lineIndex;
	else if(~hActiveWindow & vActiveWindow) patternIndex = patternIndexMult[12:0] + (lineIndex + 5'd1);
	else patternIndex = patternIndexMult[12:0];

  redRawPatternChan[2:0]   = {redLineTwoCurrSync[pixelIndex], redLineOneCurrSync[pixelIndex], redLineZeroCurrSync[pixelIndex]};
  greenRawPatternChan[2:0] = {greenLineTwoCurrSync[pixelIndex], greenLineOneCurrSync[pixelIndex], greenLineZeroCurrSync[pixelIndex]};
  blueRawPatternChan[2:0]  = {blueLineTwoCurrSync[pixelIndex], blueLineOneCurrSync[pixelIndex], blueLineZeroCurrSync[pixelIndex]};

  // Logic to strobe name and pattern table reads
	if(vgaBlank_n) begin
		readNameStrobe = (pixelIndex == 5'd2) & (displayColumnSync != 5'd0); // Delay the name read for enough time for the new column propagation
		updateLineBuffer = pixelIndex == 5'd31;
	end
  else if(~vgaBlank_n & vActiveWindow) begin  // HSYNC region next line buffer updates
    readNameStrobe = (hCount == 11'd1027) | (hCount == 11'd1059); // nameRow and nameColumn will be updated in clocked logic
    updateLineBuffer = (hCount == 11'd1057) ; // first two 32 cycles after active region next line buffers updated
  end
  else begin
    readNameStrobe = ((hCount == 11'd1027) | (hCount == 11'd1059)) & (vCount == 10'd805); // Delay 1 cycle for address and read strobe alignment
    updateLineBuffer = (hCount == 11'd1057) & (vCount == 10'd805);
  end

  // State machine to sequence the read name table and pattern table accesses
	if(reset == 1'b1)			 nextState = P_IDLE;
	else begin
		case(currState)
			P_IDLE : begin
				if(readNameStrobe) begin
					nextState = P_SAMPLE_PATTERN_INDEX;
				end
			end
			P_SAMPLE_PATTERN_INDEX : begin
				sampleIndexStrobe = 1'b1;
				nextState = P_READ_PATTERN_DATA;
			end
			P_READ_PATTERN_DATA : begin
				readPatternStrobe = 1'b1;
				nextState = R_SAMPLE_PATTERN_DATA;
			end
			R_SAMPLE_PATTERN_DATA : begin
				samplePatternData = 1'b1;
				nextState = P_IDLE;
			end
		endcase
	end
end

	// Update display row and column locations for nameIndex calculations
  always_ff @(posedge clk65 or posedge reset) begin
    if (reset) begin
      displayColumnSync <= 5'd0;
    end
    else if (newFrameSync) displayColumnSync <= 5'd0;
    else if (updateLineBuffer & hActiveWindow) displayColumnSync <= displayColumnSync + 5'd1;
    else if (nextColUpdateZero) displayColumnSync <= 5'd0;
    else if (nextColUpdateOne)  displayColumnSync <= 5'd1;
    else if (resetColToZero)    displayColumnSync <= 5'd0;
	 else                        displayColumnSync <= displayColumnSync;
  end

  assign nextColUpdateZero   = hCount == 11'd1024;
  assign nextColUpdateOne    = hCount == 11'd1056;
  assign resetColToZero      = hCount == 11'd1343;

  // Vertical counters for name Index
  always_ff @(posedge clk65 or posedge reset) begin
	  if (reset)                                      displayRowSync <= 5'd0;
	  else if (newFrameSync)                          displayRowSync <= 5'd0;
	  else if (endOfVerticalPatterns & vActiveWindow) displayRowSync <= displayRowSync + 5'd1;
	  else if (endOfRows)                             displayRowSync <= 5'd0;
	  else                                            displayRowSync <= displayRowSync;
	end

	assign endOfVerticalPatterns = (lineIndex == 5'd31) & (hCount == 11'd1023) ;
	assign updateOffsetStrobe = (hCount == 11'd0) & (vCount == 10'd805);
	assign endOfRows  = displayRowSync == 5'd24;

endmodule
