%{ open Ast
(* authors: William Hom and Joseph Isaac Baker *)

let for_counter = 
  let count = ref (0) in
  fun () -> incr count; string_of_int !count;;

  %}

%token PLUS MINUS TIMES DIVIDE MOD POW DOTTIMES DOTDIVIDE 
%token EQUAL NOTEQUAL LESS LEQ GREATER GEQ 
%token AND OR
%token NOT NEG TRAN
%token ASSIGN
%token LCURLY LPAREN LBRACK RCURLY RPAREN RBRACK
%token SEMI COMMA DOT SQUOTE DQUOTE
%token VOID_TYP STR_TYP NUM_TYP MAT_TYP
%token IF ELSE FOR IN WHILE RETURN
%token OPERATOR

%token <int> INT
%token <float> FLOAT
%token <string> STRING
%token <string> ID
%token EOF

%nonassoc NOELSE
%nonassoc ELSE
%nonassoc EQUAL NOTEQUAL LESS LEQ GREATER GEQ
%right ASSIGN
%left OR AND
%left PLUS MINUS
%left TIMES DIVIDE MOD DOTTIMES DOTDIVIDE
%right NOT NEG POW
%left TRAN

%start program
%type <Ast.program> program

%%

program:
  global_stmts EOF { List.rev $1 }

global_stmts:
  /* nothing */           { [] }
| global_stmts stmt       { Stmt($2) :: $1 }
| global_stmts func_decl  { FuncDecl($2) :: $1 }
| global_stmts oper_decl  { OperDecl($2) :: $1 }

func_decl:
  typ ID LPAREN param_list RPAREN stmt_block
  {{ 
    fdtype    = $1;
    fname     = $2;
    fparams   = $4;
    fbody     = $6
  }}

oper_decl:
  typ OPERATOR oper_symbol LPAREN param_list RPAREN stmt_block  {{ opdtype = $1; operator = $3; opparams = $5; opbody = $7 }}

oper_symbol:
  PLUS        { Bop(Add)     }
| MINUS       { Bop(Sub)     }
| TIMES       { Bop(Mul)     }
| DIVIDE      { Bop(Div)     }
| POW         { Bop(Pow)     }
| MOD         { Bop(Mod)     }
| DOTTIMES    { Bop(DotMul)  }
| DOTDIVIDE   { Bop(DotDiv)  }
| EQUAL       { Bop(Equal)   }
| NOTEQUAL    { Bop(Neq)     }
| LESS        { Bop(Less)    }
| LEQ         { Bop(Leq)     }
| GREATER     { Bop(Greater) }
| GEQ         { Bop(Geq)     }
| AND         { Bop(And)     }
| OR          { Bop(Or)      }
| NOT         { Uop(Not)     }
| TRAN        { Uop(Tran)    }

param_list:
  /* nothing */          { []          }
| param                  { List.rev $1 }

param:
  typ ID                 { [($1, $2)]   }
| param COMMA typ ID     { ($3, $4)::$1 }

typ:
  VOID_TYP { VoidTyp       }
| STR_TYP  { StringTyp     }
| NUM_TYP  { NumberTyp     }
| MAT_TYP  { MatrixTyp     }

stmt_list:
    /* nothing */  { []       }
  | stmt_list stmt { $2 :: $1 }

stmt_block:
| LCURLY stmt_list RCURLY                             { Block(List.rev $2)    }

stmt:
  expr SEMI                                           { Expr($1)              }
| typ ID SEMI                                         { VDecl(($1, $2))       }
| RETURN SEMI                                         { Return Noexpr         }
| RETURN expr SEMI                                    { Return $2             }
| IF LPAREN expr RPAREN stmt_block %prec NOELSE       { If($3, $5, Block([])) }
| IF LPAREN expr RPAREN stmt_block ELSE stmt_block    { If($3, $5, $7)        }
| FOR LPAREN lvalue IN expr RPAREN stmt_block         { For($3, $5, $7, for_counter()) }
| WHILE LPAREN expr RPAREN stmt_block                 { While($3, $5)         }

/* comment on the FOR statement: */
/* you can't do this: for (int i <- 5 to 10) {} */
/* it has to be int i; for (i <- 5 to 10) {} */
/* due to the way the expressions are set up. */
/* can remedy this changing 'ID ASSIGN expr' to 'expr' */
/* it can lead to something nonsensical like for (3 + 3 to 8) {} */
/* but it's how microC does it. */

lvalue:
  typ ID                            { VarDecl(($1, $2))          }
| ID                                { IdAsn($1)                  }
| ID LBRACK expr RBRACK             { MatCellAsn($1, $3, Number(IntTyp, 0, 0.0)) }
| ID LBRACK expr COMMA expr RBRACK  { MatCellAsn($1, $3, $5)     }

expr:
/* data types */
  INT     { Number(IntTyp, $1, 0.0)     }
| FLOAT   { Number(FloatTyp, 0, $1)     }
| STRING  { Str($1)                     }
| ID      { Id($1)                      }
/* Matrix expressions */
| ID LBRACK expr RBRACK                       { MatAcc($1, $3, Number(IntTyp, 0, 0.0)) }
| ID LBRACK expr COMMA expr RBRACK            { MatAcc($1, $3, $5)     }
| LBRACK matrix_init_vars RBRACK              { MatInit($2)            }
| LBRACK RBRACK LPAREN expr COMMA expr RPAREN { MatEmptyInit($4, $6)   }
/* arithmetic expressions */
| expr PLUS expr       { Binop($1, Add, $3)     }
| expr MINUS expr      { Binop($1, Sub, $3)     }
| expr TIMES expr      { Binop($1, Mul, $3)     }
| expr DIVIDE expr     { Binop($1, Div, $3)     }
| expr MOD expr        { Binop($1, Mod, $3)     }
| expr POW expr        { Binop($1, Pow, $3)     }
| expr DOTTIMES expr   { Binop($1, DotMul, $3)  }
| expr DOTDIVIDE expr  { Binop($1, DotDiv, $3)  }
| MINUS expr %prec NEG { Unop(Neg, $2)          }
| expr TRAN            { Unop(Tran, $1)         }
/* Relational expressions */
| expr EQUAL expr      { Binop($1, Equal, $3)   }
| expr NOTEQUAL expr   { Binop($1, Neq, $3)     }
| expr LESS expr       { Binop($1, Less, $3)    }
| expr LEQ expr        { Binop($1, Leq, $3)     }
| expr GREATER expr    { Binop($1, Greater, $3) }
| expr GEQ expr        { Binop($1, Geq, $3)     }
/* logical expressions */
| expr AND expr        { Binop($1, And, $3)     }
| expr OR expr         { Binop($1, Or, $3)      }
| NOT expr             { Unop(Not, $2)          }
/* assignment expression */
| lvalue ASSIGN expr   { Assign($1, $3)         }
/* function call expression */
| ID LPAREN args_opt RPAREN { Func($1, $3)      }
/* parenthetical expressions */
| LPAREN expr RPAREN   { $2                     }

args_opt:
  /* nothing */ { []          }
| args_list     { List.rev $1 }

args_list:
  expr                 { [$1]   }
| args_list COMMA expr { $3::$1 }

matrix_init_vars:
| rows { List.rev $1 }  

rows:
  columns            { [ List.rev $1 ]   }
| rows SEMI columns  { (List.rev $3)::$1 }

columns:
  expr { [ $1 ] }
| columns COMMA expr { $3::$1 }
