(* Top-level of the MicroC compiler: scan & parse the input,
   check the resulting AST, generate LLVM IR, and dump the module *)

type action = Ast | Compile | LLVM_IR

let _ =
  let action = if Array.length Sys.argv > 1 then
    List.assoc Sys.argv.(1) [ ("-a", Ast);  (* Print the AST only *)
            ("-l", LLVM_IR);  (* Generate LLVM, don't check *)
            ("-c", Compile) ] (* Generate, check LLVM IR *)
  else Compile in
  let lexbuf = Lexing.from_channel stdin in
  let ast = Parser.program Scanner.token lexbuf in
  let sast = Evaluator.check ast in
  match action with
    Ast ->  print_string (Ast.string_of_program ast);
            print_string "\n\nSemantically Checked AST: \n";
            print_string (Sast.string_of_checked_program sast)
  | LLVM_IR -> let p = (Compilator.translate sast) in
    print_string (Llvm.string_of_llmodule p)
  | Compile -> let p = (Compilator.translate sast) in
    Llvm_analysis.assert_valid_module p;
    print_string (Llvm.string_of_llmodule p)