(* author: Joseph Isaac Baker *)

type b_op = Add | Sub | Mul | Div | Mod | Pow | DotMul | DotDiv
          | Equal | Neq | Less | Leq | Greater | Geq 
          | And | Or
type u_op = Not | Neg | Tran

type op = Bop of b_op | Uop of u_op

type typ = 
  NumberTyp
| StringTyp
| MatrixTyp
| VoidTyp 

type num_typ = IntTyp | FloatTyp

type binding = typ * string

type lvalue = 
  VarDecl     of binding
| IdAsn       of string
| MatCellAsn  of string * expr * expr

and expr =
  Number        of num_typ * int * float
| Str           of string
| Id            of string
| MatAcc        of string * expr * expr
| MatInit       of expr list list
| MatEmptyInit  of expr * expr
| Binop         of expr * b_op * expr
| Unop          of u_op * expr
| Assign        of lvalue * expr
| Func          of string * expr list
| Noexpr

type stmt =
  Block   of stmt list
| Expr    of expr
| VDecl   of binding
| Return  of expr
| If      of expr * stmt * stmt
| For     of lvalue * expr * stmt * string
| While   of expr * stmt

type func_decl = {
    fdtype  : typ;
    fname   : string;
    fparams : binding list;
    fbody   : stmt;
  }

type oper_decl = {
    opdtype   : typ;
    operator  : op;
    opparams  : binding list;
    opbody    : stmt;
  }

type array = {
    array   : typ list;
    dtype   : typ;
    rows    : int;
    columns : int;
  }

type global_stmt =
  Stmt     of stmt
| FuncDecl of func_decl
| OperDecl of oper_decl

type program = global_stmt list

(* Complete dump from MicroC.
 * Need to retrofit it to our purposes
 * Pretty-printing functions 
 *)
let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mul -> "*"
  | Div -> "/"
  | Mod -> "%"
  | Pow -> "^"
  | DotMul -> ".*"
  | DotDiv -> "./"
  | Equal -> "="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"

let string_of_uop = function
    Neg -> "-"
  | Not -> "!"
  | Tran -> "'"

let name_of_op = function
    Bop(o) -> (match o with
                | Add -> "add"
                | Sub -> "sub"
                | Mul -> "mul"
                | Div -> "div"
                | Mod -> "mod"
                | Pow -> "pow"
                | DotMul -> "dotmul"
                | DotDiv -> "dotdiv"
                | Equal -> "equal"
                | Neq -> "notequal"
                | Less -> "less"
                | Leq -> "leq"
                | Greater -> "greater"
                | Geq -> "geq"
                | And -> "and"
                | Or -> "or")
  | Uop(o) -> (match o with
                | Neg -> "neg"
                | Not -> "not"
                | Tran -> "tran")

let rec string_of_typ = function
    NumberTyp -> "number"
  | MatrixTyp -> "matrix"
  | StringTyp -> "string"
  | VoidTyp   -> "void"

let string_of_vdecl (t, id) = string_of_typ t ^ " " ^ id ^ "; "

let rec string_of_lvalue = function
    VarDecl(b) -> string_of_typ (fst b) ^ " " ^ (snd b)
  | IdAsn(s) -> s
  | MatCellAsn(s, e1, e2) -> s ^ "[" ^ string_of_expr e1 ^ ", " ^ string_of_expr e2 ^ "]"

and string_of_expr = function
    Number(t, i, f) -> if t = IntTyp then string_of_int i else string_of_float f
  | Str(l) -> l
  | Id(s) -> s
  | MatAcc(s, e1, e2) -> s ^ "[" ^ string_of_expr e1 ^ ", " ^ string_of_expr e2 ^ "]"
  | MatInit(ell) -> "[" ^ string_of_expr_list_list ell ^ "]"
  | MatEmptyInit(e1, e2) -> "[ ](" ^ string_of_expr e1 ^ ", " ^ string_of_expr e2 ^ ")"
  | Binop(e1, o, e2) ->
      "(" ^ string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2 ^ ")"
  | Unop(o, e) -> string_of_uop o ^ string_of_expr e
  | Assign(v, e) -> string_of_lvalue v ^ " <- " ^ string_of_expr e
  | Func(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr -> ""

and string_of_expr_list e_list = String.concat ", " (List.map string_of_expr e_list) 
and string_of_expr_list_list e_list_list = String.concat "; " (List.map string_of_expr_list e_list_list)

let rec string_of_stmt p = function
    Block(stmts) ->
      p ^ "{\n" ^ String.concat "" (List.map (string_of_stmt (p ^ "\t")) stmts) ^ p ^ "}\n"
  | Expr(expr) -> p ^ string_of_expr expr ^ ";\n";
  | VDecl((t, i)) -> p ^ string_of_typ t ^ " " ^ i ^ ";\n";
  | Return(expr) -> p ^ "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> p ^ "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt p s
  | If(e, s1, s2) ->  p ^ "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt p s1 ^ p ^ "else\n" ^ string_of_stmt p s2
  | For(lv, e, s, _) ->
      p ^ "for (" ^ string_of_lvalue lv ^ " in " ^ string_of_expr e ^ ") " ^ string_of_stmt p s
  | While(e, s) -> p ^ "while (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt p s

let string_of_fdecl fdecl =
  string_of_typ fdecl.fdtype ^ " " ^
  fdecl.fname ^ "(" ^ String.concat ", " (List.map (fun (t, n) -> string_of_typ t ^ " " ^ n) fdecl.fparams) ^ ")\n" ^
  string_of_stmt "" fdecl.fbody

let string_of_odecl odecl =
  string_of_typ odecl.opdtype ^ " operator " ^
  name_of_op odecl.operator ^ "(" ^ String.concat ", " (List.map (fun (t, n) -> string_of_typ t ^ " " ^ n) odecl.opparams) ^ ")\n" ^
  string_of_stmt "" odecl.opbody

let string_of_glbl_stmt = function
  Stmt(st) -> string_of_stmt "" st
| FuncDecl(f) -> string_of_fdecl f
| OperDecl(o) -> string_of_odecl o

let string_of_program (glbl_stmts) =
  String.concat "" (List.map string_of_glbl_stmt glbl_stmts)
