open Ast
module StringMap = Map.Make(String)

let translate (nodes, graphs, functions) = 

    let named_nodes = List.map (fun node -> node.nname) nodes in

    (* Raise an exception if the given list has a duplicate *)
    let report_duplicate exceptf lst =
        let rec helper = function
            n1 :: n2 :: _ when n1 = n2 -> raise (Failure (exceptf n1))
            | _ :: t -> helper t
            | [] -> ()
        in helper (List.sort compare lst)
    in

    (* Check named nodes. Ensure no duplicate declarations. *)
    report_duplicate (fun n -> "Duplicate named node "^ n)
        (List.map (fun node -> node.nname) nodes);

    (* Node identifers in a pattern must appear exactly once. Edges can appear multiple times. *)
    let check_idpatt id isedge patt =
        if List.exists (fun x -> x = id) patt.pids then ()
        else raise (Failure (id ^" must appear in its pattern."));

        let rec count target occurrences = function
            [] -> occurrences
            | x :: xs -> if x = target then count target (occurrences+1) xs
                else count target occurrences xs in
        let nodeids =
            let rec get_nodeids accum = function
                [] -> accum
                | n::[] -> n::accum  (* Order doesn't matter for this check; it can be reversed. *)
                | n::e::xs -> get_nodeids (n::accum) xs
            in get_nodeids [] patt.pids
        in
        List.iter (fun x -> if count x 0 patt.pids = 1 then ()
            else raise (Failure (x ^" must appear exactly once in a pattern."))
            ) (if not isedge then id::nodeids else nodeids);

        if isedge then (if List.exists (fun x -> x = id) nodeids
            then raise (Failure (id ^" must be in an edge position in a pattern."))
            else () )
        else (if List.exists (fun x -> x = id) nodeids then ()
            else raise (Failure (id ^" must be in a node position in a pattern.")))
    in
    List.iter (fun node -> check_idpatt node.nidpatt false node.npatt) nodes;

    report_duplicate (fun g -> "Duplicate graph declaration "^ g)
        (List.map (fun g -> g.gname ) graphs);

    let check_assign lvaluet rvaluet err =
        if lvaluet = rvaluet then lvaluet else raise err
    in

 (* Check functions: *)

    (* Check that a function named "main" is defined *)
    if not (List.mem "main" (List.map (fun fd -> fd.fname) functions))
    then raise (Failure ("Main function not found")) else ();
    
    (* Check that stdlib functions are not re-defined *)
    if List.mem "print" (List.map (fun fd -> fd.fname) functions)
    then raise (Failure ("Function 'print' may not be defined")) else ();
    if List.mem "append" (List.map (fun fd -> fd.fname) functions)
    then raise (Failure ("Function 'append' may not be defined")) else ();
    if List.mem "remove" (List.map (fun fd -> fd.fname) functions)
    then raise (Failure ("Function 'remove' may not be defined")) else ();
    if List.mem "length" (List.map (fun fd -> fd.fname) functions)
    then raise (Failure ("Function 'length' may not be defined")) else ();

    report_duplicate (fun n -> "Duplicate function "^ n)
        (List.map (fun func -> func.fname) functions);

    (* The definitions for the "standard library" functions are generated at compile-time.
       The following basic information is only used for the static semantic analysis. *)
    let built_in_funcs = [ { fname="print"; formals=[(TStr, "x")]; freturns=TVoid; flocals=[]; fbody=[] };
        { fname="append"; formals=[(TVoid, "x"); (TVoid, "y")]; freturns=TVoid; flocals=[]; fbody=[] };
        { fname="remove"; formals=[(TVoid, "x"); (TVoid, "y")]; freturns=TVoid; flocals=[]; fbody=[] };
        { fname="length"; formals=[(TVoid, "y")]; freturns=TInt; flocals=[]; fbody=[] } ]
    in

    let function_decls = List.fold_left (fun m fd -> StringMap.add fd.fname fd m)
        StringMap.empty (built_in_funcs @ functions)
    in

    let function_decl s = try StringMap.find s function_decls
        with Not_found -> raise (Failure ("Unrecognized function "^ s))
    in

    (* NOTE: There's definitely a better way of doing this in OCaml. *)
    let sast_biop = function
        Add -> Sast.Add | And -> Sast.And | Div -> Sast.Div | Eq -> Sast.Eq
        | Geq -> Sast.Geq | Gt -> Sast.Gt | Leq -> Sast.Leq | Lt -> Sast.Lt
        | Mul -> Sast.Mul | Neq -> Sast.Neq | Or -> Sast.Or | Sub -> Sast.Sub in
    let sast_unop = function
        Neg -> Sast.Neg | Not -> Sast.Not in
    let sast_ge = function
        NodeId(a) -> Sast.NodeId(a) | EdgeId(a,b,c) -> Sast.EdgeId(a,b,c) in
    let rec sast_typ = function
        TEdge -> Sast.TEdge | TNode -> Sast.TNode | TNNode(s) -> Sast.TNNode(s)
        | TGraph -> Sast.TGraph | TInt -> Sast.TInt | TBool -> Sast.TBool
        | TStr -> Sast.TStr | TList(t) -> Sast.TList(sast_typ t) | TVoid -> Sast.TVoid in
    let sast_tnl tnl = List.map (fun (t,n) -> (sast_typ t, n)) tnl in
    let sast_patt patt = { Sast.pids = patt.pids; Sast.preds = patt.preds } in
    let sast_nds nds = List.map (fun nd ->
        { Sast.nname = nd.nname; Sast.nidpatt = nd.nidpatt; Sast.npatt = sast_patt nd.npatt }
        ) nds in
    let graph_stmt = function
        GraphSet(gel, sil) -> Sast.GraphSet(List.map sast_ge gel, sil)
        | GraphDel(gel) -> Sast.GraphDel(List.map sast_ge gel)
    in
    let sast_gds gds = List.map (fun gd ->
        { Sast.gname = gd.gname; Sast.gbody = List.map graph_stmt gd.gbody }) gds
    in

    let check_node_name_exists name = if List.exists (fun n-> n = name) named_nodes
        then () else raise (Failure (name ^" is not a defined named node."))
    in

    let translate_function func =

        let check_named_node = function TNNode(n) -> ignore (check_node_name_exists n)
            | _ -> () in

        (* Check if any node types used in the list of formal parameters or local variables are undefined *)
        List.iter (fun (typ, _) -> check_named_node typ; ()) (func.formals @ func.flocals);

        report_duplicate (fun n-> "In "^ func.fname ^", duplicate formal "^ n)
            (List.map snd func.formals);
        report_duplicate (fun n-> "In "^ func.fname ^", duplicate local "^ n)
            (List.map snd func.flocals);

        let rec find_variable (scope : symbol_table) name =
            try List.find (fun (_, s) -> s = name) scope.variables
            with Not_found -> match scope.parent with
                Some(parent) -> find_variable parent name
                | _ -> raise Not_found
        in

        let rec list_variables_names (scope : symbol_table) =
            List.map snd scope.variables @ (match scope.parent with 
                Some(parent) -> list_variables_names parent
                | _ -> [])
        in

        let rec expr env = function
            Biop(e1, o, e2) as ex -> let e1' = expr env e1 and e2' = expr env e2 in
                let t1 = fst e1' and t2 = fst e2' in 
                let biop_type = (match o with
                Add when t1 = TInt && t2 = TInt -> TInt
                | And when t1 = TBool && t2 = TBool -> TBool
                | Div when t1 = TInt && t2 = TInt -> TInt
                | Eq when t1 = t2 -> TBool
                | Geq when t1 = TInt && t2 = TInt -> TBool
                | Gt when t1 = TInt && t2 = TInt -> TBool
                | Leq when t1 = TInt && t2 = TInt -> TBool
                | Lt when t1 = TInt && t2 = TInt -> TBool
                | Mul when t1 = TInt && t2 = TInt -> TInt
                | Neq when t1 = t2 -> TBool
                | Or when t1 = TBool && t2 = TBool -> TBool
                | Sub when t1 = TInt && t2 = TInt -> TInt
                | _ -> raise (Failure ("In "^ func.fname ^", illegal binary operator "^
                    string_of_typ t1 ^" "^ string_of_biop o ^" "^ string_of_typ t2 ^
                    " in "^ string_of_expr ex))) in
                (biop_type, Sast.Biop(snd e1', sast_biop o, snd e2'))
            | BoolLit(b) -> (TBool, Sast.BoolLit(b))
            | Call(s, el) as call -> (match s with
                (* NOTE: Use of standard library functions would more generally be validated
                   like the use of any other functions. For now, we e.g. hard-code variable number of
                   argument support (as used in print()), since PLOG doesn't provide this support generally. *)
                "append" -> (match el with
                    e1 :: e2 :: [] -> (
                        let lst = expr env e2 in (match fst lst with
                        TList(t) -> let ele = expr env e1 in
                            if t <> fst ele then raise (Failure ("In "^ func.fname ^", illegal append, "^
                            string_of_typ (fst lst) ^" append "^ string_of_typ (fst ele) ^" in "^
                            "append( "^ string_of_expr e1 ^", "^ string_of_expr e2 ^" )"))
                            else (TVoid, Sast.Call(s, [snd ele; snd lst]))
                        | _ -> raise (Failure ("In "^ func.fname ^", expected list second parameter, in "^
                            "append( "^ string_of_expr e1 ^", "^ string_of_expr e2 ^" )"))) )
                    | _ -> raise (Failure ("In "^ func.fname ^", append expected 2 parameters, of type: t "^
                        "and TList(t)")) )
                | "length" -> (match el with
                    e :: [] -> let lst = expr env e in (match fst lst with
                        TList(_) -> (TInt, Sast.Call(s, [snd lst]))
                        | _ -> raise (Failure ("In "^ func.fname ^", length expects 1 TList parameter")) )
                    | _ -> raise (Failure ("In "^ func.fname ^", length expects 1 TList parameter")) )
                | "print" -> (match el with
                    StrLit(str) :: params -> (
                        (* Go through s to create a list of types, based on the presence of %b, %d, and %s *)
                        let rec tlist subs =
                            let idx = try String.index subs '%' with Not_found -> -1 in
                            (* NOTE: idx+1 or idx+2 may be at/beyond the end of subs. *)
                            if idx = -1 then [] else (match subs.[idx+1] with
                                | 'b' -> (try TBool :: tlist (String.sub subs (idx+2) ((String.length subs) - (idx+2)))
                                    with Invalid_argument(_) -> print_endline "invalid bool"; TBool :: [])
                                | 'd' -> (try TInt :: tlist (String.sub subs (idx+2) ((String.length subs) - (idx+2)))
                                    with Invalid_argument(_) -> print_endline "invalid int"; TInt :: [])
                                | 's' -> (try TStr :: tlist (String.sub subs (idx+2) ((String.length subs) - (idx+2)))
                                    with Invalid_argument(_) -> print_endline "invalid str"; TStr :: [])
                                | _ -> (try tlist (String.sub subs (idx+1) ((String.length subs) - (idx+1)))
                                    with Invalid_argument(_) -> print_endline "not %b, %d, or %s"; [])
                                ) in
                        let lst = tlist str in
                        if List.length params <> List.length lst then
                            raise (Failure ("In "^ func.fname ^", print requires same number of escaped values"))
                        (* Then compare this list with params, one by one, making sure that each match *)
                        (* Use fact that List.exists2 returns Invalid_argument if list lengths don't match? *)
                        else (let pexprs = List.map (expr env) params in
                            if List.exists2 (fun a b -> fst a <> b) pexprs lst then
                                raise (Failure ("In "^ func.fname ^", print requires matching escaped value types"))
                            else (TVoid, Sast.Call(s, Sast.StrLit(str) :: List.map snd pexprs)) ) )
                    | _ -> raise (Failure ("In "^ func.fname ^", print expects a StrLit and optional expr list")) )
                | "remove" -> (match el with
                    e1 :: e2 :: [] -> (
                        let lst = expr env e2 in (match fst lst with
                        TList(t) -> let ele = expr env e1 in
                            (if t <> fst ele then raise (Failure ("In "^ func.fname ^
                                ", type mismatch in list remove: "^ string_of_typ (fst lst) ^
                                " remove "^ string_of_typ (fst ele) ^" in: remove( "^
                                string_of_expr e1 ^", "^ string_of_expr e2 ^" )"))
                                else (TVoid, Sast.Call(s, [snd ele; snd lst])) )
                        | _ -> raise (Failure ("In "^ func.fname ^", expected list second parameter: "^
                            "remove( "^ string_of_expr e1 ^", "^ string_of_expr e2 ^" )"))) )
                    | _ -> raise (Failure ("In "^ func.fname ^", remove expected 2 parameters, of type: t "^
                        "and TList(t)")) )

                (* Otherwise, call a non-built-in function: *)
                | _ -> (let func_decl = function_decl s in
                    if List.length el <> List.length func_decl.formals then
                        raise (Failure ("In "^ func.fname ^", expecting "^ string_of_int
                        (List.length func_decl.formals) ^" arguments in call to "^ string_of_expr call))
                    else let pexprs = List.map (expr env) el in
                        List.iter2 (fun (formal, formal_name) exp -> let actual = fst exp in
                        ignore (check_assign formal actual (Failure ("In "^ func.fname ^
                        ", illegal actual argument "^ string_of_typ actual ^", expected "^
                        string_of_typ formal ^" for formal parameter '"^ formal_name ^"' in call to "^
                        string_of_expr call)))) func_decl.formals pexprs;
                        (func_decl.freturns, Sast.Call(s, List.map snd pexprs))) )
            | GraphAccess(s, ge) as ex -> let vdecl = try find_variable env.scope s
                with Not_found -> raise (Failure ("In "^ func.fname ^", undeclared identifier "^ s)) in
                if fst vdecl <> TGraph then raise (Failure ("In "^ func.fname ^", "^ s ^" is not a graph in "^
                    string_of_expr ex))
                else ( (match ge with
                    NodeId(_) -> TNode
                    | EdgeId(_,_,_) -> TEdge), Sast.GraphAccess(s, sast_ge ge, list_variables_names env.scope) )
            | Id(s) -> let vdecl = try find_variable env.scope s
                with Not_found -> raise (Failure ("In "^ func.fname ^", undeclared identifier "^ s)) in
                (fst vdecl, Sast.Id(s))
            | Inf -> (TInt, Sast.Inf)
            | IntLit(i) -> (TInt, Sast.IntLit(i))
            | Lst(t, el) -> (TList(t), Sast.Lst(sast_typ t, List.map snd (List.map (expr env) el))) (* Use composition? *)
            | Nil(t) -> (t, Sast.Nil(sast_typ t))
            | Property(e, s) -> let e' = expr env e in (match fst e' with
                TNode | TEdge | TNNode(_) -> (TInt, Sast.Property(snd e', s))
                | x -> raise (Failure ("In "^ func.fname ^", objects of type "^ string_of_typ x ^
                    " may not have properties.")))
            | StrLit(s) -> (TStr, Sast.StrLit(s))
            | Unop(o, e) as ex -> let e' = expr env e in (match o with
                Neg when fst e' = TInt -> (TInt, Sast.Unop(sast_unop o, snd e'))
                | Not when fst e' = TBool -> (TBool, Sast.Unop(sast_unop o, snd e'))
                | _ -> raise (Failure ("In "^ func.fname ^", illegal unary operator "^
                    string_of_typ (fst e') ^" "^ string_of_unop o ^" in "^ string_of_expr ex)))
        in

        let check_bool_expr env e =
            let e' = expr env e in
            if fst e' <> TBool then raise (Failure ("In "^
            func.fname ^", expected a boolean expression in: "^ string_of_expr e))
            else snd e'
        in

        let rec stmt env = function
            Assign(e1, e2) as st -> let lt = expr env e1 and rt = expr env e2 in
                check_assign (fst lt) (fst rt) (Failure ("In "^ func.fname ^", illegal assignment "^
                string_of_typ (fst lt) ^" = "^ string_of_typ (fst rt) ^" in "^ string_of_stmt st));
                Sast.Assign(snd lt, snd rt)
            | Block(tnl, sl) ->
                List.iter (fun (typ, _) -> check_named_node typ; ()) tnl;
                let scope' = { parent = Some( env.scope ); variables = tnl } in
                let env' = { env with scope = scope' } in
                Sast.Block(sast_tnl tnl, List.map (fun s -> stmt env' s) sl)
            | Expr e -> Sast.Expr (snd (expr env e))
            | For(t, s, {pids=[];preds=[]}, e, (Block(tnl,sl) as b)) ->
                (* TODO: NOW: We need to add (t,s) to the local vars AFTER this for loop, somehow, as well. *)
                let e' = expr env e in
                (*let newb = Block( (t,s)::tnl, sl ) in*)
                (* scope' and env' could be defined below, so that they're not defined if fst e' doesn't match a TGraph or TList *)
                let scope' = { env.scope with variables = (t,s)::env.scope.variables } in
                let env' = { env with scope = scope' } in
                (match fst e' with
                TGraph -> ignore (match t with
                    TNode | TEdge -> ()
                    | TNNode(nn) -> ignore (check_node_name_exists nn)
                    | _ -> raise (Failure ("For loop must have node, edge, or named node")));
                    Sast.For(sast_typ t, s, {Sast.pids=[];Sast.preds=[]}, snd e', true, stmt env' b, list_variables_names env.scope)
                | TList(lt) ->
                    (* For the purposes of iterating over lists, nodes and named nodes are both considered nodes. *)
                    let temptyp = function TNNode(_) -> TNode | other -> other in
                    if temptyp t <> temptyp lt then raise (Failure ("For loop type doesn't match list type")) else
                    Sast.For(sast_typ t, s, {Sast.pids=[];Sast.preds=[]}, snd e', false, stmt env' b, list_variables_names env.scope)
                | _ -> raise (Failure ("For loop must iterate over a graph or list")))
            | For(t, s, p, e, (Block(tnl,sl) as b)) -> if List.exists (fun x -> x = s) p.pids then
                let e' = expr env e in
                let scope' = { env.scope with variables = (t,s)::env.scope.variables } in
                let env' = { env with scope = scope' } in
                (match fst e' with
                TGraph -> (match t with
                    TNode -> ()
                    | TEdge -> ignore (if List.length p.pids <> 3 then
                        raise (Failure ("For loops iterating with edges must have exactly 3 identifiers in the pattern, "^
                            "of the form: node1 edge-> node2")) else ())
                    | TNNode(nn) -> ignore (check_node_name_exists nn)
                    | _ -> raise (Failure ("For loops over graphs must iterate with nodes, edges, or named nodes")));
                    ignore (check_idpatt s (t = TEdge) p);
                    Sast.For(sast_typ t, s, sast_patt p, snd e', true, stmt env' b, list_variables_names env.scope)
                | TList(lt) ->
                    let temptyp = function TNNode(_) -> TNode | other -> other in
                    if temptyp t <> temptyp lt then raise (Failure ("For loop type doesn't match list type"))
                    else ( (match t with
                        TNode | TNNode(_) -> ()
                        | TEdge -> ignore (if List.length p.pids <> 3 then
                            raise (Failure ("For loops iterating with edges must have exactly 3 identifiers in the pattern, "^
                                "of the form: node1 edge-> node2")) else ())
                        | _ -> raise (Failure ("For loops with patterns can only iterate with nodes, edges, or named nodes")) );
                        ignore (check_idpatt s (t = TEdge) p);
                        Sast.For(sast_typ t, s, sast_patt p, snd e', false, stmt env' b, list_variables_names env.scope) )
                | _ -> raise (Failure ("For loop must iterate over a graph or list")))
                else raise (Failure ("For loop identifier must exist in the pattern"))
            | GraphDef(s, gsl) -> Sast.GraphDef(s, List.map graph_stmt gsl, list_variables_names env.scope)
            | If(e, b1, b2) -> Sast.If(check_bool_expr env e, stmt env b1, stmt env b2)
            | Return e -> let e' = expr env e in if fst e' = func.freturns then Sast.Return (snd e')
                else raise (Failure ("In "^ func.fname ^", return type is "^ string_of_typ (fst e') ^
                "; but expecting "^ string_of_typ func.freturns))
            | While(e, s) -> Sast.While(check_bool_expr env e, stmt env s)
        in

        let newenv = {
            scope = { parent = None;
                variables = (List.map (fun gd -> (TGraph, gd.gname)) graphs) @ func.formals};
            return_type = func.freturns }
        in

        { Sast.fname = func.fname; Sast.freturns = sast_typ func.freturns; Sast.formals = sast_tnl func.formals;
          Sast.flocals = sast_tnl func.flocals; Sast.fbody = 
            [stmt newenv (Block(func.flocals, func.fbody))] }

    in (sast_nds nodes, sast_gds graphs, List.map translate_function functions);
