type unop = Neg | Not

type biop = Add | Sub | Mul | Div | Eq | Neq | Lt | Leq | Gt | Geq | And | Or

type typ =
  TEdge
| TNode
| TNNode of string
| TGraph
| TInt
| TBool
| TStr
| TList of typ
| TVoid

type typdname = typ * string

type graph_elem =
  NodeId of string
| EdgeId of string * string * string

type expr =
  IntLit of int
| BoolLit of bool
| StrLit of string
| Lst of typ * expr list
| Id of string
| Unop of unop * expr
| Biop of expr * biop * expr
| Call of string * expr list
| GraphAccess of string * graph_elem
| Property of expr * string    (* NOTE: Or consider expr * expr with use of [] chars *)
| Nil of typ
| Inf

type patt = {
  pids : string list;
  preds : (string * int) list; (* NOTE: Consider supporting id.prop = val *)
}

type graph_stmt =
  GraphSet of graph_elem list * (string * int) list  (* (string * int) is for properties *)
| GraphDel of graph_elem list

type stmt =
  Assign of expr * expr
| Block of typdname list * stmt list
| Expr of expr
| Return of expr
| If of expr * stmt * stmt
| For of typ * string * patt * expr * stmt
| While of expr * stmt
| GraphDef of string * graph_stmt list

type node_decl = {
  nname : string;
  nidpatt : string;
  npatt : patt;
}

type graph_decl = {
  gname : string;
  gbody : graph_stmt list;
}

type func_decl = {
  fname : string;
  freturns : typ;
  formals : typdname list;
  flocals : typdname list;
  fbody : stmt list;
}

type program = node_decl list * graph_decl list * func_decl list

let string_of_unop = function
  Neg -> "-"
| Not -> "!"

let string_of_biop = function
  Add -> "+"
| Sub -> "-"
| Mul -> "*"
| Div -> "/"
| Eq -> "=="
| Neq -> "!="
| Lt -> "<"
| Leq -> "<="
| Gt -> ">"
| Geq -> ">="
| And -> "and"
| Or -> "or"

let rec string_of_typ = function
  TEdge -> "edge"
| TNode -> "node"
| TNNode(s) -> "node:"^ s
| TGraph -> "graph"
| TInt -> "int"
| TBool -> "bool"
| TStr -> "string"
| TList(t) -> string_of_typ t ^" list"
| TVoid -> "void"

module Typ =
  struct
    type t = typ
    let compare t1 t2 =
      String.compare (string_of_typ t1) (string_of_typ t2)
    let equal t1 t2 = compare t1 t2 = 0
  end

let string_of_gelem = function
  NodeId(s) -> s
| EdgeId(n1, e, n2) -> n1 ^" "^ e ^"-> "^ n2

let rec string_of_expr = function
  IntLit(i) -> string_of_int i
| BoolLit(true) -> "true"
| BoolLit(false) -> "false"
| StrLit(s) -> "\""^ String.escaped s ^"\""
| Lst(t, el) -> string_of_typ t ^" ["^ String.concat ", " (List.map string_of_expr el) ^"]"
| Id(s) -> s
| Unop(o, e) -> "("^ string_of_unop o ^" "^ string_of_expr e ^")"
| Biop(e1, o, e2) -> "("^ string_of_expr e1 ^" "^ string_of_biop o ^" "^ string_of_expr e2 ^")"
| Call(f, el) ->
    f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^")"
(*| GraphScope(e, s) -> string_of_expr e ^"::"^ s
| ElemProp(s1, s2) -> s1 ^"."^ s2 *)
| GraphAccess(s, ge) -> s ^":("^ string_of_gelem ge ^")"
| Property(e, s) -> string_of_expr e ^"."^ s
(* | Property(e1, e2) -> string_of_expr e1 ^"["^ string_of_expr e2 ^"]" *)
| Nil(t) -> "NIL("^ string_of_typ t ^")"
| Inf -> ""

let string_of_propassn (s, i) = s ^" = "^ string_of_int i

let string_of_patt pat =  (* This currently doesn't display right arrows ("->") for edges. *)
  String.concat " " (List.rev pat.pids) ^" where "^ String.concat ", " (List.map string_of_propassn pat.preds) 
(* NOTE: string_of_patt uses List.rev before printing the pids because we effectively store them in "reverse order" as a result of parsing. *)

let string_of_typdname (typ, id) = string_of_typ typ ^" "^ id

let string_of_typdname_list tnlist =
  (if List.length tnlist > 0 then (String.concat "; " (List.map string_of_typdname tnlist)) ^";" else "") ^"\n"

let string_of_gstmt = function  (* NOTE: This currently doesn't check if the below lists (e.g. `sil`) are empty *)
  GraphSet(gel, sil) -> String.concat ", " (List.map string_of_gelem gel) ^" where "^ String.concat ", " (List.map string_of_propassn sil)
| GraphDel(gel) -> String.concat ", " (List.map string_of_gelem gel)

(* NOTE: Update (all) pretty-printing to nicely tab/indent lines, as is done with with the generated code. "Time permitting." *)
let rec string_of_stmt = function
  Assign(e1, e2) -> string_of_expr e1 ^" = "^ string_of_expr e2 ^";\n"
| Block(tnl, sl) ->
    "{\n"^ string_of_typdname_list tnl ^
    String.concat "" (List.map string_of_stmt sl) ^"}\n"
| Expr(expr) -> string_of_expr expr ^ ";\n";
| Return(expr) -> "return "^ string_of_expr expr ^";"
| If(e, s, Block([], [])) -> "if "^ string_of_expr e ^" {\n"^ string_of_stmt s ^"\n}\n"
| If(e, s1, s2) -> "if "^ string_of_expr e ^" {\n"^ string_of_stmt s1 ^"\n} else {\n"^ string_of_stmt s2 ^"\n}\n"
| For(t, str, pat, e, stat) ->
    "for "^ string_of_typ t ^" "^ str ^" in "^ string_of_patt pat ^" in "^ string_of_expr e ^" {\n"^ string_of_stmt stat ^"\n}\n"
| While(e, s) -> "while "^ string_of_expr e ^" {\n"^ string_of_stmt s ^"\n}\n"
| GraphDef(s, gsl) -> s ^" {\n"^ String.concat "\n" (List.map string_of_gstmt gsl) ^"\n}\n"

let string_of_formals formals = String.concat ", " (List.map string_of_typdname formals)


let string_of_node_decl node_decl =
  "node "^ node_decl.nname ^" = "^ node_decl.nidpatt ^" in "^ string_of_patt node_decl.npatt ^";"

let string_of_graph_decl graph_decl = 
  "graph "^ graph_decl.gname ^" {\n"^ 
  (if List.length graph_decl.gbody > 0 then "  "^ String.concat "\n  " (List.map string_of_gstmt graph_decl.gbody)
   else "") ^"\n}\n"

let string_of_func_decl f_decl =
  "func "^ f_decl.fname ^"("^ string_of_formals f_decl.formals ^") "^
  if f_decl.freturns <> TVoid then "return "^ string_of_typ f_decl.freturns else "" ^
  " {\n"^ string_of_typdname_list f_decl.flocals ^ 
  (if List.length f_decl.fbody > 0 then "  "^ String.concat "\n  " (List.map string_of_stmt f_decl.fbody)
   else "") ^"\n}\n"

let string_of_program (nodes, graphs, funcs) =
  (if List.length nodes > 0 then String.concat "\n\n" (List.map string_of_node_decl (List.rev nodes)) ^ "\n\n"
  else "") ^
  (if List.length graphs > 0 then String.concat "\n\n" (List.map string_of_graph_decl (List.rev graphs)) ^ "\n\n"
  else "") ^
  (if List.length funcs > 0 then String.concat "\n\n" (List.map string_of_func_decl (List.rev funcs)) ^ "\n\n"
  else "") ^ "\n"

type symbol_table = {
    parent : symbol_table option;
    variables : typdname list;
}

type translation_environment = {
    scope: symbol_table;
    return_type : typ;
}
