(* Ocamllex scanner for tail *)

{ 
  open Parser
  let unescape s = Scanf.sscanf("\"" ^ s ^ "\"") "%S%!" (fun x -> x) 
}

let digit = ['-']*['0'-'9']
let lowercase = ['a'-'z']
let uppercase = ['A'-'Z']
let letter = (lowercase | uppercase)

rule token = parse
  [' ' '\t' '\n' '\r' ]	{token lexbuf}
(* Data declarations *)
| "(*"		{ comment lexbuf }
| "int"		{ INT }
| "string"      { STRING }
| "char"        { CHAR }
| "float"       { FLOAT }
| "bool"        { BOOL }
(* Keywords *)
| "let"		{ LET }
| "if"		{ IF }
| "then"	{ THEN }
| "else"	{ ELSE }
| "fun"         { FUN }
| "in"          { IN    }
| "match"	{ MATCH }
| "with"	{ WITH }
| "rec"		{ REC }
(* Int Operators *)
| '+'		{ PLUS }
| '-'		{ MINUS }
| '*'		{ MULT	}
| '/'		{ DIV	}
| '%'		{ MOD }
| "=="		{ EQ }
| "!="		{ NEQ }
| "&&"		{ LOG_AND}
| "||"		{ OR }
| "not"		{ NOT }
| '>'		{ GREATER }
| '<'		{ LESS }
| "<="		{ LEQ }
| ">="		{ GEQ }
(* Other symbols *)
| '_'		{ WILDCARD }
| '='		{ ASSIGN }
| '('		{ LPAREN }
| ')'		{ RPAREN }
| '['		{ LBRACKET }
| ']'		{ RBRACKET }
| ','           { COMMA }
| ":"		{ COLON }
| "::"		{ COLONS }
| ';'           { SEMICOLON }
| "->"		{ RARROW }
| "|"		{ BAR }
(* Literals and identifiers *)
| digit+ as int_lit { INT_LITERAL(int_of_string int_lit)}
| digit+ '.' digit* as float_lit { FLOAT_LITERAL(float_of_string float_lit) }
| '\'' (digit | letter | '_') '\'' as char_lit { CHAR_LITERAL(int_of_string char_lit) }
| '"' (('\\' '"'| [^'"'])* as str) '"' { STRING_LITERAL(unescape str) } 
| ("true" | "false") as bool_lit { BOOL_LITERAL(bool_of_string bool_lit)}
| letter (letter | digit | '_')* as lxm { ID(lxm) }
| eof		{ EOF }

and comment = parse
  	| "*)" 	{ token lexbuf }
	| _ 	{ comment lexbuf }

