open Ast
open Sast

module StringMap = Map.Make(String)

(* Record that stores a variable's datatype (Int_Decl, Char_Decl, etc), 
name (given by the programmer), and value assigned *)
type var_struct = {
	data_type: datatype;
	name: string;
	value: sexpr
}

(* Record that stores a functions *)
type fun_struct = {
	return_typ : datatype; (* Return type of the function *)
	func_typ   : func_type; (* Iterative or recursive *)
	func_name  : string; (* Function's name *)
	formals	   : sformal list; (* Function's formal parameters, given by programmer *)
	expr_list  : sexpr list; (* list of expressions defining a function *)
}

(* a "type template" that stores 1) a map of the program's global variables.
The hashmap maps the variable's name to its var_struct (see line 8 for var_struct)
2) stores a map of the program's functions. Maps the function's name to its 
fun_struct (see line 15 for fun_struct). 3) stores a list of the expressions a
user chose to evaluate, but not assign*)
type environment = {
	vars: var_struct StringMap.t;
	funs: fun_struct StringMap.t;
	expressions: sexpr list;
}

(* "Instantiating" the type template environment *)
type real_environment = {
	vars: var_struct StringMap.t;
	vars_order: string list;
	funs: fun_struct StringMap.t;
	funs_order: string list;
	expressions: sexpr list;
}
	
let env = {
	vars = StringMap.empty;
	vars_order = [];
	funs = StringMap.empty;
	funs_order = [];
	expressions = [];
}


(* Takes two floats and determines what operation was applied to them.
See evalExpr (match on SBinop) for use *)
let evalBinopFloat f1 o f2 =
	match o with
	| Add -> SFloat_Lit(f1 +. f2)
	| Minus -> SFloat_Lit(f1 -. f2)
	| Mult -> SFloat_Lit(f1 *. f2)
	| Div -> SFloat_Lit(f1 /. f2)
	| Mod -> SFloat_Lit(mod_float f1 f2)
	| Eq -> SBool_Lit(f1 = f2) 
	| Neq -> SBool_Lit(f1 <> f2)
	| Geq -> SBool_Lit(f1 >= f2)
	| Leq -> SBool_Lit(f1 <= f2)
	| Less -> SBool_Lit(f1 < f2)
	| Greater -> SBool_Lit(f1 > f2) 
	| _ -> raise (Exceptions.OperationNotPermitted (" "^(string_of_float f1)^" ,"^(string_of_float f2)^" are floats. "^(string_of_op o)^" cannot be applied to floats."))

(* Takes two ints and determines what operation was applied to them. See evalExpr
(match on SBinop) for use *)
let evalBinopInt i1 o i2 =
	match o with
	| Add -> SInt_Lit(i1 + i2)
	| Minus -> SInt_Lit(i1 - i2)
	| Mult -> SInt_Lit(i1 * i2)
	| Div -> SInt_Lit(i1 / i2)
	| Mod -> SInt_Lit((mod) i1 i2)
	| Eq -> SBool_Lit(i1 = i2) 
	| Neq -> SBool_Lit(i1 <> i2)
	| Geq -> SBool_Lit(i1 >= i2)
	| Leq -> SBool_Lit(i1 <= i2)
	| Less -> SBool_Lit(i1 < i2)
	| Greater -> SBool_Lit(i1 > i2) 
	| _ -> raise (Exceptions.OperationNotPermitted (" "^(string_of_int i1)^" ,"^(string_of_int i2)^" are ints. "^(string_of_op o)^" cannot be applied to ints."))

(* Takes two bools and determins which operation was applied to them. See evalExpr
(match on SBinop) for use*)
let evalBinopBool b1 o b2 =
	match o with
	| Log_And -> SBool_Lit(b1 && b2)
	| Or -> SBool_Lit(b1 || b2)
	| _ -> raise (Exceptions.OperationNotPermitted ((string_of_op o)^" not a legal uniop."))

(* Takes an SBinop node (for binary operation o applied to two arguments e1 and e2)
and determines what type the arguments are to figure out which operators to apply
to them. See evalExpr (match on SBinop) for use *)
let evalBinop e1 o e2 =
	match [e1; e2] with
	| [SInt_Lit(i1); SInt_Lit(i2)] -> evalBinopInt i1 o i2
	| [SFloat_Lit(f1); SFloat_Lit(f2)] -> evalBinopFloat f1 o f2
	| [SBool_Lit(b1); SBool_Lit(b2)] -> evalBinopBool b1 o b2
	| _ -> raise (Exceptions.TypeMismatch (" Type mismatch between arguments 1 and 2 for operator "^(string_of_op o)))

let evalUniop b u =
	match b with
	| SBool_Lit(c) -> 
		(match u with 
		| Not -> SBool_Lit(not c))

	| _ -> raise (Exceptions.OperationNotPermitted " Uniop only permitted on bools.")


let rec evalExpr expression vars funs =
	match expression with
	| SLiteral(l) -> expression
	| SId(var_name, dtype) -> (resolveScope var_name dtype vars funs)
	| SBinop(e1, o, e2, dtype) -> SBinop((evalExpr e1 vars funs), o, (evalExpr e2 vars funs), dtype)
	| SUniop(e, u, dtype) -> SUniop((evalExpr e vars funs),u, dtype)
	| SCall(funcname, sexprList, dtype, _) -> functionCall funcname sexprList vars funs
	| SIf(cond, texpr, fexpr) -> if ((evalExpr cond vars funs) = SLiteral(SBool_Lit(true)))
					then evalExpr texpr vars funs
					else evalExpr fexpr vars funs
	| SEvaluate(sfunDef, se) -> evalE sfunDef se vars funs
	| SAss(sexprDef, se, dtype) -> evalAss sexprDef se vars funs
        | SMatch(pattern, dtype, caseList) -> expression
        (*| SMatch(pattern, dtype, caseList) -> ignore(matchEval pattern dtype caseList vars funs); expression*)
	| _ -> raise (Exceptions.DatatypeNotPermitted)

and matchEval pattern dtype caseList vars funs =
	let matchCase l case = match case with
		| SMatchCase(p, dtype1, e, dtype2) -> 
                            (if ( (evalExpr pattern vars funs) = (evalExpr p vars funs)) 
                            then e::l 
                            else l)
	in 
	let cl = (List.rev caseList) in
	let evalMatches = List.fold_left matchCase [] cl in
        if ((List.length evalMatches) = 0) then raise (Exceptions.OperationNotPermitted "Don't fucking do this.") else
	evalExpr (List.hd evalMatches) vars funs

and evalAss sexprDef se vars funs = 
	match sexprDef with
	| SExpr_Def(declTyp, n, v) -> 
		(let mymap = StringMap.add n 
			{	data_type = declTyp;
				name = n;
				value = evalExpr v vars funs; } vars 
			in (evalExpr se mymap funs))
	| _ -> raise (Exceptions.DatatypeNotPermitted)

and resolveScope id dtype vars funs=
	if (StringMap.mem id vars) then
		let v_struct = StringMap.find id vars in
		if (dtype = v_struct.data_type) then (evalExpr v_struct.value vars funs)
		else raise (Exceptions.TypeMismatch " declaration of datatype and stored don't match")
	else raise (Exceptions.OperationNotPermitted (id^" not in scope or undefined."))


and evalExprList eList local_vars local_funs =
	let derp = List.fold_left (fun struc e -> let ex = evalExpr e struc.vars struc.funs in 
				{ vars = struc.vars; vars_order = []; funs = struc.funs; funs_order = []; expressions = ex::struc.expressions}  )
				{ vars = local_vars; vars_order = []; funs = local_funs; funs_order = []; expressions = [] } eList 
	in (List.hd (derp.expressions))


and evalE funDef e vars funs =
	match funDef with
	| SFun_Def(sfuncDecl, sexprList) -> let myFunMap = StringMap.add sfuncDecl.sfunc_name {
						return_typ = sfuncDecl.sreturn_typ;
						func_typ = sfuncDecl.sfunc_typ;
						func_name = sfuncDecl.sfunc_name;
						formals = sfuncDecl.sformals;
						expr_list = sexprList; } funs in (evalExpr e vars myFunMap)

and functionCall funcname actualsList vars funs =
	if (StringMap.mem funcname funs) then 
		let funny = StringMap.find funcname funs 
		in
		let newfuns = 
			if (funny.func_typ == Rec) then StringMap.add funny.func_name funny funs else funs 
		in 
		let tempfun v a b = match b with
			| SFormal(p, id) -> StringMap.add id { data_type = p; name = id; value = a; } v
		in 
		let newVars = List.fold_left2 tempfun vars actualsList funny.formals in
		let s = evalExprList funny.expr_list newVars newfuns in s
	else raise (Exceptions.OperationNotPermitted ("\""^funcname^"\" function out of scope or undefined"))

(* Main method for analyzer *)
and analyzer sprogram globalVars globalFuns globalEx v_order f_order= 
	let evalProgram l s = match s with
		| SAss_Stmt(SExpr_Def(declTyp, n, v)) -> let n_globalVars = StringMap.add n {
				data_type = declTyp;
				name = n;
				value = evalExpr v l.vars l.funs; } l.vars in 
			let n_vorder = n::l.vars_order in
			{ 	vars = n_globalVars; 
				vars_order = n_vorder; 
				funs = l.funs; 
				funs_order = l.funs_order;
				expressions = l.expressions; }
		| SFun_Def_Stmt(SFun_Def(sfuncDecl, sexprList)) -> 
			let n_globalFuns = StringMap.add sfuncDecl.sfunc_name {
				return_typ = sfuncDecl.sreturn_typ;
				func_typ = sfuncDecl.sfunc_typ;
				func_name = sfuncDecl.sfunc_name;
				formals = sfuncDecl.sformals;
				expr_list = sexprList; } l.funs
			in let n_forder = sfuncDecl.sfunc_name::l.funs_order
			in { 	vars = l.vars; 
				vars_order = l.vars_order;
				funs = n_globalFuns; 
				funs_order = n_forder;
				expressions = l.expressions; }
		| SExpr_Stmt(sexpr) -> let n_globalEx = sexpr::l.expressions in 
			{ 	vars = l.vars; 
				vars_order = l.vars_order;
				funs = l.funs; 
				funs_order = l.funs_order;
				expressions = n_globalEx; }
	in List.fold_left evalProgram { vars = globalVars; 
					vars_order = v_order;
					funs = globalFuns; 
					funs_order = f_order;
					expressions = globalEx; } sprogram

let analyze sprogram = match sprogram with
        Sast.SProgram(sp) -> analyzer sp env.vars env.funs env.expressions env.vars_order env.funs_order

