(* Abstract Syntax Tree and functions for printing it *)

type primitive = Int_Decl | Float_Decl | Char_Decl | String_Decl | Bool_Decl | Int_List | Float_List | Char_List | String_List | Bool_List | Unit
type op = Add | Minus | Mult | Div | Mod | Eq | Neq | Geq | Leq | Less | Greater | Log_And | Or | FAdd | FMinus | FMult | FDiv | FGeq | FLeq | FLess | FGreater
type uniop = Not
type func_type = Iter | Rec
(* type boolean = true | false *)

type formal = Formal of primitive * string

type fun_decl = {
	return_typ : primitive;
	func_typ   : func_type;
	func_name  : string;
	formals	   : formal list;
}

and expr =
    | Literal of literal
    | Expr of expr
    | Id of string
    | Binop of expr * op * expr
    | Uniop of expr * uniop
    | Call of string * expr list
    | If of expr * expr * expr
    | Evaluate of fun_def * expr
    | Ass of expr_def * expr
    | Match of expr * match_case list
     
and literal =
    | Int_Lit of int
    | Float_Lit of float
    | Char_Lit of int
    | String_Lit of string
    | Bool_Lit of bool
    | Unit
 
and fun_def = Fun_Def of fun_decl * expr list

and expr_def = Expr_Def of primitive * string * expr
	       | List_Def of primitive * string * list_substance

and list_substance = 
    NewList of expr list
    | Prepend of string * string

and match_case = MatchCase of expr * expr

and stmt = 
	| Fun_Def_Stmt of fun_def
	| Ass_Stmt of expr_def
	| Expr_Stmt of expr

type program = Program of stmt list

(* Pretty-printing functions *)
let string_of_primitive = function
	| Int_Decl -> "int" 
	| Float_Decl -> "float" 
	| Char_Decl -> "char"  
	| String_Decl -> "string"  
	| Bool_Decl -> "bool"
	| _ -> "list"

let string_of_op = function
	| Add -> "+" 
	| Minus -> "-" 
	| Mult -> "*" 
	| Div -> "/"
	| Mod -> "%"
	| Eq -> "==" 
	| Neq -> "!=" 
	| Geq -> ">=" 
	| Leq -> "<=" 
	| Less -> "<" 
	| Greater -> ">" 
	| Log_And -> "&&" 
	| Or -> "||" 
	| _ -> "error string_of_op"

let string_of_uniop = function
	| Not -> "!"

let string_of_func_type = function
	| Iter -> ""
 	| Rec -> "rec"

let string_of_formal = function
	| Formal(p, s) -> string_of_primitive(p)^" "^s

let string_of_formals_list fl = List.fold_left (fun s f -> s^(string_of_formal f)^", ") "" fl

let string_of_fun_decl fd = 
	(string_of_primitive fd.return_typ)^" "^(string_of_func_type fd.func_typ)^" "^fd.func_name^"("^(string_of_formals_list fd.formals)^")"
		

let rec string_of_expr = function
    Literal(l) -> string_of_lit l
    | Id(i) -> i
    | Binop(e1, o, e2) -> (string_of_expr e1)^(string_of_op o)^(string_of_expr e2)
    | Uniop(e, u) -> (string_of_uniop u)^(string_of_expr e)
    | Match(e, ml) -> "Match("^(string_of_expr e)^","^(string_of_match_list ml)^")"
    | Call(s, el) -> "Call("^s^"("^(string_of_expr_list el)^")"^")"
    | If(e1, e2, e3) -> "if("^(string_of_expr e1)^") then "^(string_of_expr e2)^" else "^string_of_expr e3
    | Evaluate(fundef, exp) -> "Evaluate("^(string_of_fun_def fundef)^" \n\tin "^(string_of_expr exp)^")"
    | Ass(ed, e) -> "Ass("^(string_of_expr_def ed)^","^(string_of_expr e)^")"
    | _ -> "hey"

and string_of_lit = function
    Int_Lit(i) -> string_of_int i
    | Float_Lit(f) -> string_of_float f
    | Char_Lit(c) -> string_of_int c
    | String_Lit(s) -> s
    | Bool_Lit(b) -> string_of_bool b
    | Unit -> "()"

and string_of_fun_def = function
	| Fun_Def(fd, el) -> (string_of_fun_decl fd)^" = fun\n\t"^(string_of_expr_list el)

and string_of_match_list ml = List.fold_left (fun str form -> str^(string_of_match_case form)^";") "" ml

and string_of_match_case = function
	| MatchCase(e1, e2) -> "MatchCase("^(string_of_expr e1)^","^(string_of_expr e2)^")"

and string_of_expr_def = function
	| Expr_Def(p, s, e) -> "Expr_Def("^(string_of_primitive p)^" "^s^" = "^(string_of_expr e)^")"
	| _ -> "yo"

and string_of_stmt = function
	| Fun_Def_Stmt(fd) -> "Fun_Def_Stmt(\n\t"^(string_of_fun_def fd)^"\n)"
	| Ass_Stmt(ed) -> "Ass_Stmt(\n\t"^(string_of_expr_def ed)^"\n)"
	| Expr_Stmt(e) -> "Expr_Stmt(\n\t"^(string_of_expr e)^"\n)"

and string_of_expr_list el = List.fold_left (fun str form -> str^(string_of_expr form)^";\n\t") "" el

let string_of_program sl = match sl with
	Program(stmt_list) ->
	let result = List.fold_left (fun str form -> str^(string_of_stmt form)^";\n") "" stmt_list
	in result^"\n"
