open Ast
open Sast

module StringMap = Map.Make(String)

let rec expr_to_sexpr expr vars = match expr with
	(* exprs *)
	Ast.Literal(i) -> check_literal i vars
	| Ast.Id(s) -> vars, Sast.SId(s, get_id_type s vars)
	| Ast.Expr(e) -> expr_to_sexpr e vars
	| Ast.Binop(e1, op, e2) -> check_binop e1 op e2 vars
	| Ast.Uniop(e, op) -> check_uniop e op vars
	| Ast.Call(s, e) -> check_call s e vars
	(* defs *)
	| Ast.Evaluate(fd, e) -> check_eval fd e vars
	| Ast.Ass(ed, e) -> check_assign ed e vars
	(* conds *)
	| Ast.If(e1, e2, e3) -> check_if e1 e2 e3 vars
	(* match_expr *)
	| Ast.Match(e, ml) -> check_match e ml vars

and check_prepend s1 s2 vars =
	let d1 = get_id_type s1 vars in
	let d2 = get_id_type s2 vars in
	if (get_type_equivalence d1 d2) then 
		SPrepend(s1,s2)
	else raise (Exceptions.InvalidPrepend "Must preprend lists of same type")

and check_match e ml vars = 
	let vars', s = expr_to_sexpr e vars in
	let d = get_type_from_sexpr s in
	matchl_to_smatchl s d ml vars'

and matchl_to_smatchl s d ml vars = 
	let rec helper vars smatchl = function
		| [] -> vars, Sast.SMatch(s,d, smatchl)
		| mat :: tl -> let vars', smatch = match_to_smatch d mat vars in
			helper vars' (smatch :: smatchl) tl
	in helper vars [] ml
(*
	let rec create_vars_list vl c =
		if c >= (List.length ml) then vl
		else create_vars_list (List.append vl [vars]) (c+1)
	in 
	let vars_list = create_vars_list [] 0 in 
	let sml = List.map2 (match_to_smatch d) ml vars_list in
	Sast.SMatch(s,d,sml)
*)

and match_to_smatch d ml vars = match ml with
	MatchCase(e1,e2) ->
		let vars1, s1 = expr_to_sexpr e1 vars in
		let vars2, s2 = expr_to_sexpr e2 vars1 in
		let d1 = get_type_from_sexpr s1 in
		let d2 = get_type_from_sexpr s2 in
		if (get_type_equivalence d d1) then vars2, SMatchCase(s1,d1,s2,d2)
		else raise (Exceptions.InvalidMatch "Must match between matching types")

and check_assign ed e1 vars = match ed with 
	Ast.Expr_Def(t, s, e2) ->
		let d = Datatype(t) in 
		let vars' = StringMap.add s d vars in
		let vars1, s1 = expr_to_sexpr e1 vars' in
		let t1 = get_type_from_sexpr s1 in
		vars', Sast.SAss((check_expr_assign t s e2 vars'),s1,t1) 
	| Ast.List_Def(t, s, l) -> match l with
		NewList(nl) ->
			let vars1, s1 = expr_to_sexpr e1 vars in
			let t1 = get_type_from_sexpr s1 in
			vars1, Sast.SAss((check_list_assign t s nl vars), s1, t1)
		| Prepend(s1,s2) -> raise (Exceptions.InvalidBinopExpr "i")

and check_list_assign t s l vars = 
	let d1 = get_list_dt t in
	let vars1 = StringMap.add s d1 vars in
	let vars', sl = expr_list_to_sexpr_list l vars in
	let d2 = get_list_type sl in
	get_list_assign_type s sl vars1 (d1,d2)

and get_list_dt t = match t with
	Int_Decl|Int_List -> Datatype(Int_List)
	| Float_Decl|Float_List -> Datatype(Float_List)
	| Char_Decl|Char_List -> Datatype(Char_List)
	| String_Decl|String_List -> Datatype(String_List)
	| Bool_Decl|Bool_List -> Datatype(Bool_List)
	| Unit -> raise (Exceptions.IllegalListType "Cannot have a void list")

and get_type_equivalence t1 t2 = match (t1,t2) with
	(Datatype(Int_Decl),Datatype(Int_Decl)) -> true
	| (Datatype(Float_Decl),Datatype(Float_Decl)) -> true
	| (Datatype(Char_Decl),Datatype(Char_Decl)) -> true
	| (Datatype(String_Decl),Datatype(String_Decl)) -> true
	| (Datatype(Bool_Decl),Datatype(Bool_Decl)) -> true
	| (Special(Wildcard("_")), _) -> true
	| (_, Special(Wildcard("_"))) -> true
	| _ -> false


and get_list_type sl = 
	let hed = (List.hd sl) in
	let typ = get_type_from_sexpr hed in
	let type_check s = 
		let t = get_type_from_sexpr s in
		if (get_type_equivalence typ t) then typ
		else raise (Exceptions.InvalidAssignment "Assignments must be made between matching types")
	in 	
	List.hd (List.map type_check sl)
	

and get_list_assign_type s sl vars = function
	(Datatype(Int_Decl),Datatype(Int_Decl)) ->  Sast.SList_Def(Datatype(Int_Decl),s,SNewList(sl))
	| (Datatype(Float_Decl),Datatype(Float_Decl)) ->  Sast.SList_Def(Datatype(Float_Decl),s,SNewList(sl))
	| (Datatype(Char_Decl),Datatype(Char_Decl)) ->  Sast.SList_Def(Datatype(Char_Decl),s,SNewList(sl))
	| (Datatype(String_Decl),Datatype(String_Decl)) ->  Sast.SList_Def(Datatype(String_Decl),s,SNewList(sl))
	| (Datatype(Bool_Decl),Datatype(Bool_Decl)) ->  Sast.SList_Def(Datatype(Bool_Decl),s,SNewList(sl))
	| _ -> raise(Exceptions.InvalidAssignment "Assigments must be made between matching types")	

and check_expr_assign t s e vars = 
	let d1 = Sast.Datatype(t) in
	let vars', se = expr_to_sexpr e vars in
	let d2 = get_type_from_sexpr se in
	get_assign_type s se vars' (d1, d2)

and get_assign_type s se vars = function
	(Datatype(Int_Decl),Datatype(Int_Decl)) ->  Sast.SExpr_Def(Datatype(Int_Decl),s,se)
	| (Datatype(Float_Decl),Datatype(Float_Decl)) ->  Sast.SExpr_Def(Datatype(Float_Decl),s,se)
	| (Datatype(Char_Decl),Datatype(Char_Decl)) ->  Sast.SExpr_Def(Datatype(Char_Decl),s,se)
	| (Datatype(String_Decl),Datatype(String_Decl)) -> Sast.SExpr_Def(Datatype(String_Decl),s,se)
	| (Datatype(Bool_Decl),Datatype(Bool_Decl)) ->  Sast.SExpr_Def(Datatype(Bool_Decl),s,se)
	| _ -> raise(Exceptions.InvalidAssignment "Assigments must be made between matching types")

and check_eval fd e vars = 
	let vars1, sfd = fd_to_sfd fd vars in
	let vars', s = expr_to_sexpr e vars1 in
	vars', Sast.SEvaluate(sfd, s)

and check_fun_rt f_dec el vars = 
	let ftyp = Datatype(f_dec.Ast.return_typ) in
	let rev_el = List.rev el in
	let final_e = (List.hd rev_el) in
	let vars', s = expr_to_sexpr final_e vars in
	let typ = get_type_from_sexpr s in
	if (get_type_equivalence ftyp typ) then ()
	else raise (Exceptions.IncorrectReturn "Function return type does not match type returned")


and fd_to_sfd fd vars = match fd with
	Ast.Fun_Def(f_dec, el) -> 
	let vars', vars_fun, sf_dec = f_dec_to_sf_dec f_dec vars in 
	match f_dec.Ast.func_typ with
		Rec -> 	let vars2, sel = expr_list_to_sexpr_list el vars_fun in
			vars', Sast.SFun_Def(sf_dec, sel)
		| Iter -> check_fun_rt f_dec el vars_fun;
		let vars3, sel = expr_list_to_sexpr_list el vars_fun in
				vars', Sast.SFun_Def(sf_dec, sel)

and f_dec_to_sf_dec f_dec vars = 
	let fname = f_dec.Ast.func_name in
	let rtype = Sast.Datatype(f_dec.Ast.return_typ) in
	let vars' = StringMap.add  fname rtype vars in
	let sf_dec =
	{
		sreturn_typ = Sast.Datatype(f_dec.Ast.return_typ);
		sfunc_typ = f_dec.Ast.func_typ;
		sfunc_name = f_dec.Ast.func_name;
		sformals = (formals_to_sformals f_dec.Ast.formals);
	} in
	let vars_fun = add_parameters vars' sf_dec.sfunc_typ  fname sf_dec.sformals in 
	vars', vars_fun, sf_dec 

and add_parameters vars fun_typ fname sformals = 
	let rec helper_rec var_fun = function
		| [] -> var_fun
		| SFormal(d,s) :: tl -> 
		let vars' = StringMap.add s d var_fun
		in helper_rec vars' tl
	in helper_rec vars sformals

and formals_to_sformals formals = 
	List.map formal_to_sformal formals
	
and formal_to_sformal = function
	Ast.Formal(p,s) -> Sast.SFormal(Sast.Datatype(p), s)

and check_if e1 e2 e3 vars = 
	let vars1', s1 = expr_to_sexpr e1 vars in
	let vars2', s2 = expr_to_sexpr e2 vars1' in
	let vars3', s3 = expr_to_sexpr e3 vars1' in
	let d = check_if_bool s1 (*get_type_from_sexpr s1*) in
	get_if_type s1 s2 s3 d vars1'

and get_if_type s1 s2 s3 d vars = match d with
	Datatype(Bool_Decl) -> vars, Sast.SIf(s1, s2, s3)
	| _ -> raise (Exceptions.InvalidIfConditional "If conditional can only operate on a boolean datatype 3!.")

and expr_list_to_sexpr_list el vars =
	let rec helper vars sexprs n_el = match n_el with
		| [] -> vars, sexprs
		| expr :: tl -> let vars', sexpr = expr_to_sexpr expr vars in
			helper vars' (sexpr :: sexprs) tl
	in helper vars [] el

and check_literal literal vars = match literal with
	Ast.Int_Lit i -> vars, Sast.SLiteral(SInt_Lit(i))
	| Ast.Float_Lit f -> vars, Sast.SLiteral(SFloat_Lit(f))
	| Ast.String_Lit s -> vars, Sast.SLiteral(SString_Lit(s))
	| Ast.Char_Lit c -> vars, Sast.SLiteral(SChar_Lit(c))
	| Ast.Bool_Lit b -> vars, Sast.SLiteral(SBool_Lit(b))
	| Ast.Unit -> vars, Sast.SLiteral(SUnit)

and check_call s e vars =
	let vars', sexprs = expr_list_to_sexpr_list e vars in
	let d = get_id_type s vars in
	let d1 = match s with 
	| "return" -> get_type_from_sexpr (List.hd sexprs) 
	| _ -> d
	in
	vars', Sast.SCall(s, sexprs, d1, (List.length sexprs))

and get_id_type s vars = match s with
	"print_int"|"print_char"|"print_float"|"print_string"|"print_bool"
 -> Datatype(Unit)
	| "return" -> Datatype(Unit)
	| _ ->	try StringMap.find s vars
	with | Not_found -> raise (Exceptions.VariableNotFound (s^" variable could not be found"))

and get_type_from_sexpr sexpr = match sexpr with
	Sast.SLiteral(SInt_Lit(_)) -> Sast.Datatype(Int_Decl)
	| Sast.SLiteral(SFloat_Lit(_) )-> Sast.Datatype(Float_Decl)
	| Sast.SLiteral(SChar_Lit(_)) -> Sast.Datatype(Char_Decl)
	| Sast.SLiteral(SString_Lit("_")) -> Sast.Special(Wildcard("_"))
	| Sast.SLiteral(SString_Lit(_)) -> Sast.Datatype(String_Decl)
	| Sast.SLiteral(SBool_Lit(_)) ->	Sast.Datatype(Bool_Decl)
	| Sast.SLiteral(SUnit) -> Sast.Datatype(Unit)
	| Sast.SId(_,d) -> d
	| Sast.SBinop(_,_,_,d) -> d
	| Sast.SUniop(_,_,d) -> d
	| Sast.SCall(_,_,d,_) ->d
	| Sast.SEvaluate(SFun_Def(sfd,_),_) -> sfd.sreturn_typ
	| Sast.SAss(_,_,d) -> d
	| Sast.SMatch(_,d,_) -> d
	| Sast.SIf(e,_,_) -> check_if_bool e

and check_if_bool e = match e with
	|Sast.SLiteral(SBool_Lit(_)) -> Sast.Datatype(Bool_Decl)
	|Sast.SBinop(_,op,_,_) -> check_binop_bool op
	| Sast.SUniop(_,op,_) -> Sast.Datatype(Bool_Decl)
	| _ -> raise (Exceptions.InvalidIfConditional "If conditional can only operate on a boolean datatype 2!.")

and check_binop_bool = function
	Eq | Neq | Geq | Leq | Less | Greater | Log_And | Or -> Sast.Datatype(Bool_Decl)
	| _ -> raise (Exceptions.InvalidIfConditional "If conditional can only operate on a boolean datatype 1!.")


and check_uniop e op vars = 
	let vars', sexpr = expr_to_sexpr e vars in
	let t = get_type_from_sexpr sexpr in
        match op with
	Ast.Not -> get_uniop_type sexpr op vars' t

and get_uniop_type s o vars = function
	Datatype(Bool_Decl) -> vars, Sast.SUniop(s, o, Datatype(Bool_Decl))
	| _ ->	raise (Exceptions.InvalidBinopExpr "Wrong uniop")

and check_binop e1 op e2 vars = 
	let vars1, sexpr1 = expr_to_sexpr e1 vars in
	let vars2, sexpr2 = expr_to_sexpr e2 vars in
	let t1 = get_type_from_sexpr sexpr1 in
	let t2 = get_type_from_sexpr sexpr2 in
	match op with
	Add | Minus | Mult | Div | Mod -> get_math_type sexpr1 op sexpr2 vars (t1, t2)
	| Eq | Neq | Geq | Leq | Less | Greater -> get_comparison_type sexpr1 op sexpr2 vars (t1, t2)
	| Log_And | Or -> get_logical_type sexpr1 op sexpr2 vars (t1, t2)
	| _ -> raise (Exceptions.InvalidBinopExpr "binop not supported")

and get_math_type sexpr1 op sexpr2 vars = function
	(Datatype(Int_Decl), Datatype(Int_Decl)) -> vars, Sast.SBinop(sexpr1, op, sexpr2, Datatype(Int_Decl))
	| (Datatype(Float_Decl), Datatype(Float_Decl)) -> vars, Sast.SBinop(sexpr1,op,sexpr2,Datatype(Float_Decl))
	| _ -> raise (Exceptions.InvalidBinopExpr "Cannot perform math operations on different datatypes")

and get_comparison_type sexpr1 op sexpr2 vars = function
	(Datatype(Int_Decl), Datatype(Int_Decl)) -> vars, Sast.SBinop(sexpr1, op, sexpr2, Datatype(Int_Decl))
	| (Datatype(Float_Decl), Datatype(Float_Decl)) -> vars, Sast.SBinop(sexpr1,op,sexpr2,Datatype(Float_Decl))
	| _ -> raise(Exceptions.InvalidBinopExpr "Cannot peform comparison on different datatypes")

and get_logical_type sexpr1 op sexpr2 vars = function
	(Datatype(Bool_Decl), Datatype(Bool_Decl)) -> vars, Sast.SBinop(sexpr1,op,sexpr2,Datatype(Bool_Decl))
	| (_,_) -> raise(Exceptions.InvalidBinopExpr "type mismatch")

and check_expr_def t s e vars = 
	let d1 = Datatype(t) in
	let vars1 = StringMap.add s d1 vars in
	let vars2, sexpr = expr_to_sexpr e vars1 in
	let d2 = get_type_from_sexpr sexpr in
	if (get_type_equivalence d1 d2) then vars2, Sast.SExpr_Def(d1,s,sexpr)
	else raise  (Exceptions.InvalidAssignment "Variable assignment attempted between mismatched types")

and check_list_def t s el vars = 
	let d1 = Datatype(t) in 
	let vars1 = StringMap.add s d1  vars in match el with
	NewList(el) -> 
		let vars', sl = expr_list_to_sexpr_list el vars in
		let d2 = get_list_type sl in
		if (get_type_equivalence d1 d2) then vars1, Sast.SList_Def(d1,s,SNewList(sl))
		else raise  (Exceptions.InvalidAssignment "Variable list assignment attempted between mismatched types")
	| Prepend(s1,s2) -> vars1, Sast.SList_Def(d1, s,  check_prepend s1 s2 vars)

and expr_def_to_sexpr_def e vars = match e with
	Ast.Expr_Def(t,s,ex) -> check_expr_def t s ex vars
	| Ast.List_Def(t,s,el) -> check_list_def t s el vars

let convert_stmt_to_sstmt (stmt:Ast.stmt) vars = match stmt with 
	Ast.Fun_Def_Stmt(e) -> 
		let vars', sfun_def = fd_to_sfd e vars
		in vars', Sast.SFun_Def_Stmt(sfun_def)
	| Ast.Ass_Stmt (e)  ->
		let vars', sexpr_def =  expr_def_to_sexpr_def e vars
		in vars', Sast.SAss_Stmt(sexpr_def)
	| Ast.Expr_Stmt(e)  -> 
		let vars', sexpr = expr_to_sexpr e vars 
		in vars', SExpr_Stmt(sexpr)

let convert_stmts_to_sstmts (stmt_list:Ast.stmt list)  =
	let vars_begin = StringMap.empty in
	let rec helper vars sstmts = function
		| [] -> List.rev sstmts
		| stmt :: tl ->
			let vars', sstmt = convert_stmt_to_sstmt stmt vars in
			helper vars' (sstmt :: sstmts) tl
	in helper vars_begin [] stmt_list 
(*	let sstmt_list = List.map convert_stmt_to_sstmt stmt_list 
	in 
	sstmt_list
*)

let analyze program = match program with
    Ast.Program(stmt_list) ->
(*	let vars = StringMap.empty in*)
        let sast = convert_stmts_to_sstmts stmt_list in
	Sast.SProgram(sast)
