(* Top-level of the MicroC compiler: scan & parse the input,L
  check the resulting AST, generate LLVM IR, and dump the module *)

type action = Ast | Ast2 | AstUnsafe | LLVM_IR | Compile

let file_to_string file =  
  let array_string = ref [] in
  let ic = open_in file in
    try
     while true do 
       (* read line from in_channel and discard \n *)
       array_string := List.append !array_string [input_line ic]
       (* write the result to array_string *)
     done;
     String.concat "\n" !array_string
    with End_of_file -> 
         close_in ic; let x = String.concat "\n" !array_string in 
                      x ^ "\r\n"

let _ =
 let action = if Array.length Sys.argv > 1 then
   List.assoc Sys.argv.(1) [ ("-a", Ast);    (* Print the AST only *)
                  ("-ast", Ast2);  (* Generate a tree version of the AST *)
                  ("-uast", AstUnsafe);  (* Generate a tree version of the AST without checking it *)
                  ("-l", LLVM_IR);  (* Generate LLVM, don't check *)
                  ("-c", Compile) ] (* Generate, check LLVM IR *)
 else Compile in
 let lexbuf = if Array.length Sys.argv > 2 
              then Lexing.from_string (file_to_string Sys.argv.(2))
              else Lexing.from_channel stdin in
 let ast_nomain = Parser.program Scanner.token lexbuf in
 let ast = Clean_ast.clean ast_nomain in
  (*let ast = ast_nomain in*)
  match action with
   Ast -> Semant.check ast; print_string (Ast.string_of_program ast)
 | Ast2 -> Semant.check ast; print_string (Ast.abstract_of_program ast)
 | AstUnsafe -> print_string (Ast.abstract_of_program ast)
 | LLVM_IR -> Semant.check ast; print_string (Llvm.string_of_llmodule (Codegen.translate ast))
 | Compile -> let m = Codegen.translate ast in
   Semant.check ast; Llvm_analysis.assert_valid_module m;
   print_string (Llvm.string_of_llmodule m)