(* Abstract Syntax Tree and functions for printing it *)

type op = Add | Sub | Mult | Div | Mod | Equal | Neq | Less | Leq | Greater | Geq |
          And | Or

type uop = Neg | Not 

type expr =
    IntLit of int
  | FloatLit of float
  | StringLit of string
  | BoolLit of bool
  | TupleLit of expr list
  | Id of string
  | Binop of expr * op * expr
  | Unop of uop * expr
  | Assign of string * expr
  | Call of string * expr list
  | Element of string * expr
  | Noexpr

type typ = Int | Float | Bool | String | None | Tuple of typ * int 

type bind = typ * string

type var_decl = typ * string * expr

type stmt =
    Block of stmt list
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt * stmt
  | Elif of expr * stmt
  | For of expr * expr * expr * stmt
  | ForIn of expr * expr * stmt
  | While of expr * stmt 
  | HiddenWhile of expr * stmt * stmt
  | In of expr * expr  
  | Declaration of var_decl
  | Break
  | Continue
  | Nostmt

type func_body = {
    f_vdecls: var_decl list;
    f_stmts: stmt list;
  }

type func_decl = {
    typ : typ;
    fname : string;
    formals : bind list;
    fbody : func_body;
  }

type class_body = {
    vdecls: var_decl list;
    stmts: stmt list;
    funcs: func_decl list;
  }

type class_decl = {
    cname : string;
    cformals : typ list;
    cbody : class_body;
  }

type program = (stmt list * var_decl list) * (func_decl list * class_decl list)

(* Pretty-printing functions *)

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Mod -> "%"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "and"
  | Or -> "or"

let string_of_uop = function
    Neg -> "-"
  | Not -> "not"

let rec string_of_expr = function
    IntLit(i) -> string_of_int i
  | FloatLit(f) -> string_of_float f
  | StringLit(s) -> s
  | BoolLit(true) -> "True"
  | BoolLit(false) -> "False"
  | Id(s) -> s
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  | Unop(o, e) -> string_of_uop o ^ string_of_expr e
  | Assign(v, e) -> v ^ " = " ^ string_of_expr e
  | Call(f, el) -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | TupleLit(el) -> "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Element(s, e) -> s ^ "[" ^ string_of_expr e ^ "]"
  | Noexpr -> ""

let rec string_of_stmt = function
    Block(stmts) -> String.concat "" (List.map string_of_stmt stmts) ^ "end\n"
  | Expr(expr) -> string_of_expr expr ^ "\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ "\n";
  | If(e, s1, elifs, s2) ->  "if " ^ string_of_expr e ^ ":\n" ^
      string_of_stmt s1 ^ string_of_stmt elifs ^ "else:\n" ^ string_of_stmt s2
  | Elif(e, s1) -> "else if " ^ string_of_expr e ^ ":\n" ^ string_of_stmt s1 
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
      string_of_expr e3  ^ "):\n " ^ string_of_stmt s
  | While(e, s) -> "while " ^ string_of_expr e ^ ":\n " ^ string_of_stmt s
  | HiddenWhile(_, _, _) -> ""
  | ForIn(e1, e2, s) -> "for " ^ string_of_expr e1 ^ " in " ^ string_of_expr e2 ^ ":\n" ^ string_of_stmt s
  | In(e1, e2) -> string_of_expr e1 ^ " in " ^ string_of_expr e2 ^ ":\n"
  | Break -> "break"
  | Continue -> "continue"
  | Declaration(typ, id, expr) -> 
        let expr = string_of_expr expr in
        let typ = string_of_typ typ in
        "(DECLARATION: " ^ typ ^ " " ^ id ^ 
        ", EXPR: " ^ expr ^ ")"
  | Nostmt -> ""

and string_of_typ = function
    Int    -> "Int"
  | Float  -> "Float"
  | Bool   -> "Bool"
  | String -> "Str"
  | None   -> "None"
  | Tuple(t, l)  -> "Tuple[length "^string_of_int l^"]("^string_of_typ t^")"

let string_of_vdecl (t, id, e) = string_of_typ t ^ " " ^ id ^ " = " ^ string_of_expr e 

let string_of_fdecl fdecl =
    "def " ^ string_of_typ fdecl.typ ^ " " ^
    fdecl.fname ^ "(" ^ String.concat ", " (List.map snd fdecl.formals) ^
    "):\n" ^
    String.concat "" (List.map string_of_vdecl fdecl.fbody.f_vdecls) ^
    String.concat "" (List.map string_of_stmt fdecl.fbody.f_stmts) ^
    "end\n"

let string_of_cdecl cdecl =
    "class " ^ cdecl.cname ^ "(" ^ String.concat ", " (List.map string_of_typ cdecl.cformals) ^
    "):\n" ^
    String.concat "" (List.map string_of_vdecl cdecl.cbody.vdecls) ^ 
    String.concat "" (List.map string_of_stmt cdecl.cbody.stmts) ^
    String.concat "" (List.map string_of_fdecl cdecl.cbody.funcs) ^ "end\n"

let string_of_program (stmts_and_vdecls, funcs_and_classes) =  
  String.concat "\n" (List.map string_of_stmt (fst stmts_and_vdecls)) ^ "\n" ^
  String.concat "\n" (List.map string_of_vdecl (snd stmts_and_vdecls)) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl (fst funcs_and_classes)) ^ "\n" ^
  String.concat "\n" (List.map string_of_cdecl (snd funcs_and_classes))

let rec pt n = match n with 0 -> "\n" | _ -> (pt (n - 1)) ^ "\t"

let rec abstract_expr n = function
    IntLit(i) -> 
      "int " ^ string_of_int i
  | FloatLit(f) -> 
      "float " ^ string_of_float f
  | StringLit(s) -> 
      "string \"" ^ s ^ "\""
  | BoolLit(true) ->
      "bool " ^ "True"
  | BoolLit(false) -> 
      "bool " ^ "False"
  | Id(s) -> 
      "id " ^ s
  | Binop(e1, o, e2) ->
      (pt n) ^ "binop (" ^ 
        abstract_expr (n + 1) e1 ^ " "
        ^ string_of_op o ^ " " ^
        abstract_expr (n + 1) e2 ^ ")"
  | Unop(o, e) -> 
      (pt n) ^ "unop  (" ^ 
        string_of_uop o ^ " " ^ abstract_expr (n + 1) e ^ ")"
  | Assign(v, e) -> 
      (pt n) ^ "assign (" ^ v ^ " = " ^ 
        abstract_expr (n + 1) e ^ ")"
  | Call(f, el) -> 
    let abx x = (abstract_expr (n + 1) x) in
      (pt n) ^ "call (" ^ f ^ 
          "(args: " ^ String.concat ", " (List.map abx el) ^ ")" ^ ")"
  | TupleLit(el) -> 
    let abx x = (abstract_expr (n + 1) x) in
      (pt n) ^ "Tuple(" ^ String.concat ", " (List.map abx el) ^ ")"
  | Element(s, e) -> s ^ "[" ^ string_of_expr e ^ "]"
  | Noexpr -> ""


let rec abstract_stmt n = function
    Block(stmts) -> 
      let abm x = (abstract_stmt (n + 1) x) in
      let content = String.concat "" (List.map abm stmts) in
        (pt n) ^ "(BLOCK: " ^ content ^ ")"
  | Expr(expr) -> 
        let content = abstract_expr (n+1) expr in
        (pt n) ^ "(EXPR: " ^ content ^ ")"
  | Return(expr) -> 
        let content = abstract_expr (n+1) expr in
        (pt n) ^ "(RETURN: " ^ content ^ ")"

  | If(e, s1, elifs, s2) ->  
        let condition  = abstract_expr (n+1) e in
        let then_stmts = abstract_stmt (n+2) s1 in
        let elseifs    = abstract_stmt (n+2) elifs in
        let else_stmts = abstract_stmt (n+2) s2 in
        (pt n) ^ "(IF: " ^ 
        (pt (n+1)) ^ "(CONDITION: " ^ condition ^ ")" ^
        (pt (n+1)) ^ "(THEN: " ^ then_stmts ^ ")" ^
        (pt (n+1)) ^ "(ELSE IF: " ^ elseifs ^ ")" ^
        (pt (n+1)) ^ "(ELSE: " ^ else_stmts ^ ")" 
        ^ ")" 

  | Elif(e, s1) -> 
        let condition  = abstract_expr (n+1) e in
        let then_stmts = abstract_stmt (n+1) s1 in
        (pt n) ^ "(ELIF: " ^ 
        "(CONDITION: " ^ condition ^ ")" ^
      (pt n) ^ "(THEN: " ^ then_stmts ^ ")"  
        ^ ")" 

  | For(e1, e2, e3, s) ->
        let content_beg = abstract_expr (n+1) e1 in
        let condition   = abstract_expr (n+1) e2 in
        let iteration   = abstract_expr (n+1) e3 in
        let stmts       = abstract_stmt (n+2) s in
        (pt n) ^ "(FOR: " ^ "\n" ^
        (pt (n+1)) ^ "(BEGIN: "      ^ content_beg ^ ")" ^
        (pt (n+1)) ^ "(CONDITION: "  ^ condition ^ ")" ^
        (pt (n+1)) ^ "(ITERATION: "  ^ iteration ^ ")" ^ 
        (pt (n+1)) ^ "(STATEMENTS: " ^ stmts ^ ")" ^ 
        ")"

  | While(e, s) -> 
        let condition   = abstract_expr (n+1) e in
        let stmts       = abstract_stmt (n+2) s in
        (pt n) ^ "(WHILE: " ^ "\n" ^
        (pt (n+1)) ^ "(CONDITION: "  ^ condition ^ ")" ^
        (pt (n+1)) ^ "(STATEMENTS: " ^ stmts ^ ")" ^ 
        ")"
  | HiddenWhile(_, _, _) -> ""
  | ForIn(e1, e2, s) -> 
        let expr   = abstract_expr (n+1) e1 in
        let tuple  = abstract_expr (n+1) e2 in
        let stmts  = abstract_stmt (n+2) s in
        (pt n) ^ "(FORIN: " ^ 
                 "(EXPR: "  ^ expr ^ ")" ^
        (pt (n+1)) ^ "(ITERABLE LIST: "  ^ tuple ^ ")" ^
        (pt (n+1)) ^ "(STATEMENTS: " ^ stmts ^ ")" ^ 
        ")"
  | Declaration(typ, id, expr) -> 
        let expr = abstract_expr (n+1) expr in
        let typ = string_of_typ typ in
        (pt n) ^ "(DECLARATION: " ^ typ ^ " " ^ id ^ 
        ", EXPR: " ^ expr ^ ")"
  | In(_, _) -> "IN"
  | Break -> (pt n) ^ "BREAK"
  | Continue -> (pt n) ^ "CONTINUE"
  | Nostmt -> (pt n) ^ "NOSTMT"


let abstract_func n fdecl =
    let abm x = (abstract_stmt (n + 1) x) in
    let args = String.concat ", " (List.map snd fdecl.formals) in
    (pt n) ^ "(FUNCTION. TYPE: " ^ string_of_typ fdecl.typ ^ ", NAME:" ^
    fdecl.fname ^ ", ARGS: (" ^ args ^ ")" ^
    (pt n) ^ "STATEMENTS: " ^ String.concat "" (List.map abm fdecl.fbody.f_stmts) ^
    ")"

let abstract_of_program (stmts_and_vdecls, funcs_and_classes) = 
  let abm x = (abstract_stmt 1 x) in
  let abf x = (abstract_func 1 x) in
  let stmts = 
    String.concat "\n" (List.map abm (fst stmts_and_vdecls)) in
  let vdecls = 
    String.concat ", " (List.map string_of_vdecl (snd stmts_and_vdecls)) in
  let funcs =
    String.concat "\n" (List.map abf (fst funcs_and_classes)) in
  let classes = 
    String.concat "\n" (List.map string_of_cdecl (snd funcs_and_classes)) in
  "(stmts: "   ^ stmts   ^ ") \n" ^
  "(vdecls: "  ^ vdecls  ^ ") \n" ^
  "(funcs: "   ^ funcs   ^ ") \n" ^
  "(classes: " ^ classes ^ ") \n" 

