type id = string

(* Operators in JSJS *)
type op =
  (* num operators *)
  | Add | Sub | Mul | Div | Mod | Neg
  (* string operators *)
  | Caret
  (* boolean operators *)
  | And | Or | Not
  (* relational operators *)
  | Lte | Gte | Neq | Equals | Lt | Gt
  (* list operators *)
  | Cons

(* Types in JSJS are either a primitive type
   or a function type. both of these types are
   defined in a mutually recursive fashion *)
type primitiveType =
  (* a generic type *)
  | T of string
  (* a general type. used to define
     empty lists or empty maps *)
  | TAny
  | TNum
  | TString
  | TBool
  | TUnit
  (* type of exception associated with a message *)
  | TExn
  | TFun of funcType
  (* list type - a list of primitive types *)
  | TList of primitiveType
  (* map type - a tuple of key type, value type *)
  | TMap of primitiveType * primitiveType

(* a function type takes a list of primitive types
   and returns a primitive type *)
and funcType = primitiveType list * primitiveType

(* everything is an expression in JSJS *)
type expr =
  | UnitLit
  | NumLit of float
  | BoolLit of bool
  | StrLit of string
  (* binary operation *)
  | Binop of expr * op * expr
  (* unary operation *)
  | Unop of op * expr
  (* a list literal is a list of expressions *)
  | ListLit of expr list
  (* a map literal is a list of key-value pairs *)
  | MapLit of (expr * expr) list
  (* a block is a list of expression *)
  | Block of expr list
  (* an assignment expression takes a string,
     an annotated type and an expression *)
  | Assign of id * primitiveType * expr
  (* a value is just a id *)
  | Val of id
  (* if-then-else takes 3 expressions - predicate,
     then expr and else expr *)
  | If of expr * expr * expr
  (* a function call takes fn name and a list of arguments (exprs) *)
  | Call of expr * expr list
  (* a fn literal is of func_decl record *)
  | FunLit of id list * expr * primitiveType
  (* a module literal is a module name and expression *)
  | ModuleLit of id * expr
  (*  an exception type that is generated by throw expr *)
  | Throw of expr
  (* a try catch block has two exprs and an identifier that acts as
     a placeholder for the message *)
  | TryCatch of expr * id * expr
;;

(* annotated expression *)
type aexpr =
  | AUnitLit of primitiveType
  | ANumLit of float * primitiveType
  | ABoolLit of bool * primitiveType
  | AStrLit of string * primitiveType
  | ABinop of aexpr * op * aexpr * primitiveType
  | AUnop of op * aexpr * primitiveType
  | AListLit of aexpr list * primitiveType
  | AMapLit of (aexpr * aexpr) list * primitiveType
  | ABlock of aexpr list * primitiveType
  | AAssign of id * primitiveType * aexpr * primitiveType
  | AVal of id * primitiveType
  | AIf of aexpr * aexpr * aexpr * primitiveType
  | ACall of aexpr * aexpr list * primitiveType
  | AFunLit of id list * aexpr * primitiveType * primitiveType
  | AModuleLit of id * aexpr * primitiveType
  | AThrow of aexpr * primitiveType
  | ATryCatch of aexpr * id * aexpr * primitiveType
;;


(* a JSJS program is a list of expressions *)
type program = expr list;;
